/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.apache.jackrabbit.guava.common.base.Preconditions;
import org.apache.jackrabbit.guava.common.collect.Lists;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.segment.ListRecord;
import org.apache.jackrabbit.oak.segment.MapEntry;
import org.apache.jackrabbit.oak.segment.MapRecord;
import org.apache.jackrabbit.oak.segment.PropertyTemplate;
import org.apache.jackrabbit.oak.segment.Record;
import org.apache.jackrabbit.oak.segment.RecordId;
import org.apache.jackrabbit.oak.segment.Segment;
import org.apache.jackrabbit.oak.segment.SegmentBlob;
import org.apache.jackrabbit.oak.segment.SegmentNodeState;
import org.apache.jackrabbit.oak.segment.SegmentReader;
import org.apache.jackrabbit.oak.segment.Template;
import org.apache.jackrabbit.oak.spi.state.ChildNodeEntry;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.jetbrains.annotations.NotNull;

public class SegmentParser {
    @NotNull
    private final SegmentReader reader;

    public SegmentParser(@NotNull SegmentReader reader) {
        this.reader = (SegmentReader)Preconditions.checkNotNull((Object)reader);
    }

    protected void onNode(RecordId parentId, RecordId nodeId) {
        this.parseNode(nodeId);
    }

    protected void onTemplate(RecordId parentId, RecordId templateId) {
        this.parseTemplate(templateId);
    }

    protected void onMap(RecordId parentId, RecordId mapId, MapRecord map) {
        this.parseMap(parentId, mapId, map);
    }

    protected void onMapDiff(RecordId parentId, RecordId mapId, MapRecord map) {
        this.parseMapDiff(mapId, map);
    }

    protected void onMapLeaf(RecordId parentId, RecordId mapId, MapRecord map) {
        this.parseMapLeaf(mapId, map);
    }

    protected void onMapBranch(RecordId parentId, RecordId mapId, MapRecord map) {
        this.parseMapBranch(mapId, map);
    }

    protected void onProperty(RecordId parentId, RecordId propertyId, PropertyTemplate template) {
        this.parseProperty(parentId, propertyId, template);
    }

    protected void onValue(RecordId parentId, RecordId valueId, Type<?> type) {
        this.parseValue(parentId, valueId, type);
    }

    protected void onBlob(RecordId parentId, RecordId blobId) {
        this.parseBlob(blobId);
    }

    protected void onString(RecordId parentId, RecordId stringId) {
        this.parseString(stringId);
    }

    protected void onList(RecordId parentId, RecordId listId, int count) {
        this.parseList(parentId, listId, count);
    }

    protected void onListBucket(RecordId parentId, RecordId listId, int index, int count, int capacity) {
        this.parseListBucket(listId, index, count, capacity);
    }

    public NodeInfo parseNode(RecordId nodeId) {
        int size = 0;
        int nodeCount = 0;
        int propertyCount = 0;
        Segment segment = nodeId.getSegment();
        String stableId = this.reader.readNode(nodeId).getStableId();
        RecordId templateId = segment.readRecordId(nodeId.getRecordNumber(), 0, 1);
        this.onTemplate(nodeId, templateId);
        Template template = this.reader.readTemplate(templateId);
        if (template.getChildName() == "") {
            RecordId childMapId = segment.readRecordId(nodeId.getRecordNumber(), 0, 2);
            MapRecord childMap = this.reader.readMap(childMapId);
            this.onMap(nodeId, childMapId, childMap);
            for (ChildNodeEntry childNodeEntry : childMap.getEntries()) {
                NodeState child = childNodeEntry.getNodeState();
                if (!(child instanceof SegmentNodeState)) continue;
                RecordId childId = ((Record)child).getRecordId();
                this.onNode(nodeId, childId);
                ++nodeCount;
            }
        } else if (template.getChildName() != Template.ZERO_CHILD_NODES) {
            RecordId childId = segment.readRecordId(nodeId.getRecordNumber(), 0, 2);
            this.onNode(nodeId, childId);
            ++nodeCount;
        }
        int ids = template.getChildName() == Template.ZERO_CHILD_NODES ? 1 : 2;
        size += ids * 6;
        PropertyTemplate[] propertyTemplates = template.getPropertyTemplates();
        if (propertyTemplates.length > 0) {
            size += 6;
            RecordId id = segment.readRecordId(nodeId.getRecordNumber(), 0, ids + 1);
            ListRecord listRecord = new ListRecord(id, propertyTemplates.length);
            for (int i = 0; i < propertyTemplates.length; ++i) {
                RecordId propertyId = listRecord.getEntry(i);
                this.onProperty(nodeId, propertyId, propertyTemplates[i]);
                ++propertyCount;
            }
            this.onList(nodeId, id, propertyTemplates.length);
        }
        return new NodeInfo(nodeId, stableId, nodeCount, propertyCount, size);
    }

    public TemplateInfo parseTemplate(RecordId templateId) {
        int size = 0;
        Segment segment = templateId.getSegment();
        int head = segment.readInt(templateId.getRecordNumber(), size);
        boolean hasPrimaryType = (head & Integer.MIN_VALUE) != 0;
        boolean hasMixinTypes = (head & 0x40000000) != 0;
        boolean zeroChildNodes = (head & 0x20000000) != 0;
        boolean manyChildNodes = (head & 0x10000000) != 0;
        int mixinCount = head >> 18 & 0x3FF;
        int propertyCount = head & 0x3FFFF;
        size += 4;
        if (hasPrimaryType) {
            RecordId primaryId = segment.readRecordId(templateId.getRecordNumber(), size);
            this.onString(templateId, primaryId);
            size += 6;
        }
        if (hasMixinTypes) {
            for (int i = 0; i < mixinCount; ++i) {
                RecordId mixinId = segment.readRecordId(templateId.getRecordNumber(), size);
                this.onString(templateId, mixinId);
                size += 6;
            }
        }
        if (!zeroChildNodes && !manyChildNodes) {
            RecordId childNameId = segment.readRecordId(templateId.getRecordNumber(), size);
            this.onString(templateId, childNameId);
            size += 6;
        }
        if (propertyCount > 0) {
            RecordId listId = segment.readRecordId(templateId.getRecordNumber(), size);
            size += 6;
            ListRecord propertyNames = new ListRecord(listId, propertyCount);
            for (int i = 0; i < propertyCount; ++i) {
                RecordId propertyNameId = propertyNames.getEntry(i);
                ++size;
                this.onString(templateId, propertyNameId);
            }
            this.onList(templateId, listId, propertyCount);
        }
        return new TemplateInfo(templateId, hasPrimaryType, hasMixinTypes, zeroChildNodes, manyChildNodes, mixinCount, propertyCount, size);
    }

    public MapInfo parseMap(RecordId parentId, RecordId mapId, MapRecord map) {
        if (map.isDiff()) {
            this.onMapDiff(parentId, mapId, map);
        } else if (map.isLeaf()) {
            this.onMapLeaf(parentId, mapId, map);
        } else {
            this.onMapBranch(parentId, mapId, map);
        }
        return new MapInfo(mapId, -1);
    }

    public MapInfo parseMapDiff(RecordId mapId, MapRecord map) {
        int size = 4;
        size += 4;
        size += 6;
        size += 6;
        RecordId baseId = mapId.getSegment().readRecordId(mapId.getRecordNumber(), 8, 2);
        this.onMap(mapId, baseId, this.reader.readMap(baseId));
        return new MapInfo(mapId, size += 6);
    }

    public MapInfo parseMapLeaf(RecordId mapId, MapRecord map) {
        int size = 4;
        size += map.size() * 4;
        for (MapEntry entry : map.getEntries()) {
            size += 12;
            this.onString(mapId, entry.getKey());
        }
        return new MapInfo(mapId, size);
    }

    public MapInfo parseMapBranch(RecordId mapId, MapRecord map) {
        int size = 4;
        size += 4;
        for (MapRecord bucket : map.getBuckets()) {
            if (bucket == null) continue;
            size += 6;
            this.onMap(map.getRecordId(), bucket.getRecordId(), bucket);
        }
        return new MapInfo(mapId, size);
    }

    public PropertyInfo parseProperty(RecordId parentId, RecordId propertyId, PropertyTemplate template) {
        int size = 0;
        int count = -1;
        Segment segment = propertyId.getSegment();
        Type<?> type = template.getType();
        if (type.isArray()) {
            count = segment.readInt(propertyId.getRecordNumber());
            size += 4;
            if (count > 0) {
                RecordId listId = segment.readRecordId(propertyId.getRecordNumber(), 4);
                size += 6;
                for (RecordId valueId : new ListRecord(listId, count).getEntries()) {
                    this.onValue(propertyId, valueId, type.getBaseType());
                }
                this.onList(propertyId, listId, count);
            }
        } else {
            this.onValue(parentId, propertyId, type);
        }
        return new PropertyInfo(propertyId, count, size);
    }

    public ValueInfo parseValue(RecordId parentId, RecordId valueId, Type<?> type) {
        Preconditions.checkArgument((!type.isArray() ? 1 : 0) != 0);
        if (type == Type.BINARY) {
            this.onBlob(parentId, valueId);
        } else {
            this.onString(parentId, valueId);
        }
        return new ValueInfo(valueId, type);
    }

    public BlobInfo parseBlob(RecordId blobId) {
        BlobType blobType;
        int size = 0;
        Segment segment = blobId.getSegment();
        byte head = segment.readByte(blobId.getRecordNumber());
        if ((head & 0x80) == 0) {
            size += 1 + head;
            blobType = BlobType.SMALL;
        } else if ((head & 0xC0) == 128) {
            int length = (segment.readShort(blobId.getRecordNumber()) & 0x3FFF) + 128;
            size += 2 + length;
            blobType = BlobType.MEDIUM;
        } else if ((head & 0xE0) == 192) {
            long length = (segment.readLong(blobId.getRecordNumber()) & 0x1FFFFFFFFFFFFFFFL) + 16512L;
            int count = (int)((length + 4096L - 1L) / 4096L);
            RecordId listId = segment.readRecordId(blobId.getRecordNumber(), 8);
            this.onList(blobId, listId, count);
            size = (int)((long)size + (14L + length));
            blobType = BlobType.LONG;
        } else if ((head & 0xF0) == 224) {
            int length = StandardCharsets.UTF_8.encode(Objects.requireNonNull(SegmentBlob.readBlobId(segment, blobId.getRecordNumber()))).limit();
            size += 2 + length;
            blobType = BlobType.EXTERNAL;
        } else if ((head & 0xF8) == 240) {
            int length = StandardCharsets.UTF_8.encode(Objects.requireNonNull(SegmentBlob.readBlobId(segment, blobId.getRecordNumber()))).limit();
            size += 2 + length;
            blobType = BlobType.EXTERNAL;
        } else {
            throw new IllegalStateException(String.format("Unexpected value record type: %02x", head & 0xFF));
        }
        return new BlobInfo(blobId, blobType, size);
    }

    public BlobInfo parseString(RecordId stringId) {
        BlobType blobType;
        int size = 0;
        Segment segment = stringId.getSegment();
        long length = segment.readLength(stringId.getRecordNumber());
        if (length < 128L) {
            size = (int)((long)size + (1L + length));
            blobType = BlobType.SMALL;
        } else if (length < 16512L) {
            size = (int)((long)size + (2L + length));
            blobType = BlobType.MEDIUM;
        } else if (length < Integer.MAX_VALUE) {
            int count = (int)((length + 4096L - 1L) / 4096L);
            RecordId listId = segment.readRecordId(stringId.getRecordNumber(), 8);
            this.onList(stringId, listId, count);
            size = (int)((long)size + (14L + length));
            blobType = BlobType.LONG;
        } else {
            throw new IllegalStateException("String is too long: " + length);
        }
        return new BlobInfo(stringId, blobType, size);
    }

    public ListInfo parseList(RecordId parentId, RecordId listId, int count) {
        if (count != 0) {
            this.onListBucket(parentId, listId, 0, count, count);
        }
        return new ListInfo(listId, count, SegmentParser.noOfListSlots(count) * 6);
    }

    public ListBucketInfo parseListBucket(RecordId listId, int index, int count, int capacity) {
        Segment segment = listId.getSegment();
        int bucketSize = 1;
        while (bucketSize * 255 < capacity) {
            bucketSize *= 255;
        }
        if (capacity == 1) {
            List<RecordId> entries = Collections.singletonList(listId);
            return new ListBucketInfo(listId, true, entries, 6);
        }
        if (bucketSize == 1) {
            ArrayList entries = Lists.newArrayListWithCapacity((int)count);
            for (int i = 0; i < count; ++i) {
                entries.add(segment.readRecordId(listId.getRecordNumber(), 0, index + i));
            }
            return new ListBucketInfo(listId, true, entries, count * 6);
        }
        ArrayList entries = Lists.newArrayList();
        while (count > 0) {
            int bucketIndex = index / bucketSize;
            int bucketOffset = index % bucketSize;
            RecordId bucketId = segment.readRecordId(listId.getRecordNumber(), 0, bucketIndex);
            entries.add(bucketId);
            int c = Math.min(bucketSize, capacity - bucketIndex * bucketSize);
            int n = Math.min(c - bucketOffset, count);
            this.onListBucket(listId, bucketId, bucketOffset, n, c);
            index += n;
            count -= n;
        }
        return new ListBucketInfo(listId, false, entries, entries.size() * 6);
    }

    private static int noOfListSlots(int size) {
        if (size <= 255) {
            return size;
        }
        int fullBuckets = size / 255;
        if (size % 255 > 1) {
            return size + SegmentParser.noOfListSlots(fullBuckets + 1);
        }
        return size + SegmentParser.noOfListSlots(fullBuckets);
    }

    public static class ListBucketInfo {
        public final RecordId listId;
        public final boolean leaf;
        public final List<RecordId> entries;
        public final int size;

        public ListBucketInfo(RecordId listId, boolean leaf, List<RecordId> entries, int size) {
            this.listId = listId;
            this.leaf = leaf;
            this.entries = entries;
            this.size = size;
        }
    }

    public static class ListInfo {
        public final RecordId listId;
        public final int count;
        public final int size;

        public ListInfo(RecordId listId, int count, int size) {
            this.listId = listId;
            this.count = count;
            this.size = size;
        }
    }

    public static class BlobInfo {
        public final RecordId blobId;
        public final BlobType blobType;
        public final int size;

        public BlobInfo(RecordId blobId, BlobType blobType, int size) {
            this.blobId = blobId;
            this.blobType = blobType;
            this.size = size;
        }
    }

    public static class ValueInfo {
        public final RecordId valueId;
        public final Type<?> type;

        public ValueInfo(RecordId valueId, Type<?> type) {
            this.valueId = valueId;
            this.type = type;
        }
    }

    public static class PropertyInfo {
        public final RecordId propertyId;
        public final int count;
        public final int size;

        public PropertyInfo(RecordId propertyId, int count, int size) {
            this.propertyId = propertyId;
            this.count = count;
            this.size = size;
        }
    }

    public static class MapInfo {
        public final RecordId mapId;
        public final int size;

        public MapInfo(RecordId mapId, int size) {
            this.mapId = mapId;
            this.size = size;
        }
    }

    public static class TemplateInfo {
        public final RecordId templateId;
        public final boolean hasPrimaryType;
        public final boolean hasMixinType;
        public final boolean zeroChildNodes;
        public final boolean manyChildNodes;
        public final int mixinCount;
        public final int propertyCount;
        public final int size;

        public TemplateInfo(RecordId templateId, boolean hasPrimaryType, boolean hasMixinType, boolean zeroChildNodes, boolean manyChildNodes, int mixinCount, int propertyCount, int size) {
            this.templateId = templateId;
            this.hasPrimaryType = hasPrimaryType;
            this.hasMixinType = hasMixinType;
            this.zeroChildNodes = zeroChildNodes;
            this.manyChildNodes = manyChildNodes;
            this.mixinCount = mixinCount;
            this.propertyCount = propertyCount;
            this.size = size;
        }
    }

    public static class NodeInfo {
        public final RecordId nodeId;
        public final String stableId;
        public final int nodeCount;
        public final int propertyCount;
        public final int size;

        public NodeInfo(RecordId nodeId, String stableId, int nodeCount, int propertyCount, int size) {
            this.nodeId = nodeId;
            this.stableId = stableId;
            this.nodeCount = nodeCount;
            this.propertyCount = propertyCount;
            this.size = size;
        }
    }

    public static enum BlobType {
        SMALL,
        MEDIUM,
        LONG,
        EXTERNAL;

    }
}

