/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.tool.check;

import java.io.PrintWriter;
import java.text.DateFormat;
import java.text.MessageFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.commons.collections.CollectionUtils;
import org.apache.jackrabbit.oak.commons.conditions.Validate;
import org.apache.jackrabbit.oak.segment.SegmentNodeStoreBuilders;
import org.apache.jackrabbit.oak.segment.file.JournalEntry;
import org.apache.jackrabbit.oak.segment.file.JournalReader;
import org.apache.jackrabbit.oak.segment.file.ReadOnlyFileStore;
import org.apache.jackrabbit.oak.segment.file.tooling.ConsistencyChecker;

public class CheckHelper {
    private final boolean checkBinaries;
    private final boolean checkHead;
    private final Integer revisionsCount;
    private final Set<String> requestedCheckpoints;
    private final Set<String> filterPaths;
    private final boolean failFast;
    private final long debugInterval;
    private final PrintWriter out;
    private final PrintWriter err;
    private int currentNodeCount;
    private int currentPropertyCount;
    private int headNodeCount;
    private int headPropertyCount;
    private long lastDebugEvent;

    public static Builder builder() {
        return new Builder();
    }

    private CheckHelper(Builder builder) {
        this.debugInterval = builder.debugInterval;
        this.checkHead = builder.checkHead;
        this.checkBinaries = builder.checkBinaries;
        this.requestedCheckpoints = builder.checkpoints;
        this.filterPaths = builder.filterPaths;
        this.out = builder.outWriter;
        this.err = builder.errWriter;
        this.failFast = builder.failFast;
        this.revisionsCount = builder.revisionsCount;
    }

    public int run(ReadOnlyFileStore store, JournalReader journal) {
        Set checkpoints = this.requestedCheckpoints;
        if (this.requestedCheckpoints.contains("all")) {
            checkpoints = CollectionUtils.toLinkedSet(SegmentNodeStoreBuilders.builder(store).build().checkpoints());
        }
        ConsistencyChecker.ConsistencyCheckResult result = this.newConsistencyChecker().checkConsistency(store, (Iterator<JournalEntry>)((Object)journal), this.checkHead, checkpoints, this.filterPaths, this.checkBinaries, this.revisionsCount, this.failFast);
        this.print("\nSearched through {0} revisions and {1} checkpoints", result.getCheckedRevisionsCount(), checkpoints.size());
        if (this.isGoodRevisionFound(result)) {
            if (this.checkHead) {
                this.print("\nHead", new Object[0]);
                for (Map.Entry entry : result.getHeadRevisions().entrySet()) {
                    this.printRevision(0, (String)entry.getKey(), (ConsistencyChecker.Revision)entry.getValue());
                }
            }
            if (!checkpoints.isEmpty()) {
                this.print("\nCheckpoints", new Object[0]);
                for (String string : result.getCheckpointRevisions().keySet()) {
                    this.print("- {0}", string);
                    for (Map.Entry<String, ConsistencyChecker.Revision> e : result.getCheckpointRevisions().get(string).entrySet()) {
                        this.printRevision(2, e.getKey(), e.getValue());
                    }
                }
            }
            this.print("\nOverall", new Object[0]);
            this.printOverallRevision(result.getOverallRevision());
            return 0;
        }
        this.print("No good revision found", new Object[0]);
        return 1;
    }

    public int getHeadNodeCount() {
        return this.headNodeCount;
    }

    public int getHeadPropertyCount() {
        return this.headPropertyCount;
    }

    private boolean isGoodRevisionFound(ConsistencyChecker.ConsistencyCheckResult result) {
        return this.failFast ? CheckHelper.hasAllRevision(result) : CheckHelper.hasAnyRevision(result);
    }

    private ConsistencyChecker newConsistencyChecker() {
        return new ConsistencyChecker(){

            @Override
            protected void onCheckRevision(String revision) {
                CheckHelper.this.print("\nChecking revision {0}", revision);
            }

            @Override
            protected void onCheckHead() {
                CheckHelper.this.headNodeCount = 0;
                CheckHelper.this.headPropertyCount = 0;
                CheckHelper.this.print("\nChecking head\n", new Object[0]);
            }

            @Override
            protected void onCheckChekpoints() {
                CheckHelper.this.print("\nChecking checkpoints", new Object[0]);
            }

            @Override
            protected void onCheckCheckpoint(String checkpoint) {
                CheckHelper.this.print("\nChecking checkpoint {0}", checkpoint);
            }

            @Override
            protected void onCheckpointNotFoundInRevision(String checkpoint) {
                CheckHelper.this.printError("Checkpoint {0} not found in this revision!", checkpoint);
            }

            @Override
            protected void onCheckRevisionError(String revision, Exception e) {
                CheckHelper.this.printError("Skipping invalid record id {0}: {1}", revision, e);
            }

            @Override
            protected void onConsistentPath(String path) {
                CheckHelper.this.print("Path {0} is consistent", path);
            }

            @Override
            protected void onPathNotFound(String path) {
                CheckHelper.this.printError("Path {0} not found", path);
            }

            @Override
            protected void onCheckTree(String path, boolean head) {
                CheckHelper.this.currentNodeCount = 0;
                CheckHelper.this.currentPropertyCount = 0;
                CheckHelper.this.print("Checking {0}", path);
            }

            @Override
            protected void onCheckTreeEnd(boolean head) {
                if (head) {
                    CheckHelper.this.headNodeCount += CheckHelper.this.currentNodeCount;
                    CheckHelper.this.headPropertyCount += CheckHelper.this.currentPropertyCount;
                }
                CheckHelper.this.print("Checked {0} nodes and {1} properties", CheckHelper.this.currentNodeCount, CheckHelper.this.currentPropertyCount);
            }

            @Override
            protected void onCheckNode(String path) {
                CheckHelper.this.debug("Traversing {0}", path);
                ++CheckHelper.this.currentNodeCount;
            }

            @Override
            protected void onCheckProperty() {
                ++CheckHelper.this.currentPropertyCount;
            }

            @Override
            protected void onCheckPropertyEnd(String path, PropertyState property) {
                CheckHelper.this.debug("Checked {0}/{1}", path, property);
            }

            @Override
            protected void onCheckNodeError(String path, Exception e) {
                CheckHelper.this.printError("Error while traversing {0}: {1}", path, e);
            }

            @Override
            protected void onCheckTreeError(String path, Exception e) {
                CheckHelper.this.printError("Error while traversing {0}: {1}", path, e.getMessage());
            }
        };
    }

    private static boolean hasAnyRevision(ConsistencyChecker.ConsistencyCheckResult result) {
        return CheckHelper.hasAnyHeadRevision(result) || CheckHelper.hasAnyCheckpointRevision(result);
    }

    private static boolean hasAllRevision(ConsistencyChecker.ConsistencyCheckResult result) {
        return CheckHelper.hasAnyHeadRevision(result) && CheckHelper.hasAllCheckpointRevision(result);
    }

    private static boolean hasAnyHeadRevision(ConsistencyChecker.ConsistencyCheckResult result) {
        return result.getHeadRevisions().values().stream().anyMatch(Objects::nonNull);
    }

    private static boolean hasAnyCheckpointRevision(ConsistencyChecker.ConsistencyCheckResult result) {
        return result.getCheckpointRevisions().values().stream().flatMap(m -> m.values().stream()).anyMatch(Objects::nonNull);
    }

    private static boolean hasAllCheckpointRevision(ConsistencyChecker.ConsistencyCheckResult result) {
        return result.getCheckpointRevisions().values().stream().flatMap(m -> m.values().stream()).allMatch(Objects::nonNull);
    }

    private void printRevision(int indent, String path, ConsistencyChecker.Revision revision) {
        Optional<ConsistencyChecker.Revision> r = Optional.ofNullable(revision);
        this.print("{0}Latest good revision for path {1} is {2} from {3}", " ".repeat(indent), path, r.map(ConsistencyChecker.Revision::getRevision).orElse("none"), r.map(ConsistencyChecker.Revision::getTimestamp).map(CheckHelper::timestampToString).orElse("unknown time"));
    }

    private void printOverallRevision(ConsistencyChecker.Revision revision) {
        Optional<ConsistencyChecker.Revision> r = Optional.ofNullable(revision);
        this.print("Latest good revision for paths and checkpoints checked is {0} from {1}", r.map(ConsistencyChecker.Revision::getRevision).orElse("none"), r.map(ConsistencyChecker.Revision::getTimestamp).map(CheckHelper::timestampToString).orElse("unknown time"));
    }

    private static String timestampToString(long timestamp) {
        return DateFormat.getDateTimeInstance().format(new Date(timestamp));
    }

    private void printError(String format, Object ... args) {
        this.err.println(MessageFormat.format(format, args));
    }

    private void print(String format, Object ... arguments) {
        this.out.println(MessageFormat.format(format, arguments));
    }

    private void debug(String format, Object ... arg) {
        if (this.debug()) {
            this.print(format, arg);
        }
    }

    private boolean debug() {
        if (this.debugInterval == Long.MAX_VALUE) {
            return false;
        }
        if (this.debugInterval == 0L) {
            return true;
        }
        long t = System.currentTimeMillis();
        if ((t - this.lastDebugEvent) / 1000L > this.debugInterval) {
            this.lastDebugEvent = t;
            return true;
        }
        return false;
    }

    public static class Builder {
        private long debugInterval = Long.MAX_VALUE;
        private boolean checkBinaries;
        private boolean checkHead;
        private Integer revisionsCount;
        private Set<String> checkpoints;
        private Set<String> filterPaths;
        private PrintWriter outWriter;
        private PrintWriter errWriter;
        private boolean failFast;

        private Builder() {
        }

        public Builder withDebugInterval(long debugInterval) {
            Validate.checkArgument((debugInterval >= 0L ? 1 : 0) != 0);
            this.debugInterval = debugInterval;
            return this;
        }

        public Builder withCheckBinaries(boolean checkBinaries) {
            this.checkBinaries = checkBinaries;
            return this;
        }

        public Builder withCheckHead(boolean checkHead) {
            this.checkHead = checkHead;
            return this;
        }

        public Builder withRevisionsCount(Integer revisionsCount) {
            this.revisionsCount = revisionsCount;
            return this;
        }

        public Builder withCheckpoints(Set<String> checkpoints) {
            this.checkpoints = checkpoints;
            return this;
        }

        public Builder withFilterPaths(Set<String> filterPaths) {
            this.filterPaths = filterPaths;
            return this;
        }

        public Builder withOutWriter(PrintWriter outWriter) {
            this.outWriter = outWriter;
            return this;
        }

        public Builder withErrWriter(PrintWriter errWriter) {
            this.errWriter = errWriter;
            return this;
        }

        public Builder withFailFast(boolean failFast) {
            this.failFast = failFast;
            return this;
        }

        public CheckHelper build() {
            return new CheckHelper(this);
        }
    }
}

