/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.file.preloader;

import java.io.Closeable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Comparator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.jackrabbit.oak.commons.Buffer;
import org.apache.jackrabbit.oak.commons.conditions.Validate;
import org.apache.jackrabbit.oak.commons.internal.function.Suppliers;
import org.apache.jackrabbit.oak.segment.SegmentId;
import org.apache.jackrabbit.oak.segment.file.tar.TarFiles;
import org.apache.jackrabbit.oak.segment.spi.persistence.persistentcache.DelegatingPersistentCache;
import org.apache.jackrabbit.oak.segment.spi.persistence.persistentcache.PersistentCache;
import org.apache.jackrabbit.oak.segment.spi.persistence.persistentcache.PersistentCachePreloadingConfiguration;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentPreloader
extends DelegatingPersistentCache
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(SegmentPreloader.class);
    private final Map<Integer, String> inProgressPrefetch;
    private final ConcurrentHashMap<String, Map<UUID, Set<UUID>>> graphCache;
    private final PersistentCache delegate;
    private final ExecutorService dispatchPool;
    private final ExecutorService preloadPool;
    private final int preloadDepth;
    private final Supplier<TarFiles> tarFiles;

    @NotNull
    public static PersistentCache decorate(@NotNull PersistentCache delegate, @NotNull PersistentCachePreloadingConfiguration config, @NotNull Supplier<TarFiles> tarFiles) {
        if (config.getConcurrency() > 0 && config.getMaxPreloadDepth() > 0) {
            return new SegmentPreloader(delegate, config, tarFiles);
        }
        return delegate;
    }

    private SegmentPreloader(@NotNull PersistentCache delegate, @NotNull PersistentCachePreloadingConfiguration config, @NotNull Supplier<TarFiles> tarFiles) {
        this.delegate = delegate;
        this.tarFiles = Suppliers.memoize(tarFiles);
        this.inProgressPrefetch = new ConcurrentHashMap<Integer, String>();
        this.graphCache = new ConcurrentHashMap();
        this.preloadDepth = config.getMaxPreloadDepth();
        this.dispatchPool = new ThreadPoolExecutor(1, 1, 1L, TimeUnit.SECONDS, new PriorityBlockingQueue(), r -> new Thread(r, "segment-preload-dispatcher")){

            @Override
            protected void afterExecute(Runnable r, Throwable t) {
                super.afterExecute(r, t);
                SegmentPreloader.this.clearInProgressTask(r);
            }
        };
        int preloadThreads = config.getConcurrency();
        ThreadPoolExecutor preloadPool = new ThreadPoolExecutor(Math.max(1, preloadThreads / 4), preloadThreads, 5L, TimeUnit.SECONDS, new LinkedBlockingQueue(preloadThreads * 4), r -> {
            String threadName = String.format("segment-preload-%s", Long.toHexString(System.nanoTime() & 0xFFFFFL));
            Thread thread = new Thread(r, threadName);
            thread.setUncaughtExceptionHandler((t, e) -> {
                if (!(e instanceof InterruptedException)) {
                    LOG.warn("Uncaught exception in thread {}", (Object)t.getName(), (Object)e);
                }
            });
            return thread;
        }, (r, executor) -> {
            try {
                executor.getQueue().put(r);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }){

            @Override
            protected void afterExecute(Runnable r, Throwable t) {
                super.afterExecute(r, t);
                SegmentPreloader.this.clearInProgressTask(r);
            }
        };
        preloadPool.allowCoreThreadTimeOut(true);
        this.preloadPool = preloadPool;
    }

    @Override
    protected PersistentCache delegate() {
        return this.delegate;
    }

    @Override
    @Nullable
    public Buffer readSegment(long msb, long lsb, @NotNull Callable<Buffer> loader) {
        this.dispatch(msb, lsb);
        return this.delegate().readSegment(msb, lsb, loader);
    }

    private void dispatch(long msb, long lsb) {
        this.dispatch(msb, lsb, 1);
    }

    private void dispatch(long msb, long lsb, int depth) {
        this.execute(this.dispatchPool, this.createDispatchTask(msb, lsb, depth));
    }

    @NotNull
    DispatchTask createDispatchTask(long msb, long lsb, int depth) {
        TarFiles tars = this.tarFiles.get();
        return new DispatchTask(tars, tars::getIndices, msb, lsb, depth);
    }

    private void preload(long msb, long lsb, int depth) {
        this.execute(this.preloadPool, this.createPreloadTask(msb, lsb, depth));
    }

    @NotNull
    PreloadTask createPreloadTask(long msb, long lsb, int depth) {
        return new PreloadTask(this.tarFiles.get(), msb, lsb, depth);
    }

    private void execute(ExecutorService pool, Runnable r) {
        if (!pool.isShutdown() && this.registerInProgressTask(r)) {
            pool.execute(r);
        }
    }

    private boolean registerInProgressTask(Runnable r) {
        return this.inProgressPrefetch.putIfAbsent(r.hashCode(), Thread.currentThread().getName()) == null;
    }

    private void clearInProgressTask(Runnable r) {
        this.inProgressPrefetch.remove(r.hashCode());
    }

    @Override
    public void close() {
        try {
            this.preloadPool.shutdown();
            this.dispatchPool.shutdown();
            if (!this.preloadPool.awaitTermination(4L, TimeUnit.SECONDS)) {
                this.preloadPool.shutdownNow();
            }
            if (!this.dispatchPool.awaitTermination(1L, TimeUnit.SECONDS)) {
                this.dispatchPool.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.preloadPool.shutdownNow();
            this.dispatchPool.shutdownNow();
        }
    }

    class PreloadTask
    implements Runnable {
        private final TarFiles tarFiles;
        private final long msb;
        private final long lsb;
        private final int depth;

        private PreloadTask(TarFiles tarFiles, long msb, long lsb, int depth) {
            Validate.checkArgument((depth <= SegmentPreloader.this.preloadDepth ? 1 : 0) != 0, (String)"depth must be <= %d, is %d", (Object[])new Object[]{SegmentPreloader.this.preloadDepth, depth});
            this.tarFiles = tarFiles;
            this.msb = msb;
            this.lsb = lsb;
            this.depth = depth;
            LOG.debug("Created: {}", (Object)this);
        }

        @Override
        public void run() {
            Buffer segmentBuffer;
            LOG.debug("Running: {}", (Object)this);
            if (this.depth < SegmentPreloader.this.preloadDepth && SegmentId.isDataSegmentId(this.lsb)) {
                SegmentPreloader.this.dispatch(this.msb, this.lsb, this.depth + 1);
            }
            if (!SegmentPreloader.this.delegate.containsSegment(this.msb, this.lsb) && (segmentBuffer = this.tarFiles.readSegment(this.msb, this.lsb)) != null) {
                SegmentPreloader.this.delegate.writeSegment(this.msb, this.lsb, segmentBuffer);
            }
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o.getClass() != PreloadTask.class) {
                return false;
            }
            PreloadTask that = (PreloadTask)o;
            return this.msb == that.msb && this.lsb == that.lsb;
        }

        public int hashCode() {
            return Objects.hash(this.getClass(), this.msb, this.lsb);
        }

        public String toString() {
            return "PreloadTask{segmentId=" + String.valueOf(new UUID(this.msb, this.lsb)) + ", depth=" + this.depth + "}";
        }
    }

    class DispatchTask
    implements Runnable,
    Comparable<DispatchTask> {
        private final TarFiles tarFiles;
        private final Supplier<Map<String, Set<UUID>>> indicesSupplier;
        private final long msb;
        private final long lsb;
        private final int depth;
        private final long creationTime = System.nanoTime();

        private DispatchTask(TarFiles tarFiles, Supplier<Map<String, Set<UUID>>> indicesSupplier, long msb, long lsb, int depth) {
            Validate.checkArgument((depth <= SegmentPreloader.this.preloadDepth ? 1 : 0) != 0, (String)"depth must be <= %d, is %d", (Object[])new Object[]{SegmentPreloader.this.preloadDepth, depth});
            this.tarFiles = tarFiles;
            this.indicesSupplier = indicesSupplier;
            this.msb = msb;
            this.lsb = lsb;
            this.depth = depth;
            LOG.debug("Created: {}", (Object)this);
        }

        @Override
        public void run() {
            LOG.debug("Running: {}", (Object)this);
            UUID uuid = new UUID(this.msb, this.lsb);
            Map<String, Set<UUID>> indices = this.indicesSupplier.get();
            String archiveName = indices.entrySet().stream().filter(entry -> ((Set)entry.getValue()).contains(uuid)).findFirst().map(Map.Entry::getKey).orElse(null);
            Map graph = SegmentPreloader.this.graphCache.computeIfAbsent(archiveName, name -> {
                try {
                    return this.tarFiles.getGraph((String)name);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            });
            for (UUID reference : (Set)graph.get(uuid)) {
                long refLsb;
                long refMsb = reference.getMostSignificantBits();
                if (!SegmentPreloader.this.delegate.containsSegment(refMsb, refLsb = reference.getLeastSignificantBits())) {
                    SegmentPreloader.this.preload(refMsb, refLsb, this.depth);
                    continue;
                }
                if (this.depth >= SegmentPreloader.this.preloadDepth || !SegmentId.isDataSegmentId(refLsb)) continue;
                SegmentPreloader.this.dispatch(refMsb, refLsb, this.depth + 1);
            }
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o.getClass() != DispatchTask.class) {
                return false;
            }
            DispatchTask that = (DispatchTask)o;
            return this.msb == that.msb && this.lsb == that.lsb && this.depth == that.depth;
        }

        public int hashCode() {
            return Objects.hash(this.getClass(), this.msb, this.lsb, this.depth);
        }

        public String toString() {
            return "DispatchTask{segmentId=" + String.valueOf(new UUID(this.msb, this.lsb)) + ", depth=" + this.depth + "}";
        }

        private int getPreloadDepth() {
            return this.depth;
        }

        private long getCreationTime() {
            return this.creationTime;
        }

        @Override
        public int compareTo(@NotNull DispatchTask o) {
            return Comparator.comparing(DispatchTask::getPreloadDepth).thenComparing(DispatchTask::getCreationTime).compare(this, o);
        }
    }
}

