/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.jackrabbit.guava.common.collect.Maps;
import org.apache.jackrabbit.oak.commons.json.JsopBuilder;
import org.apache.jackrabbit.oak.commons.json.JsopTokenizer;
import org.apache.jackrabbit.oak.plugins.document.Collection;
import org.apache.jackrabbit.oak.plugins.document.CommitQueue;
import org.apache.jackrabbit.oak.plugins.document.Document;
import org.apache.jackrabbit.oak.plugins.document.DocumentNodeStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStore;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.RevisionVector;
import org.apache.jackrabbit.oak.plugins.document.StableRevisionComparator;
import org.apache.jackrabbit.oak.plugins.document.UpdateOp;
import org.apache.jackrabbit.oak.plugins.document.util.Utils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class Checkpoints {
    private static final Logger LOG = LoggerFactory.getLogger(Checkpoints.class);
    private static final String ID = "checkpoint";
    private static final String PROP_CHECKPOINT = "data";
    static final int CLEANUP_INTERVAL = 100;
    private final DocumentNodeStore nodeStore;
    private final DocumentStore store;
    private final AtomicInteger createCounter = new AtomicInteger();
    private final Object cleanupLock = new Object();

    Checkpoints(DocumentNodeStore store) {
        this.nodeStore = store;
        this.store = store.getDocumentStore();
        this.createIfNotExist();
    }

    public Revision create(long lifetimeInMillis, Map<String, String> info) {
        Revision r = this.nodeStore.commitQueue.createRevision();
        final RevisionVector[] rv = new RevisionVector[1];
        this.nodeStore.commitQueue.done(r, new CommitQueue.Callback(){

            @Override
            public void headOfQueue(@NotNull Revision revision) {
                rv[0] = Checkpoints.this.nodeStore.getHeadRevision();
            }
        });
        long endTime = Utils.sum(this.nodeStore.getClock().getTime(), lifetimeInMillis);
        this.create(r, new Info(endTime, rv[0], info));
        return r;
    }

    public Revision create(long lifetimeInMillis, @NotNull Map<String, String> info, @NotNull Revision revision) {
        if (revision.getTimestamp() > this.nodeStore.getClock().getTime()) {
            throw new IllegalArgumentException("Cannot create checkpoint with a revision in the future: " + revision);
        }
        long endTime = Utils.sum(this.nodeStore.getClock().getTime(), lifetimeInMillis);
        this.create(revision, new Info(endTime, null, info));
        return revision;
    }

    public void release(String checkpoint) {
        UpdateOp op = new UpdateOp(ID, false);
        op.removeMapEntry(PROP_CHECKPOINT, Revision.fromString(checkpoint));
        this.store.findAndUpdate(Collection.SETTINGS, op);
    }

    @Nullable
    public Revision getOldestRevisionToKeep(boolean performCleanup) {
        SortedMap<Revision, Info> checkpoints = this.getCheckpoints();
        if (checkpoints.isEmpty()) {
            LOG.debug("No checkpoint registered so far");
            return null;
        }
        long currentTime = this.nodeStore.getClock().getTime();
        UpdateOp op = new UpdateOp(ID, false);
        Revision lastAliveRevision = null;
        for (Map.Entry<Revision, Info> e : checkpoints.entrySet()) {
            long expiryTime = e.getValue().getExpiryTime();
            if (currentTime > expiryTime) {
                op.removeMapEntry(PROP_CHECKPOINT, e.getKey());
                continue;
            }
            Revision cpRev = e.getKey();
            RevisionVector rv = e.getValue().getCheckpoint();
            if (rv != null) {
                cpRev = rv.getRevision(cpRev.getClusterId());
            }
            lastAliveRevision = Utils.min(lastAliveRevision, cpRev);
        }
        if (performCleanup && op.hasChanges()) {
            try {
                this.store.findAndUpdate(Collection.SETTINGS, op);
                LOG.debug("Purged {} expired checkpoints", (Object)op.getChanges().size());
            }
            catch (UnsupportedOperationException uoe) {
                LOG.info("getOldestRevisionToKeep : could not clean up expired checkpoints due to exception : " + uoe, (Throwable)uoe);
            }
        }
        return lastAliveRevision;
    }

    public Revision getOldestRevisionToKeep() {
        return this.getOldestRevisionToKeep(true);
    }

    @NotNull
    SortedMap<Revision, Info> getCheckpoints() {
        Document cdoc = this.store.find(Collection.SETTINGS, ID, 0);
        SortedMap data = null;
        if (cdoc != null) {
            data = (SortedMap)cdoc.get(PROP_CHECKPOINT);
        }
        TreeMap<Revision, Info> checkpoints = new TreeMap<Revision, Info>(StableRevisionComparator.REVERSE);
        if (data != null) {
            for (Map.Entry entry : data.entrySet()) {
                checkpoints.put((Revision)entry.getKey(), Info.fromString((String)entry.getValue()));
            }
        }
        return checkpoints;
    }

    @Nullable
    RevisionVector retrieve(@NotNull String checkpoint) throws IllegalArgumentException {
        Revision r;
        try {
            r = Revision.fromString(Objects.requireNonNull(checkpoint));
        }
        catch (IllegalArgumentException e) {
            LOG.warn("Malformed checkpoint reference: {}", (Object)checkpoint);
            return null;
        }
        Info info = (Info)this.getCheckpoints().get(r);
        if (info == null) {
            return null;
        }
        RevisionVector rv = info.getCheckpoint();
        if (rv == null) {
            rv = this.expand(r);
        }
        return rv;
    }

    void setInfoProperty(@NotNull String checkpoint, @NotNull String key, @Nullable String value) {
        Revision r = Revision.fromString(Objects.requireNonNull(checkpoint));
        Info info = (Info)this.getCheckpoints().get(r);
        if (info == null) {
            throw new IllegalArgumentException("No such checkpoint: " + checkpoint);
        }
        LinkedHashMap<String, String> metadata = new LinkedHashMap<String, String>(info.get());
        if (value == null) {
            metadata.remove(key);
        } else {
            metadata.put(key, value);
        }
        Info newInfo = new Info(info.getExpiryTime(), info.getCheckpoint(), metadata);
        UpdateOp op = new UpdateOp(ID, false);
        op.setMapEntry(PROP_CHECKPOINT, r, newInfo.toString());
        this.store.findAndUpdate(Collection.SETTINGS, op);
    }

    int size() {
        return this.getCheckpoints().size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void performCleanupIfRequired() {
        if (this.createCounter.get() > 100) {
            Object object = this.cleanupLock;
            synchronized (object) {
                this.getOldestRevisionToKeep();
                this.createCounter.set(0);
            }
        }
    }

    private void createIfNotExist() {
        if (this.store.find(Collection.SETTINGS, ID) == null) {
            UpdateOp updateOp = new UpdateOp(ID, true);
            this.store.createOrUpdate(Collection.SETTINGS, updateOp);
        }
    }

    private void create(@NotNull Revision revision, @NotNull Info info) {
        this.createCounter.getAndIncrement();
        this.performCleanupIfRequired();
        UpdateOp op = new UpdateOp(ID, false);
        op.setMapEntry(PROP_CHECKPOINT, revision, info.toString());
        this.store.createOrUpdate(Collection.SETTINGS, op);
    }

    private RevisionVector expand(Revision checkpoint) {
        LOG.warn("Expanding {} single revision checkpoint into a RevisionVector. Please make sure all cluster nodes run with the same Oak version.", (Object)checkpoint);
        HashMap revs = Maps.newHashMap();
        RevisionVector head = this.nodeStore.getHeadRevision();
        for (Revision r : head) {
            int cId = r.getClusterId();
            if (cId == checkpoint.getClusterId()) {
                revs.put(cId, checkpoint);
                continue;
            }
            revs.put(cId, new Revision(checkpoint.getTimestamp(), 0, cId));
        }
        return head.pmin(new RevisionVector(revs.values()));
    }

    static final class Info {
        private static final String EXPIRES = "expires";
        private static final String REVISION_VECTOR = "rv";
        private final long expiryTime;
        private final RevisionVector checkpoint;
        private final Map<String, String> info;

        private Info(long expiryTime, @Nullable RevisionVector checkpoint, @NotNull Map<String, String> info) {
            this.expiryTime = expiryTime;
            this.checkpoint = checkpoint;
            this.info = Collections.unmodifiableMap(info);
        }

        static Info fromString(String info) {
            long expiryTime;
            Map<String, String> map;
            RevisionVector rv = null;
            if (info.startsWith("{")) {
                map = Maps.newHashMap();
                JsopTokenizer reader = new JsopTokenizer(info);
                reader.read(123);
                String key = reader.readString();
                if (!EXPIRES.equals(key)) {
                    throw new IllegalArgumentException("First entry in the checkpoint info must be the expires date: " + info);
                }
                reader.read(58);
                expiryTime = Long.parseLong(reader.readString());
                while (reader.matches(44)) {
                    key = reader.readString();
                    reader.read(58);
                    String value = reader.readString();
                    if (rv == null && map.isEmpty() && REVISION_VECTOR.equals(key)) {
                        try {
                            rv = RevisionVector.fromString(value);
                        }
                        catch (IllegalArgumentException e) {
                            map.put(key, value);
                        }
                        continue;
                    }
                    map.put(key, value);
                }
                reader.read(125);
                reader.read(0);
            } else {
                map = Collections.emptyMap();
                expiryTime = Long.parseLong(info);
            }
            return new Info(expiryTime, rv, map);
        }

        Map<String, String> get() {
            return this.info;
        }

        long getExpiryTime() {
            return this.expiryTime;
        }

        @Nullable
        RevisionVector getCheckpoint() {
            return this.checkpoint;
        }

        public String toString() {
            JsopBuilder writer = new JsopBuilder();
            writer.object();
            writer.key(EXPIRES).value(Long.toString(this.expiryTime));
            if (this.checkpoint != null) {
                writer.key(REVISION_VECTOR).value(this.checkpoint.toString());
            }
            for (Map.Entry<String, String> entry : this.info.entrySet()) {
                writer.key(entry.getKey()).value(entry.getValue());
            }
            writer.endObject();
            return writer.toString();
        }
    }
}

