/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jclouds.blobstore.internal;

import static com.google.common.base.Preconditions.checkNotNull;

import jakarta.inject.Inject;
import jakarta.inject.Singleton;

import org.jclouds.Context;
import org.jclouds.blobstore.BlobRequestSigner;
import org.jclouds.blobstore.BlobStore;
import org.jclouds.blobstore.BlobStoreContext;
import org.jclouds.blobstore.attr.ConsistencyModel;
import org.jclouds.internal.BaseView;
import org.jclouds.location.Provider;
import org.jclouds.rest.Utils;
import org.jclouds.util.Closeables2;

import com.google.common.reflect.TypeToken;

@Singleton
public class BlobStoreContextImpl extends BaseView implements BlobStoreContext {
   private final BlobStore blobStore;
   private final ConsistencyModel consistencyModel;
   private final Utils utils;
   private final BlobRequestSigner blobRequestSigner;

   @Inject
   public BlobStoreContextImpl(@Provider Context backend, @Provider TypeToken<? extends Context> backendType,
         Utils utils, ConsistencyModel consistencyModel, BlobStore blobStore, BlobRequestSigner blobRequestSigner) {
      super(backend, backendType);
      this.consistencyModel = checkNotNull(consistencyModel, "consistencyModel");
      this.blobStore = checkNotNull(blobStore, "blobStore");
      this.utils = checkNotNull(utils, "utils");
      this.blobRequestSigner = checkNotNull(blobRequestSigner, "blobRequestSigner");
   }

   @Override
   public ConsistencyModel getConsistencyModel() {
      return consistencyModel;
   }

   @Override
   public BlobStore getBlobStore() {
      return blobStore;
   }

   @Override
   public Utils utils() {
      return utils;
   }

   @Override
   public BlobRequestSigner getSigner() {
      return blobRequestSigner;
   }

   @Override
   public void close() {
      Closeables2.closeQuietly(delegate());
   }

   public int hashCode() {
      return delegate().hashCode();
   }

   @Override
   public String toString() {
      return delegate().toString();
   }

   @Override
   public boolean equals(Object obj) {
      return delegate().equals(obj);
   }

}
