/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jena.riot.resultset;

import static org.apache.jena.riot.resultset.ResultSetLang.SPARQLResultSetCSV ;
import static org.apache.jena.riot.resultset.ResultSetLang.SPARQLResultSetJSON ;
import static org.apache.jena.riot.resultset.ResultSetLang.SPARQLResultSetTSV ;
import static org.apache.jena.riot.resultset.ResultSetLang.SPARQLResultSetText ;
import static org.apache.jena.riot.resultset.ResultSetLang.SPARQLResultSetThrift ;
import static org.apache.jena.riot.resultset.ResultSetLang.SPARQLResultSetXML ;

import java.io.OutputStream ;
import java.io.Writer ;
import java.util.HashMap ;
import java.util.Map ;
import java.util.Objects ;

import org.apache.jena.atlas.lib.NotImplemented ;
import org.apache.jena.query.ResultSet ;
import org.apache.jena.riot.Lang ;
import org.apache.jena.riot.RiotException ;
import org.apache.jena.riot.thrift.BinRDF ;
import org.apache.jena.sparql.core.Prologue ;
import org.apache.jena.sparql.resultset.* ;
import org.apache.jena.sparql.serializer.SerializationContext ;
import org.apache.jena.sparql.util.Context ;

public class ResultSetWriterRegistry {

    private static Map<Lang, ResultSetWriterFactory> registry = new HashMap<>() ;
    
    /** Lookup a {@link Lang} to get the registered {@link ResultSetReaderFactory} (or null) */
    public static ResultSetWriterFactory lookup(Lang lang) {
        Objects.requireNonNull(lang) ;
        return registry.get(lang) ;
    }

    /** Register a {@link ResultSetReaderFactory} for a {@link Lang} */
    public static void register(Lang lang, ResultSetWriterFactory factory) {
        Objects.requireNonNull(lang) ;
        Objects.requireNonNull(factory) ;
        registry.put(lang, factory) ;
    }
    
    private static boolean initialized = false ;
    public static void init() {
        if ( initialized )
            return ;
        initialized = true ;

        ResultSetWriterFactory factory = new ResultSetWriterFactoryStd() ;
        register(SPARQLResultSetXML,    factory) ;
        register(SPARQLResultSetJSON,   factory) ;
        register(SPARQLResultSetCSV,    factory) ;
        register(SPARQLResultSetTSV,    factory) ;
        register(SPARQLResultSetThrift, new ResultSetWriterThriftFactory()) ;
        register(SPARQLResultSetText,   factory) ;
    }
 
    private static ResultSetWriter writerXML = new ResultSetWriter() {
        @Override public void write(OutputStream out, ResultSet resultSet, Context context) { 
            XMLOutput xOut = new XMLOutput(null) ;
            xOut.format(out, resultSet) ;
        }
        @Override public void write(Writer out, ResultSet resultSet, Context context) {throw new NotImplemented("Writer") ; }
        @Override public void write(OutputStream out, boolean result, Context context) {
            XMLOutput xOut = new XMLOutput(null);
            xOut.format(out, result);
        }
    } ;

    private static ResultSetWriter writerJSON = new ResultSetWriter() {
        @Override public void write(OutputStream out, ResultSet resultSet, Context context) {
            JSONOutput jOut = new JSONOutput() ;
            jOut.format(out, resultSet) ; 
        }
        @Override public void write(Writer out, ResultSet resultSet, Context context) {throw new NotImplemented("Writer") ; }
        @Override public void write(OutputStream out, boolean result, Context context) {
            JSONOutput jOut = new JSONOutput() ;
            jOut.format(out, result) ; 
        }
    } ;
    
    private static ResultSetWriter writerCSV = new ResultSetWriter() {
        @Override public void write(OutputStream out, ResultSet resultSet, Context context) {
            CSVOutput fmt = new CSVOutput() ;
            fmt.format(out, resultSet) ;
        }
        @Override public void write(Writer out, ResultSet resultSet, Context context) {throw new NotImplemented("Writer") ; }
        @Override public void write(OutputStream out, boolean result, Context context) {
            CSVOutput fmt = new CSVOutput() ;
            fmt.format(out, result) ;
        }
    } ;

    private static ResultSetWriter writerTSV = new ResultSetWriter() {
        @Override public void write(OutputStream out, ResultSet resultSet, Context context) {
            TSVOutput fmt = new TSVOutput() ;
            fmt.format(out, resultSet) ;
        }
        @Override public void write(Writer out, ResultSet resultSet, Context context) {throw new NotImplemented("Writer") ; }
        @Override public void write(OutputStream out, boolean result, Context context) {
            TSVOutput fmt = new TSVOutput() ;
            fmt.format(out, result) ;
        }
    } ;

    private static ResultSetWriter writerNo = new ResultSetWriter() {
        @Override public void write(OutputStream out, ResultSet resultSet, Context context) {}
        @Override public void write(Writer out, ResultSet resultSet, Context context)       {}
        @Override public void write(OutputStream out, boolean result, Context context) {}
    } ;

    private static ResultSetWriter writerText = new ResultSetWriter() {
        @Override public void write(OutputStream out, ResultSet resultSet, Context context) {
            // Prefix mapp
            TextOutput tFmt = new TextOutput(new SerializationContext((Prologue)null)) ;
            tFmt.format(out, resultSet) ; 
        }
        @Override public void write(Writer out, ResultSet resultSet, Context context) {throw new NotImplemented("Writer") ; }
        @Override public void write(OutputStream out, boolean result, Context context) {
            // Prefix mapp
            TextOutput tFmt = new TextOutput(new SerializationContext((Prologue)null)) ;
            tFmt.format(out, result) ; 
        }
    } ;
    
    private static class ResultSetWriterFactoryStd implements ResultSetWriterFactory {
        @Override
        public ResultSetWriter create(Lang lang) {
            lang = Objects.requireNonNull(lang, "Language must not be null") ;
            if ( lang.equals(SPARQLResultSetXML) )      return writerXML ;
            if ( lang.equals(SPARQLResultSetJSON) )     return writerJSON ;
            if ( lang.equals(SPARQLResultSetCSV) )      return writerCSV ;
            if ( lang.equals(SPARQLResultSetTSV) )      return writerTSV ;
            if ( lang.equals(SPARQLResultSetText) )     return writerText ;
            throw new RiotException("Lang not registered (ResultSet writer)") ;
        }
    }
    
    private static class ResultSetWriterThriftFactory implements ResultSetWriterFactory {
        @Override
        public ResultSetWriter create(Lang lang) {
            return new ResultSetWriter() {
                @Override
                public void write(OutputStream out, ResultSet resultSet, Context context)
                { BinRDF.writeResultSet(out, resultSet) ; }
                
                @Override
                public void write(Writer out, ResultSet resultSet, Context context) {
                    throw new NotImplemented("Writing binary data to a java.io.Writer is not possible") ;
                }
                @Override
                public void write(OutputStream out, boolean result, Context context)
                { throw new NotImplemented("No Thrift RDF encoding defined for boolean results"); }
            } ;
        }
    }
}

