/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jena.geosparql.spatial;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.lang.invoke.MethodHandles;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import org.apache.jena.atlas.RuntimeIOException;
import org.apache.jena.atlas.io.IOX;
import org.apache.jena.geosparql.configuration.GeoSPARQLOperations;
import org.apache.jena.geosparql.implementation.GeometryWrapper;
import org.apache.jena.geosparql.implementation.SRSInfo;
import org.apache.jena.geosparql.implementation.registry.SRSRegistry;
import org.apache.jena.geosparql.implementation.vocabulary.Geo;
import org.apache.jena.geosparql.implementation.vocabulary.SpatialExtension;
import org.apache.jena.geosparql.spatial.ConvertLatLon;
import org.apache.jena.geosparql.spatial.SpatialIndexException;
import org.apache.jena.geosparql.spatial.SpatialIndexItem;
import org.apache.jena.geosparql.spatial.SpatialIndexStorage;
import org.apache.jena.query.Dataset;
import org.apache.jena.query.DatasetFactory;
import org.apache.jena.query.ReadWrite;
import org.apache.jena.rdf.model.Literal;
import org.apache.jena.rdf.model.Model;
import org.apache.jena.rdf.model.NodeIterator;
import org.apache.jena.rdf.model.RDFNode;
import org.apache.jena.rdf.model.ResIterator;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.rdf.model.Statement;
import org.apache.jena.rdf.model.StmtIterator;
import org.apache.jena.sparql.engine.ExecutionContext;
import org.apache.jena.sparql.util.Context;
import org.apache.jena.sparql.util.Symbol;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.index.strtree.STRtree;
import org.opengis.geometry.MismatchedDimensionException;
import org.opengis.referencing.operation.TransformException;
import org.opengis.util.FactoryException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SpatialIndex {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    public static final Symbol SPATIAL_INDEX_SYMBOL = Symbol.create((String)"http://jena.apache.org/spatial#index");
    private final transient SRSInfo srsInfo;
    private boolean isBuilt;
    private final STRtree strTree;
    private static final int MINIMUM_CAPACITY = 2;

    private SpatialIndex() {
        this.strTree = new STRtree(2);
        this.isBuilt = true;
        this.strTree.build();
        this.srsInfo = SRSRegistry.getSRSInfo("http://www.opengis.net/def/crs/OGC/1.3/CRS84");
    }

    public SpatialIndex(int capacity, String srsURI) {
        int indexCapacity = capacity < 2 ? 2 : capacity;
        this.strTree = new STRtree(indexCapacity);
        this.isBuilt = false;
        this.srsInfo = SRSRegistry.getSRSInfo(srsURI);
    }

    public SpatialIndex(Collection<SpatialIndexItem> spatialIndexItems, String srsURI) throws SpatialIndexException {
        int indexCapacity = spatialIndexItems.size() < 2 ? 2 : spatialIndexItems.size();
        this.strTree = new STRtree(indexCapacity);
        this.insertItems(spatialIndexItems);
        this.strTree.build();
        this.isBuilt = true;
        this.srsInfo = SRSRegistry.getSRSInfo(srsURI);
    }

    public SRSInfo getSrsInfo() {
        return this.srsInfo;
    }

    public boolean isEmpty() {
        return this.strTree.isEmpty();
    }

    public boolean isBuilt() {
        return this.isBuilt;
    }

    public void build() {
        if (!this.isBuilt) {
            this.strTree.build();
            this.isBuilt = true;
        }
    }

    public final void insertItems(Collection<SpatialIndexItem> indexItems) throws SpatialIndexException {
        for (SpatialIndexItem indexItem : indexItems) {
            this.insertItem(indexItem.getEnvelope(), indexItem.getItem());
        }
    }

    public final void insertItem(Envelope envelope, Resource item) throws SpatialIndexException {
        if (this.isBuilt) {
            throw new SpatialIndexException("SpatialIndex has been built and cannot have additional items.");
        }
        this.strTree.insert(envelope, (Object)item);
    }

    public HashSet<Resource> query(Envelope searchEnvelope) {
        if (!this.strTree.isEmpty()) {
            return new HashSet<Resource>(this.strTree.query(searchEnvelope));
        }
        return new HashSet<Resource>();
    }

    public String toString() {
        return "SpatialIndex{srsInfo=" + this.srsInfo + ", isBuilt=" + this.isBuilt + ", strTree=" + this.strTree + "}";
    }

    public static final SpatialIndex retrieve(ExecutionContext execCxt) throws SpatialIndexException {
        Context context = execCxt.getContext();
        SpatialIndex spatialIndex = (SpatialIndex)context.get(SPATIAL_INDEX_SYMBOL, null);
        if (spatialIndex == null) {
            throw new SpatialIndexException("Dataset Context does not contain SpatialIndex.");
        }
        return spatialIndex;
    }

    public static final boolean isDefined(ExecutionContext execCxt) {
        Context context = execCxt.getContext();
        return context.isDefined(SPATIAL_INDEX_SYMBOL);
    }

    public static final void setSpatialIndex(Dataset dataset, SpatialIndex spatialIndex) {
        Context context = dataset.getContext();
        context.set(SPATIAL_INDEX_SYMBOL, (Object)spatialIndex);
    }

    public static SpatialIndex buildSpatialIndex(Dataset dataset, String srsURI, File spatialIndexFile) throws SpatialIndexException {
        SpatialIndex spatialIndex = SpatialIndex.load(spatialIndexFile);
        if (spatialIndex.isEmpty()) {
            Collection<SpatialIndexItem> spatialIndexItems = SpatialIndex.findSpatialIndexItems(dataset, srsURI);
            SpatialIndex.save(spatialIndexFile, spatialIndexItems, srsURI);
            spatialIndex = new SpatialIndex(spatialIndexItems, srsURI);
            spatialIndex.build();
        }
        SpatialIndex.setSpatialIndex(dataset, spatialIndex);
        return spatialIndex;
    }

    public static SpatialIndex buildSpatialIndex(Dataset dataset, File spatialIndexFile) throws SpatialIndexException {
        String srsURI = GeoSPARQLOperations.findModeSRS(dataset);
        SpatialIndex spatialIndex = SpatialIndex.buildSpatialIndex(dataset, srsURI, spatialIndexFile);
        return spatialIndex;
    }

    public static SpatialIndex buildSpatialIndex(Dataset dataset, String srsURI) throws SpatialIndexException {
        LOGGER.info("Building Spatial Index - Started");
        Collection<SpatialIndexItem> items = SpatialIndex.findSpatialIndexItems(dataset, srsURI);
        SpatialIndex spatialIndex = new SpatialIndex(items, srsURI);
        spatialIndex.build();
        SpatialIndex.setSpatialIndex(dataset, spatialIndex);
        LOGGER.info("Building Spatial Index - Completed");
        return spatialIndex;
    }

    public static Collection<SpatialIndexItem> findSpatialIndexItems(Dataset dataset, String srsURI) throws SpatialIndexException {
        dataset.begin(ReadWrite.READ);
        Model defaultModel = dataset.getDefaultModel();
        Collection<SpatialIndexItem> items = SpatialIndex.getSpatialIndexItems(defaultModel, srsURI);
        Iterator graphNames = dataset.listNames();
        while (graphNames.hasNext()) {
            String graphName = (String)graphNames.next();
            Model namedModel = dataset.getNamedModel(graphName);
            Collection<SpatialIndexItem> graphItems = SpatialIndex.getSpatialIndexItems(namedModel, srsURI);
            items.addAll(graphItems);
        }
        dataset.end();
        return items;
    }

    public static SpatialIndex buildSpatialIndex(Dataset dataset) throws SpatialIndexException {
        String srsURI = GeoSPARQLOperations.findModeSRS(dataset);
        SpatialIndex spatialIndex = SpatialIndex.buildSpatialIndex(dataset, srsURI);
        return spatialIndex;
    }

    public static final Dataset wrapModel(Model model, String srsURI) throws SpatialIndexException {
        Dataset dataset = DatasetFactory.createTxnMem();
        dataset.setDefaultModel(model);
        SpatialIndex.buildSpatialIndex(dataset, srsURI);
        return dataset;
    }

    public static final Dataset wrapModel(Model model) throws SpatialIndexException {
        Dataset dataset = DatasetFactory.createTxnMem();
        dataset.setDefaultModel(model);
        String srsURI = GeoSPARQLOperations.findModeSRS(dataset);
        SpatialIndex.buildSpatialIndex(dataset, srsURI);
        return dataset;
    }

    public static final Collection<SpatialIndexItem> getSpatialIndexItems(Model model, String srsURI) throws SpatialIndexException {
        ArrayList<SpatialIndexItem> items = new ArrayList<SpatialIndexItem>();
        if (model.contains(null, Geo.HAS_GEOMETRY_PROP, (RDFNode)((Resource)null))) {
            LOGGER.info("Feature-hasGeometry-Geometry statements found.");
            if (model.contains(null, SpatialExtension.GEO_LAT_PROP, (RDFNode)((Literal)null))) {
                LOGGER.warn("Lat/Lon Geo predicates also found but will not be added to index.");
            }
            Collection<SpatialIndexItem> geometryLiteralItems = SpatialIndex.getGeometryLiteralIndexItems(model, srsURI);
            items.addAll(geometryLiteralItems);
        } else if (model.contains(null, SpatialExtension.GEO_LAT_PROP, (RDFNode)((Literal)null))) {
            LOGGER.info("Geo predicate statements found.");
            Collection<SpatialIndexItem> geoPredicateItems = SpatialIndex.getGeoPredicateIndexItems(model, srsURI);
            items.addAll(geoPredicateItems);
        }
        return items;
    }

    private static Collection<SpatialIndexItem> getGeometryLiteralIndexItems(Model model, String srsURI) throws SpatialIndexException {
        ArrayList<SpatialIndexItem> items = new ArrayList<SpatialIndexItem>();
        StmtIterator stmtIt = model.listStatements(null, Geo.HAS_GEOMETRY_PROP, (RDFNode)((Resource)null));
        while (stmtIt.hasNext()) {
            Statement stmt = stmtIt.nextStatement();
            Resource feature = stmt.getSubject();
            Resource geometry = stmt.getResource();
            NodeIterator nodeIter = model.listObjectsOfProperty(geometry, Geo.HAS_SERIALIZATION_PROP);
            if (!nodeIter.hasNext()) {
                NodeIterator wktNodeIter = model.listObjectsOfProperty(geometry, Geo.AS_WKT_PROP);
                NodeIterator gmlNodeIter = model.listObjectsOfProperty(geometry, Geo.AS_GML_PROP);
                nodeIter = wktNodeIter.andThen((Iterator)gmlNodeIter);
            }
            while (nodeIter.hasNext()) {
                Literal geometryLiteral = ((RDFNode)nodeIter.next()).asLiteral();
                GeometryWrapper geometryWrapper = GeometryWrapper.extract(geometryLiteral);
                try {
                    GeometryWrapper transformedGeometryWrapper = geometryWrapper.convertSRS(srsURI);
                    Envelope envelope = transformedGeometryWrapper.getEnvelope();
                    SpatialIndexItem item = new SpatialIndexItem(envelope, feature);
                    items.add(item);
                }
                catch (MismatchedDimensionException | TransformException | FactoryException ex) {
                    throw new SpatialIndexException("Transformation Exception: " + geometryLiteral + ". " + ex.getMessage());
                }
            }
        }
        return items;
    }

    private static Collection<SpatialIndexItem> getGeoPredicateIndexItems(Model model, String srsURI) throws SpatialIndexException {
        ArrayList<SpatialIndexItem> items = new ArrayList<SpatialIndexItem>();
        ResIterator resIt = model.listResourcesWithProperty(SpatialExtension.GEO_LAT_PROP);
        while (resIt.hasNext()) {
            Resource feature = resIt.nextResource();
            Literal lat = feature.getRequiredProperty(SpatialExtension.GEO_LAT_PROP).getLiteral();
            Literal lon = feature.getProperty(SpatialExtension.GEO_LON_PROP).getLiteral();
            if (lon == null) {
                LOGGER.warn("Geo predicates: latitude found but not longitude. " + feature);
                continue;
            }
            Literal latLonPoint = ConvertLatLon.toLiteral(lat.getFloat(), lon.getFloat());
            GeometryWrapper geometryWrapper = GeometryWrapper.extract(latLonPoint);
            try {
                GeometryWrapper transformedGeometryWrapper = geometryWrapper.convertSRS(srsURI);
                Envelope envelope = transformedGeometryWrapper.getEnvelope();
                SpatialIndexItem item = new SpatialIndexItem(envelope, feature);
                items.add(item);
            }
            catch (MismatchedDimensionException | TransformException | FactoryException ex) {
                throw new SpatialIndexException("Transformation Exception: " + geometryWrapper.getLexicalForm() + ". " + ex.getMessage());
            }
        }
        return items;
    }

    public static final SpatialIndex load(File spatialIndexFile) throws SpatialIndexException {
        if (spatialIndexFile != null && spatialIndexFile.exists()) {
            SpatialIndex spatialIndex;
            LOGGER.info("Loading Spatial Index - Started: {}", (Object)spatialIndexFile.getAbsolutePath());
            ObjectInputStream in = new ObjectInputStream(new FileInputStream(spatialIndexFile));
            try {
                SpatialIndexStorage storage = (SpatialIndexStorage)in.readObject();
                SpatialIndex spatialIndex2 = storage.getSpatialIndex();
                LOGGER.info("Loading Spatial Index - Completed: {}", (Object)spatialIndexFile.getAbsolutePath());
                spatialIndex = spatialIndex2;
            }
            catch (Throwable throwable) {
                try {
                    try {
                        in.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                catch (IOException | ClassNotFoundException ex) {
                    throw new SpatialIndexException("Loading Exception: " + ex.getMessage(), ex);
                }
            }
            in.close();
            return spatialIndex;
        }
        return new SpatialIndex();
    }

    public static final void save(String spatialIndexFileURI, Collection<SpatialIndexItem> spatialIndexItems, String srsURI) throws SpatialIndexException {
        SpatialIndex.save(new File(spatialIndexFileURI), spatialIndexItems, srsURI);
    }

    public static final void save(File spatialIndexFile, Collection<SpatialIndexItem> spatialIndexItems, String srsURI) throws SpatialIndexException {
        if (spatialIndexFile != null) {
            LOGGER.info("Saving Spatial Index - Started: {}", (Object)spatialIndexFile.getAbsolutePath());
            SpatialIndexStorage storage = new SpatialIndexStorage(spatialIndexItems, srsURI);
            String filename = spatialIndexFile.getAbsolutePath();
            Path file = Path.of(filename, new String[0]);
            Path tmpFile = IOX.uniqueDerivedPath((Path)file, null);
            try {
                Files.deleteIfExists(file);
            }
            catch (IOException ex) {
                throw new SpatialIndexException("Failed to delete file: " + ex.getMessage());
            }
            try {
                IOX.safeWriteOrCopy((Path)file, (Path)tmpFile, out -> {
                    ObjectOutputStream oos = new ObjectOutputStream((OutputStream)out);
                    oos.writeObject(storage);
                });
            }
            catch (RuntimeIOException ex) {
                throw new SpatialIndexException("Save Exception: " + ex.getMessage());
            }
            finally {
                LOGGER.info("Saving Spatial Index - Completed: {}", (Object)spatialIndexFile.getAbsolutePath());
            }
        }
    }
}

