/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jena.geosparql.implementation.parsers.wkt;

import java.util.Arrays;
import java.util.Objects;
import org.apache.jena.datatypes.DatatypeFormatException;
import org.apache.jena.geosparql.implementation.DimensionInfo;
import org.apache.jena.geosparql.implementation.jts.CoordinateSequenceDimensions;
import org.apache.jena.geosparql.implementation.jts.CustomCoordinateSequence;
import org.apache.jena.geosparql.implementation.jts.CustomGeometryFactory;
import org.apache.jena.geosparql.implementation.parsers.ParserReader;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.Polygon;

public class WKTReader
implements ParserReader {
    private static final GeometryFactory GEOMETRY_FACTORY = CustomGeometryFactory.theInstance();
    private final CoordinateSequenceDimensions dims;
    private final Geometry geometry;
    private final DimensionInfo dimensionInfo;
    private final String srsURI;

    protected WKTReader(String geometryType, String dimensionString, String coordinates, String srsURI) {
        this.dims = WKTReader.convertDimensionString(dimensionString);
        this.geometry = this.buildGeometry(geometryType, coordinates);
        this.dimensionInfo = new DimensionInfo(this.dims, this.geometry.getDimension());
        this.srsURI = srsURI;
    }

    protected WKTReader(String geometryType, String dimensionString, String coordinates) {
        this(geometryType, dimensionString, coordinates, "http://www.opengis.net/def/crs/OGC/1.3/CRS84");
    }

    protected WKTReader() {
        this("point", "", "", "http://www.opengis.net/def/crs/OGC/1.3/CRS84");
    }

    @Override
    public Geometry getGeometry() {
        return this.geometry;
    }

    @Override
    public CoordinateSequenceDimensions getDimensions() {
        return this.dims;
    }

    @Override
    public DimensionInfo getDimensionInfo() {
        return this.dimensionInfo;
    }

    @Override
    public String getSrsURI() {
        return this.srsURI;
    }

    private static CoordinateSequenceDimensions convertDimensionString(String dimensionsString) {
        return switch (dimensionsString) {
            case "zm" -> CoordinateSequenceDimensions.XYZM;
            case "z" -> CoordinateSequenceDimensions.XYZ;
            case "m" -> CoordinateSequenceDimensions.XYM;
            default -> CoordinateSequenceDimensions.XY;
        };
    }

    private Geometry buildGeometry(String geometryType, String coordinates) throws DatatypeFormatException {
        Polygon geo;
        try {
            switch (geometryType) {
                case "point": {
                    CustomCoordinateSequence pointSequence = new CustomCoordinateSequence(this.dims, this.clean(coordinates));
                    geo = GEOMETRY_FACTORY.createPoint((CoordinateSequence)pointSequence);
                    break;
                }
                case "linestring": {
                    CustomCoordinateSequence lineSequence = new CustomCoordinateSequence(this.dims, this.clean(coordinates));
                    geo = GEOMETRY_FACTORY.createLineString((CoordinateSequence)lineSequence);
                    break;
                }
                case "linearring": {
                    CustomCoordinateSequence linearSequence = new CustomCoordinateSequence(this.dims, this.clean(coordinates));
                    geo = GEOMETRY_FACTORY.createLinearRing((CoordinateSequence)linearSequence);
                    break;
                }
                case "polygon": {
                    geo = this.buildPolygon(coordinates);
                    break;
                }
                case "multipoint": {
                    CustomCoordinateSequence multiPointSequence = new CustomCoordinateSequence(this.dims, this.clean(coordinates));
                    geo = GEOMETRY_FACTORY.createMultiPoint((CoordinateSequence)multiPointSequence);
                    break;
                }
                case "multilinestring": {
                    geo = this.buildMultiLineString(coordinates);
                    break;
                }
                case "multipolygon": {
                    geo = this.buildMultiPolygon(coordinates);
                    break;
                }
                case "geometrycollection": {
                    geo = this.buildGeometryCollection(coordinates);
                    break;
                }
                default: {
                    throw new DatatypeFormatException("Geometry type not supported: " + geometryType);
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException ex) {
            throw new DatatypeFormatException("Build WKT Geometry Exception - Type: " + geometryType + ", Coordinates: " + coordinates + ". " + ex.getMessage());
        }
        return geo;
    }

    private String clean(String unclean) {
        return unclean.replace(")", "").replace("(", "").trim();
    }

    private Geometry buildGeometryCollection(String coordinates) throws DatatypeFormatException {
        if (coordinates.isEmpty()) {
            return GEOMETRY_FACTORY.createGeometryCollection(new Geometry[0]);
        }
        String tidied = coordinates.substring(1, coordinates.length() - 1);
        tidied = tidied.replaceAll("[\\ ]?,[\\ ]?", ",");
        String[] partCoordinates = tidied.split("\\),(?=[^\\(])");
        Geometry[] geometries = new Geometry[partCoordinates.length];
        for (int i = 0; i < partCoordinates.length; ++i) {
            WKTReader partWKTInfo = WKTReader.extract(partCoordinates[i]);
            geometries[i] = partWKTInfo.geometry;
        }
        return GEOMETRY_FACTORY.createGeometryCollection(geometries);
    }

    private Geometry buildMultiLineString(String coordinates) {
        if (coordinates.isEmpty()) {
            return GEOMETRY_FACTORY.createMultiLineString(new LineString[0]);
        }
        String[] splitCoordinates = this.splitCoordinates(coordinates);
        LineString[] lineStrings = this.splitLineStrings(splitCoordinates);
        return GEOMETRY_FACTORY.createMultiLineString(lineStrings);
    }

    private Geometry buildMultiPolygon(String coordinates) {
        if (coordinates.isEmpty()) {
            return GEOMETRY_FACTORY.createMultiPolygon(new Polygon[0]);
        }
        String trimmed = coordinates.replace(")) ,", ")),");
        String[] multiCoordinates = trimmed.split("\\)\\),");
        Polygon[] polygons = new Polygon[multiCoordinates.length];
        for (int i = 0; i < multiCoordinates.length; ++i) {
            polygons[i] = this.buildPolygon(multiCoordinates[i]);
        }
        return GEOMETRY_FACTORY.createMultiPolygon(polygons);
    }

    private Polygon buildPolygon(String coordinates) {
        Polygon polygon;
        String[] splitCoordinates = this.splitCoordinates(coordinates);
        if (splitCoordinates.length == 1) {
            CustomCoordinateSequence shellSequence = new CustomCoordinateSequence(this.dims, this.clean(coordinates));
            polygon = GEOMETRY_FACTORY.createPolygon((CoordinateSequence)shellSequence);
        } else {
            String shellCoordinates = splitCoordinates[0];
            CustomCoordinateSequence shellSequence = new CustomCoordinateSequence(this.dims, this.clean(shellCoordinates));
            LinearRing shellLinearRing = GEOMETRY_FACTORY.createLinearRing((CoordinateSequence)shellSequence);
            String[] splitHoleCoordinates = Arrays.copyOfRange(splitCoordinates, 1, splitCoordinates.length);
            LinearRing[] holesLinearRing = this.splitLinearRings(this.dims, splitHoleCoordinates);
            polygon = GEOMETRY_FACTORY.createPolygon(shellLinearRing, holesLinearRing);
        }
        return polygon;
    }

    private String[] splitCoordinates(String coordinates) {
        String trimmed = coordinates.replace(") ,", "),");
        return trimmed.split("\\),");
    }

    private LineString[] splitLineStrings(String[] splitCoordinates) {
        LineString[] lineStrings = new LineString[splitCoordinates.length];
        for (int i = 0; i < splitCoordinates.length; ++i) {
            LineString lineString;
            CustomCoordinateSequence sequence = new CustomCoordinateSequence(this.dims, this.clean(splitCoordinates[i]));
            lineStrings[i] = lineString = GEOMETRY_FACTORY.createLineString((CoordinateSequence)sequence);
        }
        return lineStrings;
    }

    private LinearRing[] splitLinearRings(CoordinateSequenceDimensions dims, String[] splitCoordinates) {
        LinearRing[] linearRings = new LinearRing[splitCoordinates.length];
        for (int i = 0; i < splitCoordinates.length; ++i) {
            LinearRing linearRing;
            CustomCoordinateSequence sequence = new CustomCoordinateSequence(dims, this.clean(splitCoordinates[i]));
            linearRings[i] = linearRing = GEOMETRY_FACTORY.createLinearRing((CoordinateSequence)sequence);
        }
        return linearRings;
    }

    public static WKTReader extract(String geometryLiteral) throws DatatypeFormatException {
        WKTTextSRS wktTextSRS = new WKTTextSRS(geometryLiteral);
        String srsURI = wktTextSRS.srsURI;
        String wktText = wktTextSRS.wktText;
        String goemetryType = "point";
        String dimension = "";
        String coordinates = "";
        if (!wktText.equals("")) {
            String remainder;
            wktText = wktText.trim();
            String[] parts = (wktText = wktText.toLowerCase()).split("\\(", 2);
            if (parts.length == 1) {
                remainder = parts[0].replace("empty", "").trim();
            } else {
                int coordinatesStart = wktText.indexOf("(");
                coordinates = wktText.substring(coordinatesStart);
                remainder = parts[0].trim();
            }
            int firstSpace = remainder.indexOf(" ");
            if (firstSpace != -1) {
                goemetryType = remainder.substring(0, firstSpace);
                dimension = remainder.substring(firstSpace + 1);
            } else {
                goemetryType = remainder;
            }
        }
        return new WKTReader(goemetryType, dimension, coordinates, srsURI);
    }

    public int hashCode() {
        int hash = 7;
        hash = 83 * hash + Objects.hashCode((Object)this.dims);
        hash = 83 * hash + Objects.hashCode(this.geometry);
        hash = 83 * hash + Objects.hashCode(this.dimensionInfo);
        hash = 83 * hash + Objects.hashCode(this.srsURI);
        return hash;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        WKTReader other = (WKTReader)obj;
        if (!Objects.equals(this.srsURI, other.srsURI)) {
            return false;
        }
        if (this.dims != other.dims) {
            return false;
        }
        if (!Objects.equals(this.geometry, other.geometry)) {
            return false;
        }
        return Objects.equals(this.dimensionInfo, other.dimensionInfo);
    }

    public String toString() {
        return "WKTReader{dims=" + String.valueOf((Object)this.dims) + ", geometry=" + String.valueOf(this.geometry) + ", dimensionInfo=" + String.valueOf(this.dimensionInfo) + ", srsURI=" + this.srsURI + "}";
    }

    private static class WKTTextSRS {
        private final String wktText;
        private final String srsURI;

        public WKTTextSRS(String wktLiteral) {
            int startSRS = wktLiteral.indexOf("<");
            int endSRS = wktLiteral.indexOf(">");
            if (startSRS != -1 && endSRS != -1) {
                this.srsURI = wktLiteral.substring(startSRS + 1, endSRS);
                this.wktText = wktLiteral.substring(endSRS + 1);
            } else {
                this.srsURI = "http://www.opengis.net/def/crs/OGC/1.3/CRS84";
                this.wktText = wktLiteral;
            }
        }

        public String getWktText() {
            return this.wktText;
        }

        public String getSrsURI() {
            return this.srsURI;
        }
    }
}

