/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.record;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.FileRecords;
import org.apache.kafka.common.record.LogEntry;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.Records;
import org.apache.kafka.test.TestUtils;
import org.easymock.EasyMock;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class FileRecordsTest {
    private Record[] records = new Record[]{Record.create((byte[])"abcd".getBytes()), Record.create((byte[])"efgh".getBytes()), Record.create((byte[])"ijkl".getBytes())};
    private FileRecords fileRecords;

    @Before
    public void setup() throws IOException {
        this.fileRecords = this.createFileRecords(this.records);
    }

    @Test
    public void testFileSize() throws IOException {
        Assert.assertEquals((long)this.fileRecords.channel().size(), (long)this.fileRecords.sizeInBytes());
        for (int i = 0; i < 20; ++i) {
            this.fileRecords.append(MemoryRecords.withRecords((Record[])new Record[]{Record.create((byte[])"abcd".getBytes())}));
            Assert.assertEquals((long)this.fileRecords.channel().size(), (long)this.fileRecords.sizeInBytes());
        }
    }

    @Test
    public void testIterationOverPartialAndTruncation() throws IOException {
        this.testPartialWrite(0, this.fileRecords);
        this.testPartialWrite(2, this.fileRecords);
        this.testPartialWrite(4, this.fileRecords);
        this.testPartialWrite(5, this.fileRecords);
        this.testPartialWrite(6, this.fileRecords);
    }

    private void testPartialWrite(int size, FileRecords fileRecords) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(size);
        for (int i = 0; i < size; ++i) {
            buffer.put((byte)0);
        }
        buffer.rewind();
        fileRecords.channel().write(buffer);
        TestUtils.checkEquals(Arrays.asList(this.records), fileRecords.records());
    }

    @Test
    public void testIterationDoesntChangePosition() throws IOException {
        long position = this.fileRecords.channel().position();
        TestUtils.checkEquals(Arrays.asList(this.records), this.fileRecords.records());
        Assert.assertEquals((long)position, (long)this.fileRecords.channel().position());
    }

    @Test
    public void testRead() throws IOException {
        FileRecords read = this.fileRecords.read(0, this.fileRecords.sizeInBytes());
        TestUtils.checkEquals(this.fileRecords.shallowEntries(), read.shallowEntries());
        List<LogEntry> items = FileRecordsTest.shallowEntries((Records)read);
        LogEntry second = items.get(1);
        read = this.fileRecords.read(second.sizeInBytes(), this.fileRecords.sizeInBytes());
        Assert.assertEquals((String)"Try a read starting from the second message", items.subList(1, 3), FileRecordsTest.shallowEntries((Records)read));
        read = this.fileRecords.read(second.sizeInBytes(), second.sizeInBytes());
        Assert.assertEquals((String)"Try a read of a single message starting from the second message", Collections.singletonList(second), FileRecordsTest.shallowEntries((Records)read));
    }

    @Test
    public void testSearch() throws IOException {
        Record lastMessage = Record.create((byte[])"test".getBytes());
        this.fileRecords.append(MemoryRecords.withRecords((long)50L, (Record[])new Record[]{lastMessage}));
        List<LogEntry> entries = FileRecordsTest.shallowEntries((Records)this.fileRecords);
        int position = 0;
        int message1Size = entries.get(0).sizeInBytes();
        Assert.assertEquals((String)"Should be able to find the first message by its offset", (Object)new FileRecords.LogEntryPosition(0L, position, message1Size), (Object)this.fileRecords.searchForOffsetWithSize(0L, 0));
        int message2Size = entries.get(1).sizeInBytes();
        Assert.assertEquals((String)"Should be able to find second message when starting from 0", (Object)new FileRecords.LogEntryPosition(1L, position += message1Size, message2Size), (Object)this.fileRecords.searchForOffsetWithSize(1L, 0));
        Assert.assertEquals((String)"Should be able to find second message starting from its offset", (Object)new FileRecords.LogEntryPosition(1L, position, message2Size), (Object)this.fileRecords.searchForOffsetWithSize(1L, position));
        int message4Size = entries.get(3).sizeInBytes();
        Assert.assertEquals((String)"Should be able to find fourth message from a non-existant offset", (Object)new FileRecords.LogEntryPosition(50L, position += message2Size + entries.get(2).sizeInBytes(), message4Size), (Object)this.fileRecords.searchForOffsetWithSize(3L, position));
        Assert.assertEquals((String)"Should be able to find fourth message by correct offset", (Object)new FileRecords.LogEntryPosition(50L, position, message4Size), (Object)this.fileRecords.searchForOffsetWithSize(50L, position));
    }

    @Test
    public void testIteratorWithLimits() throws IOException {
        LogEntry entry = FileRecordsTest.shallowEntries((Records)this.fileRecords).get(1);
        int start = this.fileRecords.searchForOffsetWithSize((long)1L, (int)0).position;
        int size = entry.sizeInBytes();
        FileRecords slice = this.fileRecords.read(start, size);
        Assert.assertEquals(Collections.singletonList(entry), FileRecordsTest.shallowEntries((Records)slice));
        FileRecords slice2 = this.fileRecords.read(start, size - 1);
        Assert.assertEquals(Collections.emptyList(), FileRecordsTest.shallowEntries((Records)slice2));
    }

    @Test
    public void testTruncate() throws IOException {
        LogEntry entry = FileRecordsTest.shallowEntries((Records)this.fileRecords).get(0);
        int end = this.fileRecords.searchForOffsetWithSize((long)1L, (int)0).position;
        this.fileRecords.truncateTo(end);
        Assert.assertEquals(Collections.singletonList(entry), FileRecordsTest.shallowEntries((Records)this.fileRecords));
        Assert.assertEquals((long)entry.sizeInBytes(), (long)this.fileRecords.sizeInBytes());
    }

    @Test
    public void testTruncateNotCalledIfSizeIsSameAsTargetSize() throws IOException {
        FileChannel channelMock = (FileChannel)EasyMock.createMock(FileChannel.class);
        EasyMock.expect((Object)channelMock.size()).andReturn((Object)42L).atLeastOnce();
        EasyMock.expect((Object)channelMock.position(42L)).andReturn(null);
        EasyMock.replay((Object[])new Object[]{channelMock});
        FileRecords fileRecords = new FileRecords(TestUtils.tempFile(), channelMock, 0, Integer.MAX_VALUE, false);
        fileRecords.truncateTo(42);
        EasyMock.verify((Object[])new Object[]{channelMock});
    }

    @Test
    public void testTruncateNotCalledIfSizeIsBiggerThanTargetSize() throws IOException {
        FileChannel channelMock = (FileChannel)EasyMock.createMock(FileChannel.class);
        EasyMock.expect((Object)channelMock.size()).andReturn((Object)42L).atLeastOnce();
        EasyMock.expect((Object)channelMock.position(42L)).andReturn(null);
        EasyMock.replay((Object[])new Object[]{channelMock});
        FileRecords fileRecords = new FileRecords(TestUtils.tempFile(), channelMock, 0, Integer.MAX_VALUE, false);
        try {
            fileRecords.truncateTo(43);
            Assert.fail((String)"Should throw KafkaException");
        }
        catch (KafkaException kafkaException) {
            // empty catch block
        }
        EasyMock.verify((Object[])new Object[]{channelMock});
    }

    @Test
    public void testTruncateIfSizeIsDifferentToTargetSize() throws IOException {
        FileChannel channelMock = (FileChannel)EasyMock.createMock(FileChannel.class);
        EasyMock.expect((Object)channelMock.size()).andReturn((Object)42L).atLeastOnce();
        EasyMock.expect((Object)channelMock.position(42L)).andReturn(null).once();
        EasyMock.expect((Object)channelMock.truncate(23L)).andReturn(null).once();
        EasyMock.expect((Object)channelMock.position(23L)).andReturn(null).once();
        EasyMock.replay((Object[])new Object[]{channelMock});
        FileRecords fileRecords = new FileRecords(TestUtils.tempFile(), channelMock, 0, Integer.MAX_VALUE, false);
        fileRecords.truncateTo(23);
        EasyMock.verify((Object[])new Object[]{channelMock});
    }

    @Test
    public void testPreallocateTrue() throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords fileRecords = FileRecords.open((File)temp, (boolean)false, (int)0x20000000, (boolean)true);
        long position = fileRecords.channel().position();
        int size = fileRecords.sizeInBytes();
        Assert.assertEquals((long)0L, (long)position);
        Assert.assertEquals((long)0L, (long)size);
        Assert.assertEquals((long)0x20000000L, (long)temp.length());
    }

    @Test
    public void testPreallocateFalse() throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords set = FileRecords.open((File)temp, (boolean)false, (int)0x20000000, (boolean)false);
        long position = set.channel().position();
        int size = set.sizeInBytes();
        Assert.assertEquals((long)0L, (long)position);
        Assert.assertEquals((long)0L, (long)size);
        Assert.assertEquals((long)0L, (long)temp.length());
    }

    @Test
    public void testPreallocateClearShutdown() throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords set = FileRecords.open((File)temp, (boolean)false, (int)0x20000000, (boolean)true);
        set.append(MemoryRecords.withRecords((Record[])this.records));
        int oldPosition = (int)set.channel().position();
        int oldSize = set.sizeInBytes();
        Assert.assertEquals((long)this.fileRecords.sizeInBytes(), (long)oldPosition);
        Assert.assertEquals((long)this.fileRecords.sizeInBytes(), (long)oldSize);
        set.close();
        File tempReopen = new File(temp.getAbsolutePath());
        FileRecords setReopen = FileRecords.open((File)tempReopen, (boolean)true, (int)0x20000000, (boolean)true);
        int position = (int)setReopen.channel().position();
        int size = setReopen.sizeInBytes();
        Assert.assertEquals((long)oldPosition, (long)position);
        Assert.assertEquals((long)oldPosition, (long)size);
        Assert.assertEquals((long)oldPosition, (long)tempReopen.length());
    }

    @Test
    public void testFormatConversionWithPartialMessage() throws IOException {
        LogEntry entry = FileRecordsTest.shallowEntries((Records)this.fileRecords).get(1);
        int start = this.fileRecords.searchForOffsetWithSize((long)1L, (int)0).position;
        int size = entry.sizeInBytes();
        FileRecords slice = this.fileRecords.read(start, size - 1);
        Records messageV0 = slice.toMessageFormat((byte)0);
        Assert.assertTrue((String)"No message should be there", (boolean)FileRecordsTest.shallowEntries(messageV0).isEmpty());
        Assert.assertEquals((String)("There should be " + (size - 1) + " bytes"), (long)(size - 1), (long)messageV0.sizeInBytes());
    }

    @Test
    public void testConvertNonCompressedToMagic1() throws IOException {
        List<LogEntry> entries = Arrays.asList(LogEntry.create((long)0L, (Record)Record.create((byte)0, (long)-1L, (byte[])"k1".getBytes(), (byte[])"hello".getBytes())), LogEntry.create((long)2L, (Record)Record.create((byte)0, (long)-1L, (byte[])"k2".getBytes(), (byte[])"goodbye".getBytes())));
        MemoryRecords records = MemoryRecords.withLogEntries((CompressionType)CompressionType.NONE, entries);
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            fileRecords.append(records);
            fileRecords.flush();
            Records convertedRecords = fileRecords.toMessageFormat((byte)1);
            this.verifyConvertedMessageSet(entries, convertedRecords, (byte)1);
        }
    }

    @Test
    public void testConvertCompressedToMagic1() throws IOException {
        List<LogEntry> entries = Arrays.asList(LogEntry.create((long)0L, (Record)Record.create((byte)0, (long)-1L, (byte[])"k1".getBytes(), (byte[])"hello".getBytes())), LogEntry.create((long)2L, (Record)Record.create((byte)0, (long)-1L, (byte[])"k2".getBytes(), (byte[])"goodbye".getBytes())));
        MemoryRecords records = MemoryRecords.withLogEntries((CompressionType)CompressionType.GZIP, entries);
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            fileRecords.append(records);
            fileRecords.flush();
            Records convertedRecords = fileRecords.toMessageFormat((byte)1);
            this.verifyConvertedMessageSet(entries, convertedRecords, (byte)1);
        }
    }

    @Test
    public void testConvertNonCompressedToMagic0() throws IOException {
        List<LogEntry> entries = Arrays.asList(LogEntry.create((long)0L, (Record)Record.create((byte)1, (long)1L, (byte[])"k1".getBytes(), (byte[])"hello".getBytes())), LogEntry.create((long)2L, (Record)Record.create((byte)1, (long)2L, (byte[])"k2".getBytes(), (byte[])"goodbye".getBytes())));
        MemoryRecords records = MemoryRecords.withLogEntries((CompressionType)CompressionType.NONE, entries);
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            fileRecords.append(records);
            fileRecords.flush();
            Records convertedRecords = fileRecords.toMessageFormat((byte)0);
            this.verifyConvertedMessageSet(entries, convertedRecords, (byte)0);
        }
    }

    @Test
    public void testConvertCompressedToMagic0() throws IOException {
        List<LogEntry> entries = Arrays.asList(LogEntry.create((long)0L, (Record)Record.create((byte)1, (long)1L, (byte[])"k1".getBytes(), (byte[])"hello".getBytes())), LogEntry.create((long)2L, (Record)Record.create((byte)1, (long)2L, (byte[])"k2".getBytes(), (byte[])"goodbye".getBytes())));
        MemoryRecords records = MemoryRecords.withLogEntries((CompressionType)CompressionType.GZIP, entries);
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            fileRecords.append(records);
            fileRecords.flush();
            Records convertedRecords = fileRecords.toMessageFormat((byte)0);
            this.verifyConvertedMessageSet(entries, convertedRecords, (byte)0);
        }
    }

    private void verifyConvertedMessageSet(List<LogEntry> initialEntries, Records convertedRecords, byte magicByte) {
        int i = 0;
        for (LogEntry logEntry : FileRecordsTest.deepEntries(convertedRecords)) {
            Assert.assertEquals((String)("magic byte should be " + magicByte), (long)magicByte, (long)logEntry.record().magic());
            Assert.assertEquals((String)"offset should not change", (long)initialEntries.get(i).offset(), (long)logEntry.offset());
            Assert.assertEquals((String)"key should not change", (Object)initialEntries.get(i).record().key(), (Object)logEntry.record().key());
            Assert.assertEquals((String)"payload should not change", (Object)initialEntries.get(i).record().value(), (Object)logEntry.record().value());
            ++i;
        }
    }

    private static List<LogEntry> shallowEntries(Records buffer) {
        return TestUtils.toList(buffer.shallowEntries());
    }

    private static List<LogEntry> deepEntries(Records buffer) {
        return TestUtils.toList(buffer.deepEntries());
    }

    private FileRecords createFileRecords(Record ... records) throws IOException {
        FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());
        fileRecords.append(MemoryRecords.withRecords((Record[])records));
        fileRecords.flush();
        return fileRecords;
    }
}

