/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.compress;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Random;
import java.util.stream.Stream;
import net.jpountz.xxhash.XXHashFactory;
import org.apache.kafka.common.compress.KafkaLZ4BlockInputStream;
import org.apache.kafka.common.compress.KafkaLZ4BlockOutputStream;
import org.apache.kafka.common.utils.BufferSupplier;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.ArgumentsProvider;
import org.junit.jupiter.params.provider.ArgumentsSource;

public class KafkaLZ4Test {
    private static final Random RANDOM = new Random(0L);

    @ParameterizedTest
    @ArgumentsSource(value=Lz4ArgumentsProvider.class)
    public void testHeaderPrematureEnd(Args args) {
        ByteBuffer buffer = ByteBuffer.allocate(2);
        IOException e = (IOException)Assertions.assertThrows(IOException.class, () -> this.makeInputStream(buffer, args.ignoreFlagDescriptorChecksum));
        Assertions.assertEquals((Object)"Stream ended prematurely", (Object)e.getMessage());
    }

    private KafkaLZ4BlockInputStream makeInputStream(ByteBuffer buffer, boolean ignoreFlagDescriptorChecksum) throws IOException {
        return new KafkaLZ4BlockInputStream(buffer, BufferSupplier.create(), ignoreFlagDescriptorChecksum);
    }

    @ParameterizedTest
    @ArgumentsSource(value=Lz4ArgumentsProvider.class)
    public void testNotSupported(Args args) throws Exception {
        byte[] compressed = this.compressedBytes(args);
        compressed[0] = 0;
        ByteBuffer buffer = ByteBuffer.wrap(compressed);
        IOException e = (IOException)Assertions.assertThrows(IOException.class, () -> this.makeInputStream(buffer, args.ignoreFlagDescriptorChecksum));
        Assertions.assertEquals((Object)"Stream unsupported (invalid magic bytes)", (Object)e.getMessage());
    }

    @ParameterizedTest
    @ArgumentsSource(value=Lz4ArgumentsProvider.class)
    public void testBadFrameChecksum(Args args) throws Exception {
        byte[] compressed = this.compressedBytes(args);
        compressed[6] = -1;
        ByteBuffer buffer = ByteBuffer.wrap(compressed);
        if (args.ignoreFlagDescriptorChecksum) {
            this.makeInputStream(buffer, args.ignoreFlagDescriptorChecksum);
        } else {
            IOException e = (IOException)Assertions.assertThrows(IOException.class, () -> this.makeInputStream(buffer, args.ignoreFlagDescriptorChecksum));
            Assertions.assertEquals((Object)"Stream frame descriptor corrupted", (Object)e.getMessage());
        }
    }

    @ParameterizedTest
    @ArgumentsSource(value=Lz4ArgumentsProvider.class)
    public void testBadBlockSize(Args args) throws Exception {
        if (!args.close || args.useBrokenFlagDescriptorChecksum && !args.ignoreFlagDescriptorChecksum) {
            return;
        }
        byte[] compressed = this.compressedBytes(args);
        ByteBuffer buffer = ByteBuffer.wrap(compressed).order(ByteOrder.LITTLE_ENDIAN);
        int blockSize = buffer.getInt(7);
        blockSize = blockSize & Integer.MIN_VALUE | 0x1000000;
        buffer.putInt(7, blockSize);
        IOException e = (IOException)Assertions.assertThrows(IOException.class, () -> this.testDecompression(buffer, args));
        Assertions.assertTrue((boolean)e.getMessage().contains("exceeded max"));
    }

    @ParameterizedTest
    @ArgumentsSource(value=Lz4ArgumentsProvider.class)
    public void testCompression(Args args) throws Exception {
        boolean contentSize;
        byte[] compressed = this.compressedBytes(args);
        int offset = 0;
        Assertions.assertEquals((int)4, (int)compressed[offset++]);
        Assertions.assertEquals((int)34, (int)compressed[offset++]);
        Assertions.assertEquals((int)77, (int)compressed[offset++]);
        Assertions.assertEquals((int)24, (int)compressed[offset++]);
        byte flg = compressed[offset++];
        int version = flg >>> 6 & 3;
        Assertions.assertEquals((int)1, (int)version);
        int reserved = flg & 3;
        Assertions.assertEquals((int)0, (int)reserved);
        byte bd = compressed[offset++];
        int blockMaxSize = bd >>> 4 & 7;
        Assertions.assertTrue((blockMaxSize >= 4 ? 1 : 0) != 0);
        Assertions.assertTrue((blockMaxSize <= 7 ? 1 : 0) != 0);
        reserved = bd & 0xF;
        Assertions.assertEquals((int)0, (int)reserved);
        reserved = bd >>> 7 & 1;
        Assertions.assertEquals((int)0, (int)reserved);
        boolean bl = contentSize = (flg >>> 3 & 1) != 0;
        if (contentSize) {
            offset += 8;
        }
        int off = 4;
        int len = offset - 4;
        if (args.useBrokenFlagDescriptorChecksum) {
            off = 0;
            len = offset;
        }
        int hash = XXHashFactory.fastestInstance().hash32().hash(compressed, off, len, 0);
        byte hc = compressed[offset++];
        Assertions.assertEquals((byte)((byte)(hash >> 8 & 0xFF)), (byte)hc);
        if (args.close) {
            offset = compressed.length - 4;
            Assertions.assertEquals((int)0, (int)compressed[offset++]);
            Assertions.assertEquals((int)0, (int)compressed[offset++]);
            Assertions.assertEquals((int)0, (int)compressed[offset++]);
            Assertions.assertEquals((int)0, (int)compressed[offset++]);
        }
    }

    @ParameterizedTest
    @ArgumentsSource(value=Lz4ArgumentsProvider.class)
    public void testArrayBackedBuffer(Args args) throws IOException {
        byte[] compressed = this.compressedBytes(args);
        this.testDecompression(ByteBuffer.wrap(compressed), args);
    }

    @ParameterizedTest
    @ArgumentsSource(value=Lz4ArgumentsProvider.class)
    public void testArrayBackedBufferSlice(Args args) throws IOException {
        byte[] compressed = this.compressedBytes(args);
        int sliceOffset = 12;
        ByteBuffer buffer = ByteBuffer.allocate(compressed.length + sliceOffset + 123);
        buffer.position(sliceOffset);
        buffer.put(compressed).flip();
        buffer.position(sliceOffset);
        ByteBuffer slice = buffer.slice();
        this.testDecompression(slice, args);
        int offset = 42;
        buffer = ByteBuffer.allocate(compressed.length + sliceOffset + offset);
        buffer.position(sliceOffset + offset);
        buffer.put(compressed).flip();
        buffer.position(sliceOffset);
        slice = buffer.slice();
        slice.position(offset);
        this.testDecompression(slice, args);
    }

    @ParameterizedTest
    @ArgumentsSource(value=Lz4ArgumentsProvider.class)
    public void testDirectBuffer(Args args) throws IOException {
        byte[] compressed = this.compressedBytes(args);
        ByteBuffer buffer = ByteBuffer.allocateDirect(compressed.length);
        buffer.put(compressed).flip();
        this.testDecompression(buffer, args);
        int offset = 42;
        buffer = ByteBuffer.allocateDirect(compressed.length + offset + 123);
        buffer.position(offset);
        buffer.put(compressed).flip();
        buffer.position(offset);
        this.testDecompression(buffer, args);
    }

    @ParameterizedTest
    @ArgumentsSource(value=Lz4ArgumentsProvider.class)
    public void testSkip(Args args) throws Exception {
        if (!args.close || args.useBrokenFlagDescriptorChecksum && !args.ignoreFlagDescriptorChecksum) {
            return;
        }
        KafkaLZ4BlockInputStream in = this.makeInputStream(ByteBuffer.wrap(this.compressedBytes(args)), args.ignoreFlagDescriptorChecksum);
        int n = 100;
        int remaining = args.payload.length;
        long skipped = in.skip((long)n);
        Assertions.assertEquals((long)Math.min(n, remaining), (long)skipped);
        n = 10000;
        remaining = (int)((long)remaining - skipped);
        skipped = in.skip((long)n);
        Assertions.assertEquals((long)Math.min(n, remaining), (long)skipped);
    }

    private void testDecompression(ByteBuffer buffer, Args args) throws IOException {
        IOException error = null;
        try {
            int n;
            KafkaLZ4BlockInputStream decompressed = this.makeInputStream(buffer, args.ignoreFlagDescriptorChecksum);
            byte[] testPayload = new byte[args.payload.length];
            byte[] tmp = new byte[1024];
            int pos = 0;
            int i = 0;
            while ((n = decompressed.read(tmp, i, tmp.length - i)) != -1) {
                if ((i += n) != tmp.length) continue;
                System.arraycopy(tmp, 0, testPayload, pos, i);
                pos += i;
                i = 0;
            }
            System.arraycopy(tmp, 0, testPayload, pos, i);
            Assertions.assertEquals((int)-1, (int)decompressed.read(tmp, 0, tmp.length));
            Assertions.assertEquals((int)args.payload.length, (int)(pos += i));
            Assertions.assertArrayEquals((byte[])args.payload, (byte[])testPayload);
        }
        catch (IOException e) {
            if (!args.ignoreFlagDescriptorChecksum && args.useBrokenFlagDescriptorChecksum) {
                Assertions.assertEquals((Object)"Stream frame descriptor corrupted", (Object)e.getMessage());
                error = e;
            }
            if (!args.close) {
                Assertions.assertEquals((Object)"Stream ended prematurely", (Object)e.getMessage());
                error = e;
            }
            throw e;
        }
        if (!args.ignoreFlagDescriptorChecksum && args.useBrokenFlagDescriptorChecksum) {
            Assertions.assertNotNull((Object)error);
        }
        if (!args.close) {
            Assertions.assertNotNull((Object)error);
        }
    }

    private byte[] compressedBytes(Args args) throws IOException {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        KafkaLZ4BlockOutputStream lz4 = new KafkaLZ4BlockOutputStream((OutputStream)output, 4, args.blockChecksum, args.useBrokenFlagDescriptorChecksum);
        lz4.write(args.payload, 0, args.payload.length);
        if (args.close) {
            lz4.close();
        } else {
            lz4.flush();
        }
        return output.toByteArray();
    }

    private static class Lz4ArgumentsProvider
    implements ArgumentsProvider {
        private Lz4ArgumentsProvider() {
        }

        public Stream<? extends Arguments> provideArguments(ExtensionContext context) throws Exception {
            ArrayList<Payload> payloads = new ArrayList<Payload>();
            payloads.add(new Payload("empty", new byte[0]));
            payloads.add(new Payload("onebyte", new byte[]{1}));
            for (int size : Arrays.asList(1000, 65536, 98304)) {
                byte[] random = new byte[size];
                RANDOM.nextBytes(random);
                payloads.add(new Payload("random", random));
                byte[] ones = new byte[size];
                Arrays.fill(ones, (byte)1);
                payloads.add(new Payload("ones", ones));
            }
            ArrayList<Arguments> arguments = new ArrayList<Arguments>();
            for (Payload payload : payloads) {
                for (boolean broken : Arrays.asList(false, true)) {
                    for (boolean ignore : Arrays.asList(false, true)) {
                        for (boolean blockChecksum : Arrays.asList(false, true)) {
                            for (boolean close : Arrays.asList(false, true)) {
                                arguments.add(Arguments.of((Object[])new Object[]{new Args(broken, ignore, blockChecksum, close, payload)}));
                            }
                        }
                    }
                }
            }
            return arguments.stream();
        }
    }

    private static class Args {
        final boolean useBrokenFlagDescriptorChecksum;
        final boolean ignoreFlagDescriptorChecksum;
        final byte[] payload;
        final boolean close;
        final boolean blockChecksum;

        Args(boolean useBrokenFlagDescriptorChecksum, boolean ignoreFlagDescriptorChecksum, boolean blockChecksum, boolean close, Payload payload) {
            this.useBrokenFlagDescriptorChecksum = useBrokenFlagDescriptorChecksum;
            this.ignoreFlagDescriptorChecksum = ignoreFlagDescriptorChecksum;
            this.blockChecksum = blockChecksum;
            this.close = close;
            this.payload = payload.payload;
        }

        public String toString() {
            return "useBrokenFlagDescriptorChecksum=" + this.useBrokenFlagDescriptorChecksum + ", ignoreFlagDescriptorChecksum=" + this.ignoreFlagDescriptorChecksum + ", blockChecksum=" + this.blockChecksum + ", close=" + this.close + ", payload=" + Arrays.toString(this.payload);
        }
    }

    private static class Payload {
        String name;
        byte[] payload;

        Payload(String name, byte[] payload) {
            this.name = name;
            this.payload = payload;
        }

        public String toString() {
            return "Payload{size=" + this.payload.length + ", name='" + this.name + '\'' + '}';
        }
    }
}

