/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.ProducerFencedException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.serialization.Serializer;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.processor.StreamPartitioner;
import org.apache.kafka.streams.processor.internals.RecordCollector;
import org.slf4j.Logger;

public class RecordCollectorImpl
implements RecordCollector {
    private static final int MAX_SEND_ATTEMPTS = 3;
    private static final long SEND_RETRY_BACKOFF = 100L;
    private final Logger log;
    private final Producer<byte[], byte[]> producer;
    private final Map<TopicPartition, Long> offsets;
    private final String logPrefix;
    private volatile KafkaException sendException;

    public RecordCollectorImpl(Producer<byte[], byte[]> producer, String streamTaskId, LogContext logContext) {
        this.producer = producer;
        this.offsets = new HashMap<TopicPartition, Long>();
        this.logPrefix = String.format("task [%s] ", streamTaskId);
        this.log = logContext.logger(this.getClass());
    }

    @Override
    public <K, V> void send(String topic, K key, V value, Long timestamp, Serializer<K> keySerializer, Serializer<V> valueSerializer, StreamPartitioner<? super K, ? super V> partitioner) {
        Integer partition = null;
        if (partitioner != null) {
            List partitions = this.producer.partitionsFor(topic);
            if (partitions.size() > 0) {
                partition = partitioner.partition(key, value, partitions.size());
            } else {
                throw new StreamsException("Could not get partition information for topic '" + topic + "'." + " This can happen if the topic does not exist.");
            }
        }
        this.send(topic, key, value, partition, timestamp, keySerializer, valueSerializer);
    }

    @Override
    public <K, V> void send(final String topic, final K key, final V value, Integer partition, final Long timestamp, Serializer<K> keySerializer, Serializer<V> valueSerializer) {
        this.checkForException();
        byte[] keyBytes = keySerializer.serialize(topic, key);
        byte[] valBytes = valueSerializer.serialize(topic, value);
        ProducerRecord serializedRecord = new ProducerRecord(topic, partition, timestamp, (Object)keyBytes, (Object)valBytes);
        for (int attempt = 1; attempt <= 3; ++attempt) {
            try {
                this.producer.send(serializedRecord, new Callback(){

                    public void onCompletion(RecordMetadata metadata, Exception exception) {
                        if (exception == null) {
                            if (RecordCollectorImpl.this.sendException != null) {
                                return;
                            }
                            TopicPartition tp = new TopicPartition(metadata.topic(), metadata.partition());
                            RecordCollectorImpl.this.offsets.put(tp, metadata.offset());
                        } else if (RecordCollectorImpl.this.sendException == null) {
                            RecordCollectorImpl.this.log.error("Error sending record (key {} value {} timestamp {}) to topic {} due to {}; No more records will be sent and no more offsets will be recorded for this task.", new Object[]{key, value, timestamp, topic, exception});
                            if (exception instanceof ProducerFencedException) {
                                RecordCollectorImpl.this.sendException = (KafkaException)((Object)new ProducerFencedException(String.format("%sAbort sending since producer got fenced with a previous record (key %s value %s timestamp %d) to topic %s, error message: %s", RecordCollectorImpl.this.logPrefix, key, value, timestamp, topic, exception.getMessage())));
                            } else {
                                RecordCollectorImpl.this.sendException = new StreamsException(String.format("%sAbort sending since an error caught with a previous record (key %s value %s timestamp %d) to topic %s due to %s.", RecordCollectorImpl.this.logPrefix, key, value, timestamp, topic, exception), exception);
                            }
                        }
                    }
                });
                return;
            }
            catch (TimeoutException e) {
                if (attempt == 3) {
                    throw new StreamsException(String.format("%sFailed to send record to topic %s due to timeout after %d attempts", this.logPrefix, topic, attempt));
                }
                this.log.warn("Timeout exception caught when sending record to topic {}; retrying with {} attempt", (Object)topic, (Object)attempt);
                Utils.sleep((long)100L);
                continue;
            }
        }
    }

    private void checkForException() {
        if (this.sendException != null) {
            throw this.sendException;
        }
    }

    @Override
    public void flush() {
        this.log.debug("Flushing producer");
        this.producer.flush();
        this.checkForException();
    }

    @Override
    public void close() {
        this.log.debug("Closing producer");
        this.producer.close();
        this.checkForException();
    }

    @Override
    public Map<TopicPartition, Long> offsets() {
        return this.offsets;
    }

    Producer<byte[], byte[]> producer() {
        return this.producer;
    }
}

