/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.state.internals;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.metrics.MeasurableStat;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Avg;
import org.apache.kafka.common.metrics.stats.Max;
import org.apache.kafka.common.metrics.stats.Min;
import org.apache.kafka.common.utils.Bytes;
import org.apache.kafka.streams.KeyValue;
import org.apache.kafka.streams.processor.internals.metrics.StreamsMetricsImpl;
import org.apache.kafka.streams.state.internals.LRUCacheEntry;
import org.apache.kafka.streams.state.internals.ThreadCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class NamedCache {
    private static final Logger log = LoggerFactory.getLogger(NamedCache.class);
    private final String name;
    private final TreeMap<Bytes, LRUNode> cache = new TreeMap();
    private final Set<Bytes> dirtyKeys = new LinkedHashSet<Bytes>();
    private ThreadCache.DirtyEntryFlushListener listener;
    private LRUNode tail;
    private LRUNode head;
    private long currentSizeBytes;
    private final NamedCacheMetrics namedCacheMetrics;
    private long numReadHits = 0L;
    private long numReadMisses = 0L;
    private long numOverwrites = 0L;
    private long numFlushes = 0L;

    NamedCache(String name, StreamsMetricsImpl metrics) {
        this.name = name;
        this.namedCacheMetrics = new NamedCacheMetrics(metrics, name);
    }

    final synchronized String name() {
        return this.name;
    }

    synchronized long hits() {
        return this.numReadHits;
    }

    synchronized long misses() {
        return this.numReadMisses;
    }

    synchronized long overwrites() {
        return this.numOverwrites;
    }

    synchronized long flushes() {
        return this.numFlushes;
    }

    synchronized LRUCacheEntry get(Bytes key) {
        if (key == null) {
            return null;
        }
        LRUNode node = this.getInternal(key);
        if (node == null) {
            return null;
        }
        this.updateLRU(node);
        return node.entry;
    }

    synchronized void setListener(ThreadCache.DirtyEntryFlushListener listener) {
        this.listener = listener;
    }

    synchronized void flush() {
        this.flush(null);
    }

    private void flush(LRUNode evicted) {
        ++this.numFlushes;
        if (log.isTraceEnabled()) {
            log.trace("Named cache {} stats on flush: #hits={}, #misses={}, #overwrites={}, #flushes={}", new Object[]{this.name, this.hits(), this.misses(), this.overwrites(), this.flushes()});
        }
        if (this.listener == null) {
            throw new IllegalArgumentException("No listener for namespace " + this.name + " registered with cache");
        }
        if (this.dirtyKeys.isEmpty()) {
            return;
        }
        ArrayList<ThreadCache.DirtyEntry> entries = new ArrayList<ThreadCache.DirtyEntry>();
        ArrayList<Bytes> deleted = new ArrayList<Bytes>();
        if (evicted != null) {
            entries.add(new ThreadCache.DirtyEntry(evicted.key, evicted.entry.value(), evicted.entry));
            this.dirtyKeys.remove(evicted.key);
        }
        for (Bytes key : this.dirtyKeys) {
            LRUNode node = this.getInternal(key);
            if (node == null) {
                throw new IllegalStateException("Key = " + key + " found in dirty key set, but entry is null");
            }
            entries.add(new ThreadCache.DirtyEntry(key, node.entry.value(), node.entry));
            node.entry.markClean();
            if (node.entry.value() != null) continue;
            deleted.add(node.key);
        }
        this.dirtyKeys.clear();
        this.listener.apply(entries);
        for (Bytes key : deleted) {
            this.delete(key);
        }
    }

    synchronized void put(Bytes key, LRUCacheEntry value) {
        if (!value.isDirty() && this.dirtyKeys.contains(key)) {
            throw new IllegalStateException(String.format("Attempting to put a clean entry for key [%s] into NamedCache [%s] when it already contains a dirty entry for the same key", key, this.name));
        }
        LRUNode node = this.cache.get(key);
        if (node != null) {
            ++this.numOverwrites;
            this.currentSizeBytes -= node.size();
            node.update(value);
            this.updateLRU(node);
        } else {
            node = new LRUNode(key, value);
            this.putHead(node);
            this.cache.put(key, node);
        }
        if (value.isDirty()) {
            this.dirtyKeys.remove(key);
            this.dirtyKeys.add(key);
        }
        this.currentSizeBytes += node.size();
    }

    synchronized long sizeInBytes() {
        return this.currentSizeBytes;
    }

    private LRUNode getInternal(Bytes key) {
        LRUNode node = this.cache.get(key);
        if (node == null) {
            ++this.numReadMisses;
            return null;
        }
        ++this.numReadHits;
        this.namedCacheMetrics.hitRatioSensor.record((double)this.numReadHits / (double)(this.numReadHits + this.numReadMisses));
        return node;
    }

    private void updateLRU(LRUNode node) {
        this.remove(node);
        this.putHead(node);
    }

    private void remove(LRUNode node) {
        if (node.previous != null) {
            node.previous.next = node.next;
        } else {
            this.head = node.next;
        }
        if (node.next != null) {
            node.next.previous = node.previous;
        } else {
            this.tail = node.previous;
        }
    }

    private void putHead(LRUNode node) {
        node.next = this.head;
        node.previous = null;
        if (this.head != null) {
            this.head.previous = node;
        }
        this.head = node;
        if (this.tail == null) {
            this.tail = this.head;
        }
    }

    synchronized void evict() {
        if (this.tail == null) {
            return;
        }
        LRUNode eldest = this.tail;
        this.currentSizeBytes -= eldest.size();
        this.remove(eldest);
        this.cache.remove(eldest.key);
        if (eldest.entry.isDirty()) {
            this.flush(eldest);
        }
    }

    synchronized LRUCacheEntry putIfAbsent(Bytes key, LRUCacheEntry value) {
        LRUCacheEntry originalValue = this.get(key);
        if (originalValue == null) {
            this.put(key, value);
        }
        return originalValue;
    }

    synchronized void putAll(List<KeyValue<byte[], LRUCacheEntry>> entries) {
        for (KeyValue<byte[], LRUCacheEntry> entry : entries) {
            this.put(Bytes.wrap((byte[])((byte[])entry.key)), (LRUCacheEntry)entry.value);
        }
    }

    synchronized LRUCacheEntry delete(Bytes key) {
        LRUNode node = this.cache.remove(key);
        if (node == null) {
            return null;
        }
        this.remove(node);
        this.dirtyKeys.remove(key);
        this.currentSizeBytes -= node.size();
        return node.entry();
    }

    public long size() {
        return this.cache.size();
    }

    synchronized Iterator<Bytes> keyRange(Bytes from, Bytes to) {
        return this.keySetIterator(this.cache.navigableKeySet().subSet(from, true, to, true));
    }

    public boolean isEmpty() {
        return this.cache.isEmpty();
    }

    private Iterator<Bytes> keySetIterator(Set<Bytes> keySet) {
        return new TreeSet<Bytes>(keySet).iterator();
    }

    synchronized Iterator<Bytes> allKeys() {
        return this.keySetIterator(this.cache.navigableKeySet());
    }

    synchronized LRUCacheEntry first() {
        if (this.head == null) {
            return null;
        }
        return this.head.entry;
    }

    synchronized LRUCacheEntry last() {
        if (this.tail == null) {
            return null;
        }
        return this.tail.entry;
    }

    synchronized LRUNode head() {
        return this.head;
    }

    synchronized LRUNode tail() {
        return this.tail;
    }

    synchronized void close() {
        this.tail = null;
        this.head = null;
        this.listener = null;
        this.currentSizeBytes = 0L;
        this.dirtyKeys.clear();
        this.cache.clear();
        this.namedCacheMetrics.removeAllSensors();
    }

    private static class NamedCacheMetrics {
        private final StreamsMetricsImpl metrics;
        private final Sensor hitRatioSensor;
        private final String taskName;
        private final String cacheName;

        private NamedCacheMetrics(StreamsMetricsImpl metrics, String cacheName) {
            this.taskName = ThreadCache.taskIDfromCacheName(cacheName);
            this.cacheName = cacheName;
            this.metrics = metrics;
            String group = "stream-record-cache-metrics";
            Map<String, String> allMetricTags = metrics.tagMap("task-id", this.taskName, "record-cache-id", "all");
            Sensor taskLevelHitRatioSensor = metrics.taskLevelSensor(this.taskName, "hitRatio", Sensor.RecordingLevel.DEBUG, new Sensor[0]);
            taskLevelHitRatioSensor.add(new MetricName("hitRatio-avg", "stream-record-cache-metrics", "The average cache hit ratio.", allMetricTags), (MeasurableStat)new Avg());
            taskLevelHitRatioSensor.add(new MetricName("hitRatio-min", "stream-record-cache-metrics", "The minimum cache hit ratio.", allMetricTags), (MeasurableStat)new Min());
            taskLevelHitRatioSensor.add(new MetricName("hitRatio-max", "stream-record-cache-metrics", "The maximum cache hit ratio.", allMetricTags), (MeasurableStat)new Max());
            Map<String, String> metricTags = metrics.tagMap("task-id", this.taskName, "record-cache-id", ThreadCache.underlyingStoreNamefromCacheName(cacheName));
            this.hitRatioSensor = metrics.cacheLevelSensor(this.taskName, cacheName, "hitRatio", Sensor.RecordingLevel.DEBUG, taskLevelHitRatioSensor);
            this.hitRatioSensor.add(new MetricName("hitRatio-avg", "stream-record-cache-metrics", "The average cache hit ratio.", metricTags), (MeasurableStat)new Avg());
            this.hitRatioSensor.add(new MetricName("hitRatio-min", "stream-record-cache-metrics", "The minimum cache hit ratio.", metricTags), (MeasurableStat)new Min());
            this.hitRatioSensor.add(new MetricName("hitRatio-max", "stream-record-cache-metrics", "The maximum cache hit ratio.", metricTags), (MeasurableStat)new Max());
        }

        private void removeAllSensors() {
            this.metrics.removeAllCacheLevelSensors(this.taskName, this.cacheName);
        }
    }

    static class LRUNode {
        private final Bytes key;
        private LRUCacheEntry entry;
        private LRUNode previous;
        private LRUNode next;

        LRUNode(Bytes key, LRUCacheEntry entry) {
            this.key = key;
            this.entry = entry;
        }

        LRUCacheEntry entry() {
            return this.entry;
        }

        Bytes key() {
            return this.key;
        }

        long size() {
            return (long)(this.key.get().length + 8 + 8 + 8) + this.entry.size();
        }

        LRUNode next() {
            return this.next;
        }

        LRUNode previous() {
            return this.previous;
        }

        private void update(LRUCacheEntry entry) {
            this.entry = entry;
        }
    }
}

