/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.kafka.streams.kstream.internals;

import org.apache.kafka.clients.producer.internals.BuiltInPartitioner;
import org.apache.kafka.streams.kstream.Windowed;
import org.apache.kafka.streams.processor.StreamPartitioner;

public class WindowedStreamPartitioner<K, V> implements StreamPartitioner<Windowed<K>, V> {

    private final WindowedSerializer<K> serializer;

    public WindowedStreamPartitioner(final WindowedSerializer<K> serializer) {
        this.serializer = serializer;
    }

    /**
     * WindowedStreamPartitioner determines the partition number for a record with the given windowed key and value
     * and the current number of partitions. The partition number id determined by the original key of the windowed key
     * using the same logic as DefaultPartitioner so that the topic is partitioned by the original key.
     *
     * @param topic the topic name this record is sent to
     * @param windowedKey the key of the record
     * @param value the value of the record
     * @param numPartitions the total number of partitions
     * @return an integer between 0 and {@code numPartitions-1}, or {@code null} if the default partitioning logic should be used
     */
    @Override
    public Integer partition(final String topic, final Windowed<K> windowedKey, final V value, final int numPartitions) {
        // for windowed key, the key bytes should never be null
        final byte[] keyBytes = serializer.serializeBaseKey(topic, windowedKey);

        // stick with the same built-in partitioner util functions that producer used
        // to make sure its behavior is consistent with the producer
        return BuiltInPartitioner.partitionForKey(keyBytes, numPartitions);
    }
}
