/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.helper;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import io.kyligence.kap.guava20.shaded.common.io.ByteSource;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.file.FileSystems;
import java.nio.file.Paths;
import java.time.Clock;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.NavigableSet;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import javax.sql.DataSource;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.kylin.common.KylinConfig;
import org.apache.kylin.common.KylinConfigBase;
import org.apache.kylin.common.StorageURL;
import org.apache.kylin.common.exception.KylinException;
import org.apache.kylin.common.exception.code.ErrorCodeProducer;
import org.apache.kylin.common.exception.code.ErrorCodeTool;
import org.apache.kylin.common.metrics.MetricsCategory;
import org.apache.kylin.common.metrics.MetricsGroup;
import org.apache.kylin.common.metrics.MetricsName;
import org.apache.kylin.common.persistence.ImageDesc;
import org.apache.kylin.common.persistence.RawResource;
import org.apache.kylin.common.persistence.ResourceStore;
import org.apache.kylin.common.persistence.metadata.AuditLogStore;
import org.apache.kylin.common.persistence.metadata.JdbcDataSource;
import org.apache.kylin.common.persistence.metadata.MetadataStore;
import org.apache.kylin.common.persistence.metadata.jdbc.JdbcUtil;
import org.apache.kylin.common.persistence.transaction.UnitOfWork;
import org.apache.kylin.common.persistence.transaction.UnitOfWorkParams;
import org.apache.kylin.common.util.HadoopUtil;
import org.apache.kylin.common.util.JsonUtil;
import org.apache.kylin.common.util.MetadataChecker;
import org.apache.kylin.helper.HelperConstants;
import org.apache.kylin.metadata.project.ProjectInstance;
import org.apache.kylin.tool.HDFSMetadataTool;
import org.apache.kylin.tool.garbage.StorageCleaner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetadataToolHelper {
    public static final DateTimeFormatter DATE_TIME_FORMATTER = HelperConstants.DATE_TIME_FORMATTER;
    private static final String GLOBAL = "global";
    private static final String HDFS_METADATA_URL_FORMATTER = "kylin_metadata@hdfs,path=%s";
    private static final Logger logger = LoggerFactory.getLogger(MetadataToolHelper.class);

    public void rotateAuditLog() {
        ResourceStore resourceStore = ResourceStore.getKylinMetaStore((KylinConfig)KylinConfig.getInstanceFromEnv());
        AuditLogStore auditLogStore = resourceStore.getAuditLogStore();
        auditLogStore.rotate();
    }

    public void backup(KylinConfig kylinConfig) throws Exception {
        HDFSMetadataTool.cleanBeforeBackup(kylinConfig);
        new MetadataToolHelper().backup(kylinConfig, null, HadoopUtil.getBackupFolder((KylinConfig)kylinConfig), null, true, false);
    }

    public void backup(KylinConfig kylinConfig, String dir, String folder) throws Exception {
        HDFSMetadataTool.cleanBeforeBackup(kylinConfig);
        new MetadataToolHelper().backup(kylinConfig, null, dir, folder, true, false);
    }

    public void backup(KylinConfig kylinConfig, String project, String path, String folder, boolean compress, boolean excludeTableExd) throws Exception {
        boolean isGlobal = null == project;
        long startAt = System.currentTimeMillis();
        try {
            this.doBackup(kylinConfig, project, path, folder, compress, excludeTableExd);
        }
        catch (Exception be) {
            if (isGlobal) {
                MetricsGroup.hostTagCounterInc((MetricsName)MetricsName.METADATA_BACKUP_FAILED, (MetricsCategory)MetricsCategory.GLOBAL, (String)GLOBAL);
            } else {
                MetricsGroup.hostTagCounterInc((MetricsName)MetricsName.METADATA_BACKUP_FAILED, (MetricsCategory)MetricsCategory.PROJECT, (String)project);
            }
            throw be;
        }
        finally {
            if (isGlobal) {
                MetricsGroup.hostTagCounterInc((MetricsName)MetricsName.METADATA_BACKUP, (MetricsCategory)MetricsCategory.GLOBAL, (String)GLOBAL);
                MetricsGroup.hostTagCounterInc((MetricsName)MetricsName.METADATA_BACKUP_DURATION, (MetricsCategory)MetricsCategory.GLOBAL, (String)GLOBAL, (long)(System.currentTimeMillis() - startAt));
            } else {
                MetricsGroup.hostTagCounterInc((MetricsName)MetricsName.METADATA_BACKUP, (MetricsCategory)MetricsCategory.PROJECT, (String)project);
                MetricsGroup.hostTagCounterInc((MetricsName)MetricsName.METADATA_BACKUP_DURATION, (MetricsCategory)MetricsCategory.PROJECT, (String)project, (long)(System.currentTimeMillis() - startAt));
            }
        }
    }

    void doBackup(KylinConfig kylinConfig, String project, String path, String folder, boolean compress, boolean excludeTableExd) throws Exception {
        ResourceStore resourceStore = ResourceStore.getKylinMetaStore((KylinConfig)kylinConfig);
        boolean isUTEnv = kylinConfig.isUTEnv();
        if (StringUtils.isBlank((CharSequence)path)) {
            path = KylinConfigBase.getKylinHome() + File.separator + "meta_backups";
        }
        if (StringUtils.isEmpty((CharSequence)folder)) {
            folder = LocalDateTime.now(Clock.systemDefaultZone()).format(DATE_TIME_FORMATTER) + "_backup";
        }
        String backupPath = StringUtils.appendIfMissing((String)path, (CharSequence)"/", (CharSequence[])new CharSequence[0]) + folder;
        System.out.printf(Locale.ROOT, "The metadata backup path is %s.%n", backupPath);
        String backupMetadataUrl = this.getMetadataUrl(backupPath, compress, kylinConfig);
        KylinConfig backupConfig = KylinConfig.createKylinConfig((KylinConfig)kylinConfig);
        backupConfig.setMetadataUrl(backupMetadataUrl);
        this.abortIfAlreadyExists(backupPath);
        logger.info("The backup metadataUrl is {} and backup path is {}", (Object)backupMetadataUrl, (Object)backupPath);
        try (ResourceStore backupResourceStore = ResourceStore.getKylinMetaStore((KylinConfig)backupConfig);){
            MetadataStore backupMetadataStore = backupResourceStore.getMetadataStore();
            if (StringUtils.isBlank((CharSequence)project)) {
                logger.info("start to copy all projects from ResourceStore.");
                long finalOffset = this.getOffset(isUTEnv, resourceStore);
                backupResourceStore.putResourceWithoutCheck("/_image", ByteSource.wrap((byte[])JsonUtil.writeValueAsBytes((Object)new ImageDesc(Long.valueOf(finalOffset)))), System.currentTimeMillis(), -1L);
                NavigableSet projectFolders = resourceStore.listResources("/");
                if (projectFolders == null) {
                    return;
                }
                UnitOfWork.doInTransactionWithRetry(() -> {
                    this.backupProjects(projectFolders, resourceStore, backupResourceStore, excludeTableExd);
                    return null;
                }, (String)"_global");
                RawResource uuid = resourceStore.getResource("/UUID");
                if (uuid != null) {
                    backupResourceStore.putResourceWithoutCheck(uuid.getResPath(), uuid.getByteSource(), uuid.getTimestamp(), -1L);
                }
                logger.info("start to backup all projects");
            } else {
                logger.info("start to copy project {} from ResourceStore.", (Object)project);
                UnitOfWork.doInTransactionWithRetry((UnitOfWorkParams)UnitOfWorkParams.builder().readonly(true).unitName(project).processor(() -> {
                    this.copyResourceStore("/" + project, resourceStore, backupResourceStore, true, excludeTableExd);
                    RawResource uuid = resourceStore.getResource("/UUID");
                    backupResourceStore.putResourceWithoutCheck(uuid.getResPath(), uuid.getByteSource(), uuid.getTimestamp(), -1L);
                    return null;
                }).build());
                if (Thread.currentThread().isInterrupted()) {
                    throw new InterruptedException("metadata task is interrupt");
                }
                logger.info("start to backup project {}", (Object)project);
            }
            backupResourceStore.deleteResource("/_global/trash_record");
            backupMetadataStore.dump(backupResourceStore);
            logger.info("backup successfully at {}", (Object)backupPath);
        }
    }

    public String getMetadataUrl(String rootPath, boolean compressed, KylinConfig kylinConfig) {
        if (HadoopUtil.isHdfsCompatibleSchema((String)rootPath, (KylinConfig)kylinConfig)) {
            String url = String.format(Locale.ROOT, HDFS_METADATA_URL_FORMATTER, Path.getPathWithoutSchemeAndAuthority((Path)new Path(rootPath)).toString() + "/");
            return compressed ? url + ",zip=1" : url;
        }
        if (rootPath.startsWith("file://")) {
            rootPath = rootPath.replace("file://", "");
            return StringUtils.appendIfMissing((String)rootPath, (CharSequence)"/", (CharSequence[])new CharSequence[0]);
        }
        return StringUtils.appendIfMissing((String)rootPath, (CharSequence)"/", (CharSequence[])new CharSequence[0]);
    }

    private void backupProjects(NavigableSet<String> projectFolders, ResourceStore resourceStore, ResourceStore backupResourceStore, boolean excludeTableExd) throws InterruptedException {
        for (String projectPath : projectFolders) {
            if (projectPath.equals("/UUID") || projectPath.equals("/_image")) continue;
            this.copyResourceStore(projectPath, resourceStore, backupResourceStore, false, excludeTableExd);
            if (!Thread.currentThread().isInterrupted()) continue;
            throw new InterruptedException("metadata task is interrupt");
        }
    }

    private void copyResourceStore(String projectPath, ResourceStore srcResourceStore, ResourceStore destResourceStore, boolean isProjectLevel, boolean excludeTableExd) {
        if (excludeTableExd) {
            String tableExdPath = projectPath + "/table_exd";
            NavigableSet projectItems = srcResourceStore.listResources(projectPath);
            for (String item : projectItems) {
                if (item.equals(tableExdPath)) continue;
                srcResourceStore.copy(item, destResourceStore);
            }
        } else {
            srcResourceStore.copy(projectPath, destResourceStore);
        }
        if (isProjectLevel) {
            String projectName = Paths.get(projectPath, new String[0]).getFileName().toString();
            srcResourceStore.copy(ProjectInstance.concatResourcePath((String)projectName), destResourceStore);
        }
    }

    private long getOffset(boolean isUTEnv, ResourceStore resourceStore) {
        AuditLogStore auditLogStore = resourceStore.getAuditLogStore();
        if (isUTEnv) {
            return auditLogStore.getMaxId();
        }
        return auditLogStore.getLogOffset() == 0L ? resourceStore.getOffset() : auditLogStore.getLogOffset();
    }

    private void abortIfAlreadyExists(String path) throws IOException {
        URI uri = HadoopUtil.makeURI((String)path);
        if (!uri.isAbsolute()) {
            logger.info("no scheme specified for {}, try local file system file://", (Object)path);
            File localFile = new File(path);
            if (localFile.exists()) {
                logger.error("[UNEXPECTED_THINGS_HAPPENED] local file {} already exists ", (Object)path);
                throw new KylinException((ErrorCodeProducer)ErrorCodeTool.FILE_ALREADY_EXISTS, new Object[]{path});
            }
            return;
        }
        FileSystem fs = HadoopUtil.getWorkingFileSystem();
        if (fs.exists(new Path(path))) {
            logger.error("[UNEXPECTED_THINGS_HAPPENED] specified file {} already exists ", (Object)path);
            throw new KylinException((ErrorCodeProducer)ErrorCodeTool.FILE_ALREADY_EXISTS, new Object[]{path});
        }
    }

    public void restore(KylinConfig kylinConfig, String project, String path, boolean delete) throws Exception {
        logger.info("Restore metadata with delete : {}", (Object)delete);
        ResourceStore resourceStore = ResourceStore.getKylinMetaStore((KylinConfig)kylinConfig);
        String restoreMetadataUrl = this.getMetadataUrl(path, false, kylinConfig);
        KylinConfig restoreConfig = KylinConfig.createKylinConfig((KylinConfig)kylinConfig);
        restoreConfig.setMetadataUrl(restoreMetadataUrl);
        logger.info("The restore metadataUrl is {} and restore path is {} ", (Object)restoreMetadataUrl, (Object)path);
        ResourceStore restoreResourceStore = ResourceStore.getKylinMetaStore((KylinConfig)restoreConfig);
        MetadataStore restoreMetadataStore = restoreResourceStore.getMetadataStore();
        MetadataChecker metadataChecker = new MetadataChecker(restoreMetadataStore);
        MetadataChecker.VerifyResult verifyResult = metadataChecker.verify();
        Preconditions.checkState((boolean)verifyResult.isQualified(), (Object)(verifyResult.getResultMessage() + "\n the metadata dir is not qualified"));
        this.restore(resourceStore, restoreResourceStore, project, delete);
        this.backup(kylinConfig);
    }

    public void restore(ResourceStore currentResourceStore, ResourceStore restoreResourceStore, String project, boolean delete) {
        if (StringUtils.isBlank((CharSequence)project)) {
            logger.info("start to restore all projects");
            NavigableSet srcProjectFolders = restoreResourceStore.listResources("/");
            NavigableSet destProjectFolders = currentResourceStore.listResources("/");
            srcProjectFolders = srcProjectFolders == null ? Sets.newTreeSet() : srcProjectFolders;
            destProjectFolders = destProjectFolders == null ? Sets.newTreeSet() : destProjectFolders;
            Sets.SetView projectFolders = Sets.union((Set)srcProjectFolders, (Set)destProjectFolders);
            for (String projectPath : projectFolders) {
                if (projectPath.equals("/UUID") || projectPath.equals("/_image")) continue;
                String projectName = Paths.get(projectPath, new String[0]).getName(0).toString();
                NavigableSet destResources = currentResourceStore.listResourcesRecursively(projectPath);
                NavigableSet srcResources = restoreResourceStore.listResourcesRecursively(projectPath);
                UnitOfWork.doInTransactionWithRetry(() -> this.doRestore(currentResourceStore, restoreResourceStore, destResources, srcResources, delete), (String)projectName, (int)1);
            }
        } else {
            logger.info("start to restore project {}", (Object)project);
            NavigableSet destGlobalProjectResources = currentResourceStore.listResourcesRecursively("/_global/project");
            Set globalDestResources = null;
            if (Objects.nonNull(destGlobalProjectResources)) {
                globalDestResources = destGlobalProjectResources.stream().filter(x -> Paths.get(x, new String[0]).getFileName().toString().equals(String.format(Locale.ROOT, "%s.json", project))).collect(Collectors.toSet());
            }
            Set globalSrcResources = restoreResourceStore.listResourcesRecursively("/_global/project").stream().filter(x -> Paths.get(x, new String[0]).getFileName().toString().equals(String.format(Locale.ROOT, "%s.json", project))).collect(Collectors.toSet());
            Set finalGlobalDestResources = globalDestResources;
            UnitOfWork.doInTransactionWithRetry(() -> this.doRestore(currentResourceStore, restoreResourceStore, finalGlobalDestResources, globalSrcResources, delete), (String)"_global", (int)1);
            String projectPath = FileSystems.getDefault().getSeparator() + project;
            NavigableSet destResources = currentResourceStore.listResourcesRecursively(projectPath);
            NavigableSet srcResources = restoreResourceStore.listResourcesRecursively(projectPath);
            UnitOfWork.doInTransactionWithRetry(() -> this.doRestore(currentResourceStore, restoreResourceStore, destResources, srcResources, delete), (String)project, (int)1);
        }
        logger.info("restore successfully");
    }

    private int doRestore(ResourceStore currentResourceStore, ResourceStore restoreResourceStore, Set<String> destResources, Set<String> srcResources, boolean delete) throws IOException {
        ResourceStore threadViewRS = ResourceStore.getKylinMetaStore((KylinConfig)KylinConfig.getInstanceFromEnv());
        destResources = destResources == null ? Collections.emptySet() : destResources;
        srcResources = srcResources == null ? Collections.emptySet() : srcResources;
        logger.info("Start insert metadata resource...");
        Sets.SetView insertRes = Sets.difference(srcResources, destResources);
        for (Object res : insertRes) {
            RawResource metadataRaw = restoreResourceStore.getResource((String)res);
            threadViewRS.checkAndPutResource((String)res, metadataRaw.getByteSource(), -1L);
        }
        logger.info("Start update metadata resource...");
        Sets.SetView updateRes = Sets.intersection(destResources, srcResources);
        for (String res : updateRes) {
            RawResource raw = currentResourceStore.getResource(res);
            RawResource metadataRaw = restoreResourceStore.getResource(res);
            threadViewRS.checkAndPutResource(res, metadataRaw.getByteSource(), raw.getMvcc());
        }
        if (delete) {
            logger.info("Start delete metadata resource...");
            Sets.SetView deleteRes = Sets.difference(destResources, srcResources);
            for (String res : deleteRes) {
                threadViewRS.deleteResource(res);
            }
        }
        return 0;
    }

    public void cleanStorage(boolean storageCleanup, List<String> projects, double requestFSRate, int retryTimes) {
        try {
            StorageCleaner storageCleaner = new StorageCleaner(storageCleanup, projects, requestFSRate, retryTimes);
            System.out.println("Start to cleanup HDFS");
            storageCleaner.execute();
            System.out.println("cleanup HDFS finished");
        }
        catch (Exception e) {
            logger.error("cleanup HDFS failed", (Throwable)e);
            System.out.println("\u001b[31mcleanup HDFS failed. Detailed Message is at ${KYLIN_HOME}/logs/shell.stderr\u001b[0m");
        }
    }

    public DataSource getDataSource(KylinConfig kylinConfig) throws Exception {
        StorageURL url = kylinConfig.getMetadataUrl();
        Properties props = JdbcUtil.datasourceParameters((StorageURL)url);
        return JdbcDataSource.getDataSource((Properties)props);
    }

    public void fetch(KylinConfig kylinConfig, String path, String folder, String target, boolean excludeTableExd) throws Exception {
        ResourceStore resourceStore = ResourceStore.getKylinMetaStore((KylinConfig)kylinConfig);
        if (StringUtils.isBlank((CharSequence)path)) {
            path = KylinConfigBase.getKylinHome() + File.separator + "meta_fetch";
        }
        if (StringUtils.isEmpty((CharSequence)folder)) {
            folder = LocalDateTime.now(Clock.systemDefaultZone()).format(DATE_TIME_FORMATTER) + "_fetch";
        }
        if (target == null) {
            System.out.println("target file must be set with fetch mode");
        } else {
            String fetchPath = StringUtils.appendIfMissing((String)path, (CharSequence)"/", (CharSequence[])new CharSequence[0]) + folder;
            String fetchMetadataUrl = this.getMetadataUrl(fetchPath, false, kylinConfig);
            KylinConfig fetchConfig = KylinConfig.createKylinConfig((KylinConfig)kylinConfig);
            fetchConfig.setMetadataUrl(fetchMetadataUrl);
            this.abortIfAlreadyExists(fetchPath);
            logger.info("The fetch metadataUrl is {} and backup path is {}", (Object)fetchMetadataUrl, (Object)fetchPath);
            try (ResourceStore fetchResourceStore = ResourceStore.getKylinMetaStore((KylinConfig)fetchConfig);){
                MetadataStore fetchMetadataStore = fetchResourceStore.getMetadataStore();
                String targetPath = target.startsWith("/") ? target.substring(1) : target;
                logger.info("start to copy target file {} from ResourceStore.", (Object)target);
                UnitOfWork.doInTransactionWithRetry((UnitOfWorkParams)UnitOfWorkParams.builder().readonly(true).unitName(target).processor(() -> {
                    this.copyResourceStore("/" + targetPath, resourceStore, fetchResourceStore, true, excludeTableExd);
                    RawResource uuid = resourceStore.getResource("/UUID");
                    fetchResourceStore.putResourceWithoutCheck(uuid.getResPath(), uuid.getByteSource(), uuid.getTimestamp(), -1L);
                    return null;
                }).build());
                if (Thread.currentThread().isInterrupted()) {
                    throw new InterruptedException("metadata task is interrupt");
                }
                logger.info("start to fetch target file {}", (Object)target);
                fetchMetadataStore.dump(fetchResourceStore);
                logger.info("fetch successfully at {}", (Object)fetchPath);
            }
        }
    }

    public NavigableSet<String> list(KylinConfig kylinConfig, String target) throws Exception {
        ResourceStore resourceStore = ResourceStore.getKylinMetaStore((KylinConfig)kylinConfig);
        NavigableSet res = resourceStore.listResources(target);
        if (res == null) {
            System.out.printf("%s is not exist%n", target);
        } else {
            System.out.println("" + res);
        }
        return res;
    }
}

