/*
Copyright � 1999 CERN - European Organization for Nuclear Research.
Permission to use, copy, modify, distribute and sell this software and its documentation for any purpose 
is hereby granted without fee, provided that the above copyright notice appear in all copies and 
that both that copyright notice and this permission notice appear in supporting documentation. 
CERN makes no representations about the suitability of this software for any purpose. 
It is provided "as is" without expressed or implied warranty.
*/
package org.apache.mahout.math.jet.random;

import org.apache.mahout.common.RandomUtils;

import java.util.Random;

/** @deprecated until unit tests are in place.  Until this time, this class/interface is unsupported. */
@Deprecated
public class Logarithmic extends AbstractContinousDistribution {

  // The uniform random number generated shared by all <b>static</b> methods.
  private static final Logarithmic SHARED = new Logarithmic(0.5, RandomUtils.getRandom());

  private double myP;

  // cached vars for method nextDouble(a) (for performance only)
  private double t;
  private double h;
  private double aPrev = -1.0;

  /** Constructs a Logarithmic distribution. */
  public Logarithmic(double p, Random randomGenerator) {
    setRandomGenerator(randomGenerator);
    myP = p;
  }

  /** Returns a random number from the distribution. */
  @Override
  public double nextDouble() {
    return nextDouble(this.myP);
  }

  /** Returns a random number from the distribution; bypasses the internal state. */
  public double nextDouble(double a) {
/******************************************************************
 *                                                                *
 *      Logarithmic Distribution - Inversion/Transformation       *
 *                                                                *
 ******************************************************************
 *                                                                *
 * The algorithm combines Inversion and Transformation.           *
 * It is based on the following fact: A random variable X from    *
 * the Logarithmic distribution has the property that X for fixed *
 * Y=y is Geometric distributed with P(X=x|Y=y)=(1-y)*y^(x-1) (*) *
 * where Y has distribution function F(y)=ln(1-y)/ln(1-p).        *
 * So first random numbers y are generated by simple Inversion,   *
 * then k=(long int) (1+ln(u)/ln(y)) is a Geometric random number *
 * and because of (*) a Logarithmic one.                          *
 * To speed up the algorithm squeezes are used as well as the     *
 * fact, that many of the random numbers are 1 or 2 (depending on *
 * special circumstances).                                        *
 * On an IBM/PC 486 optimal performance is achieved, if for p<.97 *
 * simple inversion is used and otherwise the transformation.     *
 * On an IBM/PC 286 inversion should be restricted to p<.90.      *
 *                                                                *
 ******************************************************************
 *                                                                *
 * FUNCTION:    - lsk  samples a random number from the           *
 *                Logarithmic distribution with                   *
 *                parameter  0 < p < 1 .                          *
 * REFERENCE:   - A.W. Kemp (1981): Efficient generation of       *
 *                logarithmically distributed pseudo-random       *
 *                variables, Appl. Statist. 30, 249-253.          *
 * SUBPROGRAMS: - drand(seed) ... (0,1)-Uniform generator with    *
 *                unsigned long integer *seed.                    *
 *                                                                *
 ******************************************************************/

    if (a != aPrev) {                   // Set-up
      aPrev = a;
      if (a < 0.97) {
        t = -a / Math.log(1.0 - a);
      } else {
        h = Math.log(1.0 - a);
      }
    }

    double u = randomGenerator.nextDouble();
    int k;
    if (a < 0.97) {                        // Inversion/Chop-down
      k = 1;
      double p = t;
      while (u > p) {
        u -= p;
        k++;
        p *= a * (k - 1.0) / (double) k;
      }
      return k;
    }

    if (u > a) {
      return 1;
    }                 // Transformation
    u = randomGenerator.nextDouble();
    double v = u;
    double q = 1.0 - Math.exp(v * h);
    if (u <= q * q) {
      k = (int) (1 + Math.log(u) / Math.log(q));
      return k;
    }
    if (u > q) {
      return 1;
    }
    return 2;
  }

  /** Sets the distribution parameter. */
  public void setState(double p) {
    this.myP = p;
  }

  /** Returns a random number from the distribution. */
  public static double staticNextDouble(double p) {
    synchronized (SHARED) {
      return SHARED.nextDouble(p);
    }
  }

  /** Returns a String representation of the receiver. */
  public String toString() {
    return this.getClass().getName() + '(' + myP + ')';
  }

}
