package org.apache.maven.surefire.report;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import org.apache.maven.surefire.util.NestedRuntimeException;

/**
 * Surefire output consumer proxy that writes test output to a {@link java.io.File} for each test suite.
 * <p/>
 * This class is not threadsafe, but can be encapsulated with a SynchronizedOutputConsumer. It may still be
 * accessed from different threads (serially).
 *
 * @author <a href="mailto:carlos@apache.org">Carlos Sanchez</a>
 * @version $Id: ConsoleOutputFileReporter.java 1100808 2011-05-08 20:37:27Z krosenvold $
 * @since 2.1
 */
public class ConsoleOutputFileReporter
    implements Reporter
{

    private static final String LINE_SEPARATOR = System.getProperty( "line.separator" );

    private final File reportsDirectory;

    private final StringBuffer outputBuffer = new StringBuffer();


    public ConsoleOutputFileReporter( File reportsDirectory )
    {
        this.reportsDirectory = reportsDirectory;
    }

    public void testSetStarting( ReportEntry reportEntry )
    {
    }

    public void testSetCompleted( ReportEntry report )
        throws ReporterException
    {
        PrintWriter printWriter;

        if ( !reportsDirectory.exists() )
        {
            //noinspection ResultOfMethodCallIgnored
            reportsDirectory.mkdirs();
        }

        try
        {
            if ( outputBuffer.length() > 0 )
            {
                File file = new File( reportsDirectory, report.getName() + "-output.txt" );
                printWriter = new PrintWriter( new BufferedWriter( new FileWriter( file ) ) );
                printWriter.write( outputBuffer.toString() );
                printWriter.write( LINE_SEPARATOR );
                outputBuffer.setLength( 0 );
                printWriter.close();
            }
        }
        catch ( IOException e )
        {
            throw new NestedRuntimeException( e );
        }
    }

    public void testStarting( ReportEntry report )
    {
    }

    public void testSucceeded( ReportEntry report )
    {
    }

    public void testSkipped( ReportEntry report )
    {
    }

    public void testError( ReportEntry report, String stdOut, String stdErr )
    {
    }

    public void testFailed( ReportEntry report, String stdOut, String stdErr )
    {
    }

    public void writeMessage( String message )
    {
    }

    public void writeMessage( byte[] b, int off, int len )
    {
        String line = new String( b, off, len );
        outputBuffer.append( line );
    }

    public void reset()
    {
    }
}
