/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.nifi.annotation.behavior.DynamicProperty;
import org.apache.nifi.annotation.behavior.EventDriven;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.SideEffectFree;
import org.apache.nifi.annotation.behavior.SupportsBatching;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.DeprecationNotice;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.security.util.crypto.HashAlgorithm;
import org.apache.nifi.security.util.crypto.HashService;

@EventDriven
@SideEffectFree
@SupportsBatching
@Tags(value={"attributes", "hash", "md5", "sha", "keccak", "blake2", "cryptography"})
@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@CapabilityDescription(value="Calculates a hash value for each of the specified attributes using the given algorithm and writes it to an output attribute. Please refer to https://csrc.nist.gov/Projects/Hash-Functions/NIST-Policy-on-Hash-Functions for help to decide which algorithm to use. ")
@WritesAttribute(attribute="<Specified Attribute Name per Dynamic Property>", description="This Processor adds an attribute whose value is the result of hashing the specified attribute. The name of this attribute is specified by the value of the dynamic property.")
@DynamicProperty(name="A flowfile attribute key for attribute inspection", value="Attribute Name", description="The property name defines the attribute to look for and hash in the incoming flowfile. The property value defines the name to give the generated attribute. Attribute names must be unique.")
@DeprecationNotice(classNames={"org.apache.nifi.processors.attributes.UpdateAttribute"}, reason="UpdateAttribute can be configured using the hash Expression Language function to digest one or more attributes")
public class CryptographicHashAttribute
extends AbstractProcessor {
    private static final AllowableValue ALLOW_PARTIAL_ATTRIBUTES_VALUE = new AllowableValue(PartialAttributePolicy.ALLOW.name(), "Allow missing attributes", "Do not route to failure if there are attributes configured for hashing that are not present in the flowfile");
    private static final AllowableValue FAIL_PARTIAL_ATTRIBUTES_VALUE = new AllowableValue(PartialAttributePolicy.PROHIBIT.name(), "Fail if missing attributes", "Route to failure if there are attributes configured for hashing that are not present in the flowfile");
    static final PropertyDescriptor CHARACTER_SET = new PropertyDescriptor.Builder().name("character_set").displayName("Character Set").description("The Character Set used to decode the attribute being hashed -- this applies to the incoming data encoding, not the resulting hash encoding. ").required(true).allowableValues(HashService.buildCharacterSetAllowableValues()).addValidator(StandardValidators.CHARACTER_SET_VALIDATOR).defaultValue("UTF-8").build();
    static final PropertyDescriptor FAIL_WHEN_EMPTY = new PropertyDescriptor.Builder().name("fail_when_empty").displayName("Fail when no attributes present").description("Route to failure when none of the attributes that are configured for hashing are found. If set to false, then flow files that do not contain any of the attributes that are configured for hashing will just pass through to success.").allowableValues(new String[]{"true", "false"}).required(true).addValidator(StandardValidators.BOOLEAN_VALIDATOR).defaultValue("true").build();
    static final PropertyDescriptor HASH_ALGORITHM = new PropertyDescriptor.Builder().name("hash_algorithm").displayName("Hash Algorithm").description("The cryptographic hash algorithm to use. Note that not all of the algorithms available are recommended for use (some are provided for legacy use). There are many things to consider when picking an algorithm; it is recommended to use the most secure algorithm possible.").required(true).allowableValues(HashService.buildHashAlgorithmAllowableValues()).defaultValue(HashAlgorithm.SHA256.getName()).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    static final PropertyDescriptor PARTIAL_ATTR_ROUTE_POLICY = new PropertyDescriptor.Builder().name("missing_attr_policy").displayName("Missing attribute policy").description("Policy for how the processor handles attributes that are configured for hashing but are not found in the flowfile.").required(true).allowableValues(new AllowableValue[]{ALLOW_PARTIAL_ATTRIBUTES_VALUE, FAIL_PARTIAL_ATTRIBUTES_VALUE}).addValidator(StandardValidators.NON_BLANK_VALIDATOR).defaultValue(ALLOW_PARTIAL_ATTRIBUTES_VALUE.getValue()).build();
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("Used for flowfiles that have a hash value added").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("Used for flowfiles that are missing required attributes").build();
    private static final Set<Relationship> relationships;
    private static final List<PropertyDescriptor> properties;
    private final AtomicReference<Map<String, String>> attributeToGenerateNameMapRef = new AtomicReference(Collections.emptyMap());

    public Set<Relationship> getRelationships() {
        return relationships;
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return properties;
    }

    protected PropertyDescriptor getSupportedDynamicPropertyDescriptor(String propertyDescriptorName) {
        return new PropertyDescriptor.Builder().name(propertyDescriptorName).addValidator(StandardValidators.NON_BLANK_VALIDATOR).build();
    }

    public void onPropertyModified(PropertyDescriptor descriptor, String oldValue, String newValue) {
        if (descriptor.isRequired()) {
            return;
        }
        HashMap<String, String> attributeToGeneratedNameMap = new HashMap<String, String>(this.attributeToGenerateNameMapRef.get());
        if (newValue == null) {
            attributeToGeneratedNameMap.remove(descriptor.getName());
        } else {
            attributeToGeneratedNameMap.put(descriptor.getName(), newValue);
        }
        this.attributeToGenerateNameMapRef.set(Collections.unmodifiableMap(attributeToGeneratedNameMap));
    }

    public void onTrigger(ProcessContext context, ProcessSession session) {
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        Charset charset = Charset.forName(context.getProperty(CHARACTER_SET).getValue());
        Map<String, String> attributeToGeneratedNameMap = this.attributeToGenerateNameMapRef.get();
        ComponentLog logger = this.getLogger();
        SortedMap<String, String> relevantAttributes = CryptographicHashAttribute.getRelevantAttributes(flowFile, attributeToGeneratedNameMap);
        if (relevantAttributes.isEmpty() && context.getProperty(FAIL_WHEN_EMPTY).asBoolean().booleanValue()) {
            logger.info("Routing {} to 'failure' because of missing all attributes: {}", new Object[]{flowFile, CryptographicHashAttribute.getMissingKeysString(null, attributeToGeneratedNameMap.keySet())});
            session.transfer(flowFile, REL_FAILURE);
            return;
        }
        if (relevantAttributes.size() != attributeToGeneratedNameMap.size() && PartialAttributePolicy.valueOf(context.getProperty(PARTIAL_ATTR_ROUTE_POLICY).getValue()) == PartialAttributePolicy.PROHIBIT) {
            logger.info("Routing {} to 'failure' because of missing attributes: {}", new Object[]{flowFile, CryptographicHashAttribute.getMissingKeysString(relevantAttributes.keySet(), attributeToGeneratedNameMap.keySet())});
            session.transfer(flowFile, REL_FAILURE);
            return;
        }
        String algorithmName = context.getProperty(HASH_ALGORITHM).getValue();
        logger.debug("Using algorithm {}", new Object[]{algorithmName});
        HashAlgorithm algorithm = HashAlgorithm.fromName((String)algorithmName);
        for (Map.Entry<String, String> entry : relevantAttributes.entrySet()) {
            logger.debug("Generating {} hash of attribute '{}'", new Object[]{algorithmName, entry.getKey()});
            String value = this.hashValue(algorithm, entry.getValue(), charset);
            session.putAttribute(flowFile, attributeToGeneratedNameMap.get(entry.getKey()), value);
        }
        session.getProvenanceReporter().modifyAttributes(flowFile);
        session.transfer(flowFile, REL_SUCCESS);
    }

    private static SortedMap<String, String> getRelevantAttributes(FlowFile flowFile, Map<String, String> attributeToGeneratedNameMap) {
        TreeMap<String, String> attributeMap = new TreeMap<String, String>();
        for (Map.Entry<String, String> entry : attributeToGeneratedNameMap.entrySet()) {
            String attributeName = entry.getKey();
            String attributeValue = flowFile.getAttribute(attributeName);
            if (attributeValue == null) continue;
            attributeMap.put(attributeName, attributeValue);
        }
        return attributeMap;
    }

    private String hashValue(HashAlgorithm algorithm, String value, Charset charset) {
        if (value == null) {
            this.getLogger().warn("Tried to calculate {} hash of null value; returning empty string", new Object[]{algorithm.getName()});
            return "";
        }
        return HashService.hashValue((HashAlgorithm)algorithm, (String)value, (Charset)charset);
    }

    private static String getMissingKeysString(Set<String> foundKeys, Set<String> wantedKeys) {
        StringBuilder missingKeys = new StringBuilder();
        for (String wantedKey : wantedKeys) {
            if (foundKeys != null && foundKeys.contains(wantedKey)) continue;
            missingKeys.append(wantedKey).append(" ");
        }
        return missingKeys.toString();
    }

    static {
        HashSet<Relationship> _relationships = new HashSet<Relationship>();
        _relationships.add(REL_FAILURE);
        _relationships.add(REL_SUCCESS);
        relationships = Collections.unmodifiableSet(_relationships);
        ArrayList<PropertyDescriptor> _properties = new ArrayList<PropertyDescriptor>();
        _properties.add(CHARACTER_SET);
        _properties.add(FAIL_WHEN_EMPTY);
        _properties.add(HASH_ALGORITHM);
        _properties.add(PARTIAL_ATTR_ROUTE_POLICY);
        properties = Collections.unmodifiableList(_properties);
    }

    public static enum PartialAttributePolicy {
        ALLOW,
        PROHIBIT;

    }
}

