/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.util.file;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Random;
import org.apache.nifi.util.security.MessageDigestUtils;
import org.slf4j.Logger;

public class FileUtils {
    public static final long TRANSFER_CHUNK_SIZE_BYTES = 0x800000L;
    public static final long MILLIS_BETWEEN_ATTEMPTS = 50L;

    public static void closeQuietly(Closeable closeable) {
        if (null != closeable) {
            try {
                closeable.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public static void releaseQuietly(FileLock lock) {
        if (null != lock) {
            try {
                lock.release();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public static void ensureDirectoryExistAndCanAccess(File dir) throws IOException {
        boolean made;
        if (dir.exists() && !dir.isDirectory()) {
            throw new IOException(dir.getAbsolutePath() + " is not a directory");
        }
        if (!dir.exists() && !(made = dir.mkdirs())) {
            throw new IOException(dir.getAbsolutePath() + " could not be created");
        }
        if (!dir.canRead() || !dir.canWrite()) {
            throw new IOException(dir.getAbsolutePath() + " directory does not have read/write privilege");
        }
    }

    public static boolean deleteFile(File file, Logger logger) {
        return FileUtils.deleteFile(file, logger, 1);
    }

    public static boolean deleteFile(File file, Logger logger, int attempts) {
        boolean isGone;
        block6: {
            if (file == null) {
                return false;
            }
            isGone = false;
            try {
                if (file.exists()) {
                    int effectiveAttempts = Math.max(1, attempts);
                    for (int i = 0; i < effectiveAttempts && !isGone; ++i) {
                        boolean bl = isGone = file.delete() || !file.exists();
                        if (isGone || effectiveAttempts - i <= 1) continue;
                        FileUtils.sleepQuietly(50L);
                    }
                    if (!isGone && logger != null) {
                        logger.warn("File appears to exist but unable to delete file: " + file.getAbsolutePath());
                    }
                }
            }
            catch (Throwable t) {
                if (logger == null) break block6;
                logger.warn("Unable to delete file: '" + file.getAbsolutePath() + "' due to " + String.valueOf(t));
            }
        }
        return isGone;
    }

    public static void deleteFile(List<File> files, Logger logger) {
        FileUtils.deleteFile(files, logger, 1);
    }

    public static void deleteFile(List<File> files, Logger logger, int attempts) {
        if (null == files || files.isEmpty()) {
            return;
        }
        int effectiveAttempts = Math.max(1, attempts);
        for (File file : files) {
            try {
                boolean isGone = false;
                for (int i = 0; i < effectiveAttempts && !isGone; ++i) {
                    boolean bl = isGone = file.delete() || !file.exists();
                    if (isGone || effectiveAttempts - i <= 1) continue;
                    FileUtils.sleepQuietly(50L);
                }
                if (isGone || logger == null) continue;
                logger.warn("File appears to exist but unable to delete file: " + file.getAbsolutePath());
            }
            catch (Throwable t) {
                if (null == logger) continue;
                logger.warn("Unable to delete file given from path: '" + file.getPath() + "' due to " + String.valueOf(t));
            }
        }
    }

    public static void deleteFilesInDir(File directory, FilenameFilter filter, Logger logger) {
        FileUtils.deleteFilesInDir(directory, filter, logger, false);
    }

    public static void deleteFilesInDir(File directory, FilenameFilter filter, Logger logger, boolean recurse) {
        FileUtils.deleteFilesInDir(directory, filter, logger, recurse, false);
    }

    public static void deleteFilesInDir(File directory, FilenameFilter filter, Logger logger, boolean recurse, boolean deleteEmptyDirectories) {
        if (null != directory && directory.isDirectory()) {
            File[] ingestFiles;
            for (File ingestFile : ingestFiles = directory.listFiles()) {
                boolean process;
                boolean bl = process = filter == null ? true : filter.accept(directory, ingestFile.getName());
                if (ingestFile.isFile() && process) {
                    FileUtils.deleteFile(ingestFile, logger, 3);
                }
                if (!ingestFile.isDirectory() || !recurse) continue;
                FileUtils.deleteFilesInDir(ingestFile, filter, logger, recurse, deleteEmptyDirectories);
                if (!deleteEmptyDirectories || ingestFile.list().length != 0) continue;
                FileUtils.deleteFile(ingestFile, logger, 3);
            }
        }
    }

    public static void deleteFiles(Collection<File> files, boolean recurse) throws IOException {
        for (File file : files) {
            FileUtils.deleteFile(file, recurse);
        }
    }

    public static void deleteFile(File file, boolean recurse) throws IOException {
        if (file.isDirectory() && recurse) {
            FileUtils.deleteFiles(Arrays.asList(file.listFiles()), recurse);
        }
        if (!FileUtils.deleteFile(file, null, 5)) {
            Files.delete(file.toPath());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void shredFile(File file, int passes) throws IOException {
        Random generator = new Random();
        long fileLength = file.length();
        int byteArraySize = (int)Math.min(fileLength, 0x100000L);
        byte[] b = new byte[byteArraySize];
        long numOfRandomWrites = fileLength / (long)b.length + 1L;
        FileOutputStream fos = new FileOutputStream(file);
        try {
            FileChannel channel = fos.getChannel();
            for (int i = 0; i < passes; ++i) {
                generator.nextBytes(b);
                int j = 0;
                while ((long)j <= numOfRandomWrites) {
                    fos.write(b);
                    ++j;
                }
                fos.flush();
                channel.position(0L);
            }
            Arrays.fill(b, (byte)0);
            int j = 0;
            while ((long)j < numOfRandomWrites) {
                fos.write(b);
                ++j;
            }
            fos.flush();
            fos.close();
            if (!FileUtils.deleteFile(file, null, 5)) {
                throw new IOException("Failed to delete file after shredding");
            }
        }
        finally {
            FileUtils.closeQuietly(fos);
        }
    }

    public static long copy(InputStream in, OutputStream out) throws IOException {
        int len;
        byte[] buffer = new byte[65536];
        long copied = 0L;
        while ((len = in.read(buffer)) > 0) {
            out.write(buffer, 0, len);
            copied += (long)len;
        }
        return copied;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static long copyBytes(byte[] bytes, File destination, boolean lockOutputFile) throws FileNotFoundException, IOException {
        FileOutputStream fos = null;
        FileLock outLock = null;
        long fileSize = 0L;
        try {
            fos = new FileOutputStream(destination);
            FileChannel out = fos.getChannel();
            if (lockOutputFile && null == (outLock = out.tryLock(0L, Long.MAX_VALUE, false))) {
                throw new IOException("Unable to obtain exclusive file lock for: " + destination.getAbsolutePath());
            }
            fos.write(bytes);
            fos.flush();
            fileSize = bytes.length;
        }
        catch (Throwable throwable) {
            FileUtils.releaseQuietly(outLock);
            FileUtils.closeQuietly(fos);
            throw throwable;
        }
        FileUtils.releaseQuietly(outLock);
        FileUtils.closeQuietly(fos);
        return fileSize;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static long copyFile(File source, File destination, boolean lockInputFile, boolean lockOutputFile, boolean move, Logger logger) throws FileNotFoundException, IOException {
        FileInputStream fis = null;
        FileOutputStream fos = null;
        FileLock inLock = null;
        FileLock outLock = null;
        long fileSize = 0L;
        if (!source.canRead()) {
            throw new IOException("Must at least have read permission");
        }
        if (move && source.renameTo(destination)) {
            return destination.length();
        }
        try {
            fis = new FileInputStream(source);
            fos = new FileOutputStream(destination);
            FileChannel in = fis.getChannel();
            FileChannel out = fos.getChannel();
            if (lockInputFile && null == (inLock = in.tryLock(0L, Long.MAX_VALUE, true))) {
                throw new IOException("Unable to obtain shared file lock for: " + source.getAbsolutePath());
            }
            if (lockOutputFile && null == (outLock = out.tryLock(0L, Long.MAX_VALUE, false))) {
                throw new IOException("Unable to obtain exclusive file lock for: " + destination.getAbsolutePath());
            }
            long bytesWritten = 0L;
            while ((bytesWritten += out.transferFrom(in, bytesWritten, 0x800000L)) < (fileSize = in.size())) {
            }
            out.force(false);
            FileUtils.closeQuietly(fos);
            FileUtils.closeQuietly(fis);
            fos = null;
            fis = null;
            if (move && !FileUtils.deleteFile(source, null, 5)) {
                if (logger == null) {
                    FileUtils.deleteFile(destination, null, 5);
                    throw new IOException("Could not remove file " + source.getAbsolutePath());
                }
                logger.warn("Configured to delete source file when renaming/move not successful.  However, unable to delete file at: " + source.getAbsolutePath());
            }
        }
        catch (Throwable throwable) {
            FileUtils.releaseQuietly(inLock);
            FileUtils.releaseQuietly(outLock);
            FileUtils.closeQuietly(fos);
            FileUtils.closeQuietly(fis);
            throw throwable;
        }
        FileUtils.releaseQuietly(inLock);
        FileUtils.releaseQuietly(outLock);
        FileUtils.closeQuietly(fos);
        FileUtils.closeQuietly(fis);
        return fileSize;
    }

    public static long copyFile(File source, File destination, boolean lockInputFile, boolean lockOutputFile, Logger logger) throws FileNotFoundException, IOException {
        return FileUtils.copyFile(source, destination, lockInputFile, lockOutputFile, false, logger);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static long copyFile(File source, OutputStream stream, boolean closeOutputStream, boolean lockInputFile) throws FileNotFoundException, IOException {
        FileInputStream fis = null;
        FileLock inLock = null;
        long fileSize = 0L;
        try {
            fis = new FileInputStream(source);
            FileChannel in = fis.getChannel();
            if (lockInputFile && (inLock = in.tryLock(0L, Long.MAX_VALUE, true)) == null) {
                throw new IOException("Unable to obtain exclusive file lock for: " + source.getAbsolutePath());
            }
            byte[] buffer = new byte[262144];
            int bytesRead = -1;
            while ((bytesRead = fis.read(buffer)) != -1) {
                stream.write(buffer, 0, bytesRead);
            }
            in.force(false);
            stream.flush();
            fileSize = in.size();
        }
        catch (Throwable throwable) {
            FileUtils.releaseQuietly(inLock);
            FileUtils.closeQuietly(fis);
            if (closeOutputStream) {
                FileUtils.closeQuietly(stream);
            }
            throw throwable;
        }
        FileUtils.releaseQuietly(inLock);
        FileUtils.closeQuietly(fis);
        if (closeOutputStream) {
            FileUtils.closeQuietly(stream);
        }
        return fileSize;
    }

    public static long copyFile(InputStream stream, File destination, boolean closeInputStream, boolean lockOutputFile) throws FileNotFoundException, IOException {
        Path destPath = destination.toPath();
        long size = Files.copy(stream, destPath, new CopyOption[0]);
        if (closeInputStream) {
            stream.close();
        }
        return size;
    }

    public static void renameFile(File source, File destination, int maxAttempts) throws IOException {
        FileUtils.renameFile(source, destination, maxAttempts, false);
    }

    public static void renameFile(File source, File destination, int maxAttempts, boolean replace) throws IOException {
        int attempts = replace || maxAttempts < 1 ? Math.max(2, maxAttempts) : maxAttempts;
        boolean renamed = false;
        for (int i = 0; i < attempts && !(renamed = source.renameTo(destination)); ++i) {
            FileUtils.deleteFile(destination, null, 5);
        }
        if (!renamed) {
            throw new IOException("Attempted " + maxAttempts + " times but unable to rename from '" + source.getPath() + "' to '" + destination.getPath() + "'");
        }
    }

    public static void sleepQuietly(long millis) {
        try {
            Thread.sleep(millis);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public static void syncWithRestore(File primaryFile, File restoreFile, Logger logger) throws IOException {
        if (primaryFile.exists() && !restoreFile.exists()) {
            FileUtils.copyFile(primaryFile, restoreFile, false, false, logger);
        } else if (restoreFile.exists() && !primaryFile.exists()) {
            FileUtils.copyFile(restoreFile, primaryFile, false, false, logger);
        } else if (primaryFile.exists() && restoreFile.exists() && !FileUtils.isSame(primaryFile, restoreFile)) {
            throw new IllegalStateException(String.format("Primary file '%s' is different than restore file '%s'", primaryFile.getAbsoluteFile(), restoreFile.getAbsolutePath()));
        }
    }

    public static boolean isSame(File file1, File file2) throws IOException {
        return Arrays.equals(FileUtils.computeDigest(file1), FileUtils.computeDigest(file2));
    }

    public static byte[] computeDigest(File file) throws IOException {
        try (FileInputStream fis = new FileInputStream(file);){
            byte[] byArray = MessageDigestUtils.getDigest(fis);
            return byArray;
        }
    }

    public static long getContainerCapacity(Path path) {
        return path.toFile().getTotalSpace();
    }

    public static long getContainerUsableSpace(Path path) {
        return path.toFile().getUsableSpace();
    }
}

