/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue.impl;

import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.time.Duration;
import java.util.concurrent.ExecutionException;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.ozone.common.ChunkBuffer;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.helpers.ChunkInfo;
import org.apache.hadoop.ozone.container.common.helpers.ContainerMetrics;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.impl.ContainerLayoutVersion;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.DispatcherContext;
import org.apache.hadoop.ozone.container.common.utils.StorageVolumeUtil;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.VolumeSet;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainer;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.ChunkUtils;
import org.apache.hadoop.ozone.container.keyvalue.impl.KeyValueStreamDataChannel;
import org.apache.hadoop.ozone.container.keyvalue.interfaces.BlockManager;
import org.apache.hadoop.ozone.container.keyvalue.interfaces.ChunkManager;
import org.apache.ratis.statemachine.StateMachine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FilePerBlockStrategy
implements ChunkManager {
    private static final Logger LOG = LoggerFactory.getLogger(FilePerBlockStrategy.class);
    private final boolean doSyncWrite;
    private final OpenFiles files = new OpenFiles();
    private final int defaultReadBufferCapacity;
    private final int readMappedBufferThreshold;
    private final VolumeSet volumeSet;

    public FilePerBlockStrategy(boolean sync, BlockManager manager, VolumeSet volSet) {
        this.doSyncWrite = sync;
        this.defaultReadBufferCapacity = manager == null ? 0 : manager.getDefaultReadBufferCapacity();
        this.readMappedBufferThreshold = manager == null ? 0 : manager.getReadMappedBufferThreshold();
        this.volumeSet = volSet;
    }

    private static void checkLayoutVersion(Container container) {
        Preconditions.checkArgument((((ContainerData)container.getContainerData()).getLayoutVersion() == ContainerLayoutVersion.FILE_PER_BLOCK ? 1 : 0) != 0);
    }

    @Override
    public String streamInit(Container container, BlockID blockID) throws StorageContainerException {
        FilePerBlockStrategy.checkLayoutVersion(container);
        File chunkFile = this.getChunkFile(container, blockID, null);
        return chunkFile.getAbsolutePath();
    }

    @Override
    public StateMachine.DataChannel getStreamDataChannel(Container container, BlockID blockID, ContainerMetrics metrics) throws StorageContainerException {
        FilePerBlockStrategy.checkLayoutVersion(container);
        File chunkFile = this.getChunkFile(container, blockID, null);
        return new KeyValueStreamDataChannel(chunkFile, (ContainerData)container.getContainerData(), metrics);
    }

    @Override
    public void writeChunk(Container container, BlockID blockID, ChunkInfo info, ChunkBuffer data, DispatcherContext dispatcherContext) throws StorageContainerException {
        boolean overwrite;
        FilePerBlockStrategy.checkLayoutVersion(container);
        Preconditions.checkNotNull((Object)dispatcherContext);
        DispatcherContext.WriteChunkStage stage = dispatcherContext.getStage();
        if (info.getLen() <= 0L) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Skip writing empty chunk {} in stage {}", (Object)info, (Object)stage);
            }
            return;
        }
        if (stage == DispatcherContext.WriteChunkStage.COMMIT_DATA) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Ignore chunk {} in stage {}", (Object)info, (Object)stage);
            }
            return;
        }
        KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
        File chunkFile = this.getChunkFile(container, blockID, info);
        long len = info.getLen();
        long offset = info.getOffset();
        HddsVolume volume = containerData.getVolume();
        FileChannel channel = null;
        try {
            channel = this.files.getChannel(chunkFile, this.doSyncWrite);
            overwrite = ChunkUtils.validateChunkForOverwrite(channel, info);
        }
        catch (IOException e) {
            StorageVolumeUtil.onFailure(volume);
            throw e;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Writing chunk {} (overwrite: {}) in stage {} to file {}", new Object[]{info, overwrite, stage, chunkFile});
        }
        if (!overwrite) {
            ChunkUtils.validateChunkSize(channel, info, chunkFile.getName());
        }
        ChunkUtils.writeData(channel, chunkFile.getName(), data, offset, len, volume);
        containerData.updateWriteStats(len, overwrite);
    }

    @Override
    public ChunkBuffer readChunk(Container container, BlockID blockID, ChunkInfo info, DispatcherContext dispatcherContext) throws StorageContainerException {
        FilePerBlockStrategy.checkLayoutVersion(container);
        if (info.getLen() <= 0L) {
            LOG.debug("Skip reading empty chunk {}", (Object)info);
            return ChunkBuffer.wrap((ByteBuffer)ByteBuffer.wrap(new byte[0]));
        }
        ChunkUtils.limitReadSize(info.getLen());
        KeyValueContainerData containerData = (KeyValueContainerData)container.getContainerData();
        HddsVolume volume = containerData.getVolume();
        File chunkFile = this.getChunkFile(container, blockID, info);
        long len = info.getLen();
        long offset = info.getOffset();
        int bufferCapacity = ChunkManager.getBufferCapacityForChunkRead(info, this.defaultReadBufferCapacity);
        return ChunkUtils.readData(len, bufferCapacity, chunkFile, offset, volume, this.readMappedBufferThreshold);
    }

    @Override
    public void deleteChunk(Container container, BlockID blockID, ChunkInfo info) throws StorageContainerException {
        this.deleteChunk(container, blockID, info, true);
    }

    @Override
    public void deleteChunks(Container container, BlockData blockData) throws StorageContainerException {
        this.deleteChunk(container, blockData.getBlockID(), null, false);
    }

    @Override
    public void finishWriteChunks(KeyValueContainer container, BlockData blockData) throws IOException {
        File chunkFile = this.getChunkFile(container, blockData.getBlockID(), null);
        try {
            this.files.close(chunkFile);
            ChunkUtils.verifyChunkFileExists(chunkFile);
        }
        catch (IOException e) {
            StorageVolumeUtil.onFailure(container.getContainerData().getVolume());
            throw e;
        }
    }

    private void deleteChunk(Container container, BlockID blockID, ChunkInfo info, boolean verifyLength) throws StorageContainerException {
        FilePerBlockStrategy.checkLayoutVersion(container);
        Preconditions.checkNotNull((Object)blockID, (Object)"Block ID cannot be null.");
        File file = this.getChunkFile(container, blockID, info);
        if (!file.exists()) {
            LOG.warn("Block file to be deleted does not exist: {}", (Object)file);
            return;
        }
        if (verifyLength) {
            Preconditions.checkNotNull((Object)info, (Object)"Chunk info cannot be null for single chunk delete");
            FilePerBlockStrategy.checkFullDelete(info, file);
        }
        FileUtil.fullyDelete((File)file);
        LOG.info("Deleted block file: {}", (Object)file);
    }

    private File getChunkFile(Container container, BlockID blockID, ChunkInfo info) throws StorageContainerException {
        return ContainerLayoutVersion.FILE_PER_BLOCK.getChunkFile((ContainerData)container.getContainerData(), blockID, info);
    }

    private static void checkFullDelete(ChunkInfo info, File chunkFile) throws StorageContainerException {
        long fileLength = chunkFile.length();
        if (info.getOffset() > 0L || info.getLen() != fileLength) {
            String msg = String.format("Trying to delete partial chunk %s from file %s with length %s", info, chunkFile, fileLength);
            LOG.error(msg);
            throw new StorageContainerException(msg, ContainerProtos.Result.UNSUPPORTED_REQUEST);
        }
    }

    private static final class OpenFile {
        private final RandomAccessFile file;

        private OpenFile(File file, boolean sync) throws FileNotFoundException {
            String mode = sync ? "rws" : "rw";
            this.file = new RandomAccessFile(file, mode);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Opened file {}", (Object)file);
            }
        }

        public FileChannel getChannel() {
            return this.file.getChannel();
        }

        public void close() {
            try {
                this.file.close();
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    }

    private static final class OpenFiles {
        private static final RemovalListener<String, OpenFile> ON_REMOVE = event -> OpenFiles.close((String)event.getKey(), (OpenFile)event.getValue());
        private final Cache<String, OpenFile> files = CacheBuilder.newBuilder().expireAfterAccess(Duration.ofMinutes(10L)).removalListener(ON_REMOVE).build();

        private OpenFiles() {
        }

        public FileChannel getChannel(File file, boolean sync) throws StorageContainerException {
            try {
                return ((OpenFile)this.files.get((Object)file.getPath(), () -> OpenFiles.open(file, sync))).getChannel();
            }
            catch (ExecutionException e) {
                if (e.getCause() instanceof IOException) {
                    throw new UncheckedIOException((IOException)e.getCause());
                }
                throw new StorageContainerException(e.getCause(), ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
            }
        }

        private static OpenFile open(File file, boolean sync) {
            try {
                return new OpenFile(file, sync);
            }
            catch (FileNotFoundException e) {
                throw new UncheckedIOException(e);
            }
        }

        public void close(File file) {
            if (file != null) {
                this.files.invalidate((Object)file.getPath());
            }
        }

        private static void close(String filename, OpenFile openFile) {
            if (openFile != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Closing file {}", (Object)filename);
                }
                openFile.close();
            } else if (LOG.isDebugEnabled()) {
                LOG.debug("File {} not open", (Object)filename);
            }
        }
    }
}

