/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.utils.db;

import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.hdds.StringUtils;
import org.apache.hadoop.hdds.utils.HddsServerUtil;
import org.apache.hadoop.hdds.utils.db.StringCodec;
import org.apache.hadoop.hdds.utils.db.TableConfig;
import org.apache.hadoop.hdds.utils.db.managed.ManagedCheckpoint;
import org.apache.hadoop.hdds.utils.db.managed.ManagedColumnFamilyOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedCompactRangeOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedDBOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedFlushOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedIngestExternalFileOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedReadOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedRocksDB;
import org.apache.hadoop.hdds.utils.db.managed.ManagedRocksIterator;
import org.apache.hadoop.hdds.utils.db.managed.ManagedRocksObjectUtils;
import org.apache.hadoop.hdds.utils.db.managed.ManagedTransactionLogIterator;
import org.apache.hadoop.hdds.utils.db.managed.ManagedWriteBatch;
import org.apache.hadoop.hdds.utils.db.managed.ManagedWriteOptions;
import org.apache.ozone.rocksdiff.RocksDiffUtils;
import org.rocksdb.Checkpoint;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.ColumnFamilyOptions;
import org.rocksdb.CompactRangeOptions;
import org.rocksdb.DBOptions;
import org.rocksdb.FlushOptions;
import org.rocksdb.Holder;
import org.rocksdb.IngestExternalFileOptions;
import org.rocksdb.KeyMayExist;
import org.rocksdb.LiveFileMetaData;
import org.rocksdb.Options;
import org.rocksdb.ReadOptions;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.TransactionLogIterator;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class RocksDatabase
implements Closeable {
    static final Logger LOG = LoggerFactory.getLogger(RocksDatabase.class);
    public static final String ESTIMATE_NUM_KEYS = "rocksdb.estimate-num-keys";
    private static final ManagedReadOptions DEFAULT_READ_OPTION;
    private static Map<String, List<ColumnFamilyHandle>> dbNameToCfHandleMap;
    private final StackTraceElement[] stackTrace;
    private final String name;
    private final ManagedRocksDB db;
    private final ManagedDBOptions dbOptions;
    private final ManagedWriteOptions writeOptions;
    private final List<ColumnFamilyDescriptor> descriptors;
    private final Map<String, ColumnFamily> columnFamilies;
    private final AtomicBoolean isClosed = new AtomicBoolean();
    private final AtomicLong counter;

    static IOException toIOException(Object name, String op, RocksDBException e) {
        return HddsServerUtil.toIOException(name + ": Failed to " + op, e);
    }

    private static List<TableConfig> getExtraColumnFamilies(File file, Set<TableConfig> families) throws RocksDBException {
        Set existingFamilyNames = families.stream().map(TableConfig::getName).collect(Collectors.toSet());
        List<TableConfig> columnFamilies = RocksDatabase.listColumnFamiliesEmptyOptions(file.getAbsolutePath()).stream().map(TableConfig::toName).filter(familyName -> !existingFamilyNames.contains(familyName)).map(TableConfig::newTableConfig).collect(Collectors.toList());
        if (LOG.isDebugEnabled()) {
            LOG.debug("Found column families in DB {}: {}", (Object)file, columnFamilies);
        }
        return columnFamilies;
    }

    public static List<byte[]> listColumnFamiliesEmptyOptions(String path) throws RocksDBException {
        try (ManagedOptions emptyOptions = new ManagedOptions();){
            List list = RocksDB.listColumnFamilies((Options)emptyOptions, (String)path);
            return list;
        }
    }

    static RocksDatabase open(File dbFile, ManagedDBOptions dbOptions, ManagedWriteOptions writeOptions, Set<TableConfig> families, boolean readOnly) throws IOException {
        List<ColumnFamilyDescriptor> descriptors = null;
        ManagedRocksDB db = null;
        HashMap<String, ColumnFamily> columnFamilies = new HashMap<String, ColumnFamily>();
        try {
            List<TableConfig> extra = RocksDatabase.getExtraColumnFamilies(dbFile, families);
            descriptors = Stream.concat(families.stream(), extra.stream()).map(TableConfig::getDescriptor).collect(Collectors.toList());
            ArrayList handles = new ArrayList();
            db = readOnly ? ManagedRocksDB.openReadOnly((ManagedDBOptions)dbOptions, (String)dbFile.getAbsolutePath(), descriptors, handles) : ManagedRocksDB.open((DBOptions)dbOptions, (String)dbFile.getAbsolutePath(), descriptors, handles);
            dbNameToCfHandleMap.put(((RocksDB)db.get()).getName(), handles);
            AtomicLong counter = new AtomicLong(0L);
            for (ColumnFamilyHandle h : handles) {
                ColumnFamily f = new ColumnFamily(h, counter);
                columnFamilies.put(f.getName(), f);
            }
            return new RocksDatabase(dbFile, db, dbOptions, writeOptions, descriptors, Collections.unmodifiableMap(columnFamilies), counter);
        }
        catch (RocksDBException e) {
            RocksDatabase.close(columnFamilies, db, descriptors, writeOptions, dbOptions);
            throw RocksDatabase.toIOException(RocksDatabase.class, "open " + dbFile, e);
        }
    }

    private static void close(final ColumnFamilyDescriptor d) {
        ManagedColumnFamilyOptions options = (ManagedColumnFamilyOptions)d.getOptions();
        if (options.isReused()) {
            return;
        }
        RocksDatabase.runWithTryCatch(() -> ManagedColumnFamilyOptions.closeDeeply((ColumnFamilyOptions)options), new Object(){

            public String toString() {
                return d.getClass() + ":" + StringUtils.bytes2String((byte[])d.getName());
            }
        });
    }

    private static void close(Map<String, ColumnFamily> columnFamilies, ManagedRocksDB db, List<ColumnFamilyDescriptor> descriptors, ManagedWriteOptions writeOptions, ManagedDBOptions dbOptions) {
        if (columnFamilies != null) {
            for (ColumnFamily f : columnFamilies.values()) {
                RocksDatabase.runWithTryCatch(() -> f.getHandle().close(), f);
            }
        }
        if (db != null) {
            RocksDatabase.runWithTryCatch(() -> db.close(), "db");
        }
        if (descriptors != null) {
            descriptors.forEach(RocksDatabase::close);
        }
        if (writeOptions != null) {
            RocksDatabase.runWithTryCatch(() -> ((ManagedWriteOptions)writeOptions).close(), "writeOptions");
        }
        if (dbOptions != null) {
            RocksDatabase.runWithTryCatch(() -> ((ManagedDBOptions)dbOptions).close(), "dbOptions");
        }
    }

    private static void runWithTryCatch(Runnable runnable, Object name) {
        try {
            runnable.run();
        }
        catch (Throwable t) {
            LOG.error("Failed to close " + name, t);
        }
    }

    public boolean isClosed() {
        return this.isClosed.get();
    }

    private RocksDatabase(File dbFile, ManagedRocksDB db, ManagedDBOptions dbOptions, ManagedWriteOptions writeOptions, List<ColumnFamilyDescriptor> descriptors, Map<String, ColumnFamily> columnFamilies, AtomicLong counter) {
        this.name = this.getClass().getSimpleName() + "[" + dbFile + "]";
        this.db = db;
        this.dbOptions = dbOptions;
        this.writeOptions = writeOptions;
        this.descriptors = descriptors;
        this.columnFamilies = columnFamilies;
        this.counter = counter;
        this.stackTrace = Thread.currentThread().getStackTrace();
    }

    @Override
    public void close() {
        if (this.isClosed.compareAndSet(false, true)) {
            ((RocksDB)this.db.get()).cancelAllBackgroundWork(true);
            this.dbOptions.listeners().forEach(listener -> listener.close());
            if (this.columnFamilies != null) {
                this.columnFamilies.values().stream().forEach(f -> f.markClosed());
            }
            while (this.counter.get() != 0L) {
                try {
                    Thread.currentThread();
                    Thread.sleep(1L);
                }
                catch (InterruptedException e) {
                    RocksDatabase.close(this.columnFamilies, this.db, this.descriptors, this.writeOptions, this.dbOptions);
                    Thread.currentThread().interrupt();
                    return;
                }
            }
            RocksDatabase.close(this.columnFamilies, this.db, this.descriptors, this.writeOptions, this.dbOptions);
        }
    }

    private void closeOnError(RocksDBException e, boolean isCounted) {
        if (this.shouldClose(e)) {
            try {
                if (isCounted) {
                    this.counter.decrementAndGet();
                }
                this.close();
            }
            finally {
                if (isCounted) {
                    this.counter.incrementAndGet();
                }
            }
        }
    }

    private boolean shouldClose(RocksDBException e) {
        switch (e.getStatus().getCode()) {
            case Corruption: 
            case IOError: {
                return true;
            }
        }
        return false;
    }

    private void assertClose() throws IOException {
        if (this.isClosed()) {
            throw new IOException("Rocks Database is closed");
        }
    }

    public void ingestExternalFile(ColumnFamily family, List<String> files, ManagedIngestExternalFileOptions ingestOptions) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ((RocksDB)this.db.get()).ingestExternalFile(family.getHandle(), files, (IngestExternalFileOptions)ingestOptions);
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            String msg = "Failed to ingest external files " + files.stream().collect(Collectors.joining(", ")) + " of " + family.getName();
            throw RocksDatabase.toIOException(this, msg, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void put(ColumnFamily family, byte[] key, byte[] value) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ((RocksDB)this.db.get()).put(family.getHandle(), (WriteOptions)this.writeOptions, key, value);
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "put " + StringUtils.bytes2String((byte[])key), e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void put(ColumnFamily family, ByteBuffer key, ByteBuffer value) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ((RocksDB)this.db.get()).put(family.getHandle(), (WriteOptions)this.writeOptions, key, value);
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "put " + StringUtils.bytes2String((ByteBuffer)key), e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void flush() throws IOException {
        this.assertClose();
        try (ManagedFlushOptions options = new ManagedFlushOptions();){
            this.counter.incrementAndGet();
            options.setWaitForFlush(true);
            ((RocksDB)this.db.get()).flush((FlushOptions)options);
            for (ColumnFamily columnFamily : this.getExtraColumnFamilies()) {
                ((RocksDB)this.db.get()).flush((FlushOptions)options, columnFamily.handle);
            }
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "flush", e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void flush(String cfName) throws IOException {
        this.assertClose();
        ColumnFamilyHandle handle = this.getColumnFamilyHandle(cfName);
        try (ManagedFlushOptions options = new ManagedFlushOptions();){
            options.setWaitForFlush(true);
            if (handle != null) {
                ((RocksDB)this.db.get()).flush((FlushOptions)options, handle);
            } else {
                LOG.error("Provided column family doesn't exist. Calling flush on null columnFamily");
                this.flush();
            }
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "flush", e);
        }
    }

    public void flushWal(boolean sync) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ((RocksDB)this.db.get()).flushWal(sync);
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "flushWal with sync=" + sync, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void compactRange() throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ((RocksDB)this.db.get()).compactRange();
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "compactRange", e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void compactRangeDefault(ManagedCompactRangeOptions options) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ((RocksDB)this.db.get()).compactRange(null, null, null, (CompactRangeOptions)options);
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "compactRange", e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void compactDB(ManagedCompactRangeOptions options) throws IOException {
        this.assertClose();
        this.compactRangeDefault(options);
        for (ColumnFamily columnFamily : this.getExtraColumnFamilies()) {
            this.compactRange(columnFamily, null, null, options);
        }
    }

    public int getLiveFilesMetaDataSize() throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            int n = ((RocksDB)this.db.get()).getLiveFilesMetaData().size();
            return n;
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void compactRange(String cfName) throws IOException {
        this.assertClose();
        ColumnFamilyHandle handle = this.getColumnFamilyHandle(cfName);
        try {
            if (handle != null) {
                ((RocksDB)this.db.get()).compactRange(handle);
            } else {
                LOG.error("Provided column family doesn't exist. Calling compactRange on null columnFamily");
                ((RocksDB)this.db.get()).compactRange();
            }
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "compactRange", e);
        }
    }

    private ColumnFamilyHandle getColumnFamilyHandle(String cfName) throws IOException {
        this.assertClose();
        for (ColumnFamilyHandle cf : RocksDatabase.getCfHandleMap().get(((RocksDB)this.db.get()).getName())) {
            try {
                String table = new String(cf.getName(), StandardCharsets.UTF_8);
                if (!cfName.equals(table)) continue;
                return cf;
            }
            catch (RocksDBException e) {
                this.closeOnError(e, true);
                throw RocksDatabase.toIOException(this, "columnFamilyHandle.getName", e);
            }
        }
        return null;
    }

    public void compactRange(ColumnFamily family, byte[] begin, byte[] end, ManagedCompactRangeOptions options) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ((RocksDB)this.db.get()).compactRange(family.getHandle(), begin, end, (CompactRangeOptions)options);
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "compactRange", e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public List<LiveFileMetaData> getLiveFilesMetaData() throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            List list = ((RocksDB)this.db.get()).getLiveFilesMetaData();
            return list;
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    RocksCheckpoint createCheckpoint() {
        return new RocksCheckpoint();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Supplier<byte[]> keyMayExist(ColumnFamily family, byte[] key) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            Holder out = new Holder();
            Supplier<byte[]> supplier = ((RocksDB)this.db.get()).keyMayExist(family.getHandle(), key, out) ? () -> ((Holder)out).getValue() : null;
            return supplier;
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    Supplier<Integer> keyMayExist(ColumnFamily family, ByteBuffer key, ByteBuffer out) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            KeyMayExist result = ((RocksDB)this.db.get()).keyMayExist(family.getHandle(), key, out);
            switch (result.exists) {
                case kNotExist: {
                    Supplier<Integer> supplier = null;
                    return supplier;
                }
                case kExistsWithValue: {
                    Supplier<Integer> supplier = () -> result.valueLength;
                    return supplier;
                }
                case kExistsWithoutValue: {
                    Supplier<Integer> supplier = () -> null;
                    return supplier;
                }
            }
            throw new IllegalStateException("Unexpected KeyMayExistEnum case " + result.exists);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public ColumnFamily getColumnFamily(String key) {
        return this.columnFamilies.get(key);
    }

    public Collection<ColumnFamily> getExtraColumnFamilies() {
        return Collections.unmodifiableCollection(this.columnFamilies.values());
    }

    byte[] get(ColumnFamily family, byte[] key) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            byte[] byArray = ((RocksDB)this.db.get()).get(family.getHandle(), key);
            return byArray;
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            String message = "get " + StringUtils.bytes2String((byte[])key) + " from " + family;
            throw RocksDatabase.toIOException(this, message, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    Integer get(ColumnFamily family, ByteBuffer key, ByteBuffer outValue) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            int size = ((RocksDB)this.db.get()).get(family.getHandle(), (ReadOptions)DEFAULT_READ_OPTION, key, outValue);
            LOG.debug("get: size={}, remaining={}", (Object)size, (Object)outValue.asReadOnlyBuffer().remaining());
            Integer n = size == -1 ? null : Integer.valueOf(size);
            return n;
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            String message = "get " + StringUtils.bytes2String((ByteBuffer)key) + " from " + family;
            throw RocksDatabase.toIOException(this, message, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public long estimateNumKeys() throws IOException {
        return this.getLongProperty(ESTIMATE_NUM_KEYS);
    }

    public long estimateNumKeys(ColumnFamily family) throws IOException {
        return this.getLongProperty(family, ESTIMATE_NUM_KEYS);
    }

    private long getLongProperty(String key) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            long l = ((RocksDB)this.db.get()).getLongProperty(key);
            return l;
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "getLongProperty " + key, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    private long getLongProperty(ColumnFamily family, String key) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            long l = ((RocksDB)this.db.get()).getLongProperty(family.getHandle(), key);
            return l;
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            String message = "getLongProperty " + key + " from " + family;
            throw RocksDatabase.toIOException(this, message, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public String getProperty(String key) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            String string = ((RocksDB)this.db.get()).getProperty(key);
            return string;
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "getProperty " + key, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public String getProperty(ColumnFamily family, String key) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            String string = ((RocksDB)this.db.get()).getProperty(family.getHandle(), key);
            return string;
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "getProperty " + key + " from " + family, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public ManagedTransactionLogIterator getUpdatesSince(long sequenceNumber) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ManagedTransactionLogIterator managedTransactionLogIterator = ManagedTransactionLogIterator.managed((TransactionLogIterator)((RocksDB)this.db.get()).getUpdatesSince(sequenceNumber));
            return managedTransactionLogIterator;
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "getUpdatesSince " + sequenceNumber, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public long getLatestSequenceNumber() throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            long l = ((RocksDB)this.db.get()).getLatestSequenceNumber();
            return l;
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public ManagedRocksIterator newIterator(ColumnFamily family) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ManagedRocksIterator managedRocksIterator = ManagedRocksIterator.managed((RocksIterator)((RocksDB)this.db.get()).newIterator(family.getHandle()));
            return managedRocksIterator;
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    /*
     * Loose catch block
     */
    public ManagedRocksIterator newIterator(ColumnFamily family, boolean fillCache) throws IOException {
        this.assertClose();
        try {
            try (ManagedReadOptions readOptions = new ManagedReadOptions();){
                this.counter.incrementAndGet();
                readOptions.setFillCache(fillCache);
                ManagedRocksIterator managedRocksIterator = ManagedRocksIterator.managed((RocksIterator)((RocksDB)this.db.get()).newIterator(family.getHandle(), (ReadOptions)readOptions));
                return managedRocksIterator;
            }
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void batchWrite(ManagedWriteBatch writeBatch, ManagedWriteOptions options) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ((RocksDB)this.db.get()).write((WriteOptions)options, (WriteBatch)writeBatch);
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            throw RocksDatabase.toIOException(this, "batchWrite", e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void batchWrite(ManagedWriteBatch writeBatch) throws IOException {
        this.batchWrite(writeBatch, this.writeOptions);
    }

    public void delete(ColumnFamily family, byte[] key) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ((RocksDB)this.db.get()).delete(family.getHandle(), key);
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            String message = "delete " + StringUtils.bytes2String((byte[])key) + " from " + family;
            throw RocksDatabase.toIOException(this, message, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void delete(ColumnFamily family, ByteBuffer key) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ((RocksDB)this.db.get()).delete(family.getHandle(), (WriteOptions)this.writeOptions, key);
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            String message = "delete " + StringUtils.bytes2String((ByteBuffer)key) + " from " + family;
            throw RocksDatabase.toIOException(this, message, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public void deleteRange(ColumnFamily family, byte[] beginKey, byte[] endKey) throws IOException {
        this.assertClose();
        try {
            this.counter.incrementAndGet();
            ((RocksDB)this.db.get()).deleteRange(family.getHandle(), beginKey, endKey);
        }
        catch (RocksDBException e) {
            this.closeOnError(e, true);
            String message = "delete range " + StringUtils.bytes2String((byte[])beginKey) + " to " + StringUtils.bytes2String((byte[])endKey) + " from " + family;
            throw RocksDatabase.toIOException(this, message, e);
        }
        finally {
            this.counter.decrementAndGet();
        }
    }

    public String toString() {
        return this.name;
    }

    @VisibleForTesting
    public List<LiveFileMetaData> getSstFileList() throws IOException {
        this.assertClose();
        return ((RocksDB)this.db.get()).getLiveFilesMetaData();
    }

    private int getLastLevel() throws IOException {
        return this.getSstFileList().stream().max(Comparator.comparing(LiveFileMetaData::level)).get().level();
    }

    public void deleteFilesNotMatchingPrefix(Map<String, String> prefixPairs) throws IOException, RocksDBException {
        this.assertClose();
        for (LiveFileMetaData liveFileMetaData : this.getSstFileList()) {
            String lastDbKey;
            String firstDbKey;
            String prefixForColumnFamily;
            boolean isKeyWithPrefixPresent;
            String sstFileColumnFamily = new String(liveFileMetaData.columnFamilyName(), StandardCharsets.UTF_8);
            int lastLevel = this.getLastLevel();
            if (!prefixPairs.containsKey(sstFileColumnFamily) || liveFileMetaData.level() != lastLevel || lastLevel == 0 || (isKeyWithPrefixPresent = RocksDiffUtils.isKeyWithPrefixPresent((String)(prefixForColumnFamily = prefixPairs.get(sstFileColumnFamily)), (String)(firstDbKey = new String(liveFileMetaData.smallestKey(), StandardCharsets.UTF_8)), (String)(lastDbKey = new String(liveFileMetaData.largestKey(), StandardCharsets.UTF_8))))) continue;
            LOG.info("Deleting sst file: {} with start key: {} and end key: {} corresponding to column family {} from db: {}. Prefix for the column family: {}.", new Object[]{liveFileMetaData.fileName(), firstDbKey, lastDbKey, StringUtils.bytes2String((byte[])liveFileMetaData.columnFamilyName()), ((RocksDB)this.db.get()).getName(), prefixForColumnFamily});
            this.db.deleteFile(liveFileMetaData);
        }
    }

    public static Map<String, List<ColumnFamilyHandle>> getCfHandleMap() {
        return dbNameToCfHandleMap;
    }

    protected void finalize() throws Throwable {
        if (!this.isClosed()) {
            String warning = "RocksDatabase is not closed properly.";
            if (LOG.isDebugEnabled()) {
                String debugMessage = String.format("%n StackTrace for unclosed RocksDatabase instance: %s", Arrays.toString(this.stackTrace));
                warning = warning.concat(debugMessage);
            }
            LOG.warn(warning);
        }
        super.finalize();
    }

    public ManagedRocksDB getManagedRocksDb() {
        return this.db;
    }

    static {
        ManagedRocksObjectUtils.loadRocksDBLibrary();
        DEFAULT_READ_OPTION = new ManagedReadOptions();
        dbNameToCfHandleMap = new HashMap<String, List<ColumnFamilyHandle>>();
    }

    public static final class ColumnFamily {
        private final byte[] nameBytes;
        private AtomicLong counter;
        private final String name;
        private final ColumnFamilyHandle handle;
        private AtomicBoolean isClosed = new AtomicBoolean(false);

        public ColumnFamily(ColumnFamilyHandle handle, AtomicLong counter) throws RocksDBException {
            this.nameBytes = handle.getName();
            this.counter = counter;
            this.name = StringUtils.bytes2String((byte[])this.nameBytes);
            this.handle = handle;
            LOG.debug("new ColumnFamily for {}", (Object)this.name);
        }

        public String getName() {
            return this.name;
        }

        public String getName(StringCodec codec) {
            return codec.fromPersistedFormat(this.nameBytes);
        }

        @VisibleForTesting
        public ColumnFamilyHandle getHandle() {
            return this.handle;
        }

        public int getID() {
            return this.getHandle().getID();
        }

        public void batchDelete(ManagedWriteBatch writeBatch, byte[] key) throws IOException {
            this.assertClosed();
            try {
                this.counter.incrementAndGet();
                writeBatch.delete(this.getHandle(), key);
            }
            catch (RocksDBException e) {
                throw RocksDatabase.toIOException(this, "batchDelete key " + StringUtils.bytes2String((byte[])key), e);
            }
            finally {
                this.counter.decrementAndGet();
            }
        }

        public void batchPut(ManagedWriteBatch writeBatch, byte[] key, byte[] value) throws IOException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("batchPut array key {}", (Object)StringUtils.bytes2String((byte[])key));
                LOG.debug("batchPut array value {}", (Object)StringUtils.bytes2String((byte[])value));
            }
            this.assertClosed();
            try {
                this.counter.incrementAndGet();
                writeBatch.put(this.getHandle(), key, value);
            }
            catch (RocksDBException e) {
                throw RocksDatabase.toIOException(this, "batchPut key " + StringUtils.bytes2String((byte[])key), e);
            }
            finally {
                this.counter.decrementAndGet();
            }
        }

        public void batchPut(ManagedWriteBatch writeBatch, ByteBuffer key, ByteBuffer value) throws IOException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("batchPut buffer key {}", (Object)StringUtils.bytes2String((ByteBuffer)key.duplicate()));
                LOG.debug("batchPut buffer value {}", (Object)StringUtils.bytes2String((ByteBuffer)value.duplicate()));
            }
            this.assertClosed();
            try {
                this.counter.incrementAndGet();
                writeBatch.put(this.getHandle(), key.duplicate(), value);
            }
            catch (RocksDBException e) {
                throw RocksDatabase.toIOException(this, "batchPut ByteBuffer key " + StringUtils.bytes2String((ByteBuffer)key), e);
            }
            finally {
                this.counter.decrementAndGet();
            }
        }

        public void markClosed() {
            this.isClosed.set(true);
        }

        private void assertClosed() throws IOException {
            if (this.isClosed.get()) {
                throw new IOException("Rocks Database is closed");
            }
        }

        public String toString() {
            return "ColumnFamily-" + this.getName();
        }
    }

    final class RocksCheckpoint
    implements Closeable {
        private final ManagedCheckpoint checkpoint;

        private RocksCheckpoint() {
            this.checkpoint = ManagedCheckpoint.create((ManagedRocksDB)RocksDatabase.this.db);
        }

        public void createCheckpoint(Path path) throws IOException {
            RocksDatabase.this.assertClose();
            try {
                RocksDatabase.this.counter.incrementAndGet();
                ((Checkpoint)this.checkpoint.get()).createCheckpoint(path.toString());
            }
            catch (RocksDBException e) {
                RocksDatabase.this.closeOnError(e, true);
                throw RocksDatabase.toIOException(this, "createCheckpoint " + path, e);
            }
            finally {
                RocksDatabase.this.counter.decrementAndGet();
            }
        }

        public long getLatestSequenceNumber() throws IOException {
            return RocksDatabase.this.getLatestSequenceNumber();
        }

        @Override
        public void close() throws IOException {
            this.checkpoint.close();
        }
    }
}

