/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.ha;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.proto.SCMRatisProtocol;
import org.apache.hadoop.hdds.ratis.RatisHelper;
import org.apache.hadoop.hdds.scm.AddSCMRequest;
import org.apache.hadoop.hdds.scm.RemoveSCMRequest;
import org.apache.hadoop.hdds.scm.ha.HASecurityUtils;
import org.apache.hadoop.hdds.scm.ha.RatisUtil;
import org.apache.hadoop.hdds.scm.ha.SCMHADBTransactionBuffer;
import org.apache.hadoop.hdds.scm.ha.SCMNodeDetails;
import org.apache.hadoop.hdds.scm.ha.SCMRatisRequest;
import org.apache.hadoop.hdds.scm.ha.SCMRatisResponse;
import org.apache.hadoop.hdds.scm.ha.SCMRatisServer;
import org.apache.hadoop.hdds.scm.ha.SCMStateMachine;
import org.apache.hadoop.hdds.scm.server.StorageContainerManager;
import org.apache.hadoop.hdds.security.SecurityConfig;
import org.apache.hadoop.util.Time;
import org.apache.ratis.conf.Parameters;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.grpc.GrpcTlsConfig;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.ClientId;
import org.apache.ratis.protocol.RaftClientReply;
import org.apache.ratis.protocol.RaftClientRequest;
import org.apache.ratis.protocol.RaftGroup;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftPeer;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.protocol.SetConfigurationRequest;
import org.apache.ratis.protocol.SnapshotManagementRequest;
import org.apache.ratis.protocol.exceptions.NotLeaderException;
import org.apache.ratis.server.RaftServer;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMRatisServerImpl
implements SCMRatisServer {
    private static final Logger LOG = LoggerFactory.getLogger(SCMRatisServerImpl.class);
    private final OzoneConfiguration ozoneConf = new OzoneConfiguration();
    private final RaftServer server;
    private final SCMStateMachine stateMachine;
    private final StorageContainerManager scm;
    private final ClientId clientId = ClientId.randomId();
    private final AtomicLong callId = new AtomicLong();
    private final RaftServer.Division division;
    private final GrpcTlsConfig grpcTlsConfig;
    private boolean isStopped;
    private final long requestTimeout;

    SCMRatisServerImpl(ConfigurationSource conf, StorageContainerManager scm, SCMHADBTransactionBuffer buffer) throws IOException {
        this.scm = scm;
        this.requestTimeout = this.ozoneConf.getTimeDuration("ozone.scm.ha.ratis.request.timeout", 30000L, TimeUnit.MILLISECONDS);
        Preconditions.checkArgument((this.requestTimeout > 1000L ? 1 : 0) != 0, (Object)"Ratis request timeout cannot be less than 1000ms.");
        RaftGroupId groupId = SCMRatisServerImpl.buildRaftGroupId(scm.getClusterId());
        LOG.info("starting Raft server for scm:{}", (Object)scm.getScmId());
        this.grpcTlsConfig = HASecurityUtils.createSCMRatisTLSConfig(new SecurityConfig(conf), scm.getScmCertificateClient());
        Parameters parameters = RatisHelper.setServerTlsConf((GrpcTlsConfig)this.grpcTlsConfig);
        this.server = SCMRatisServerImpl.newRaftServer(scm.getScmId(), conf).setStateMachineRegistry(gId -> new SCMStateMachine(scm, buffer)).setOption(RaftStorage.StartupOption.RECOVER).setGroup(RaftGroup.valueOf((RaftGroupId)groupId, (RaftPeer[])new RaftPeer[0])).setParameters(parameters).build();
        this.stateMachine = (SCMStateMachine)this.server.getDivision(groupId).getStateMachine();
        this.division = this.server.getDivision(groupId);
        this.isStopped = false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void initialize(String clusterId, String scmId, SCMNodeDetails details, OzoneConfiguration conf) throws IOException {
        RaftGroup group = SCMRatisServerImpl.buildRaftGroup(details, scmId, clusterId);
        try (RaftServer server = null;){
            server = SCMRatisServerImpl.newRaftServer(scmId, (ConfigurationSource)conf).setGroup(group).setStateMachineRegistry(groupId -> new SCMStateMachine()).setOption(RaftStorage.StartupOption.RECOVER).build();
            server.start();
            SCMRatisServerImpl.waitForLeaderToBeReady(server, conf, group);
        }
    }

    @Override
    public GrpcTlsConfig getGrpcTlsConfig() {
        return this.grpcTlsConfig;
    }

    private static void waitForLeaderToBeReady(RaftServer server, OzoneConfiguration conf, RaftGroup group) throws IOException {
        boolean ready;
        long st = Time.monotonicNow();
        long waitTimeout = conf.getTimeDuration("ozone.scm.ha.ratis.leader.ready.wait.timeout", 60000L, TimeUnit.MILLISECONDS);
        long retryInterval = conf.getTimeDuration("ozone.scm.ha.ratis.leader.ready.check.interval", 2000L, TimeUnit.MILLISECONDS);
        do {
            if (ready = server.getDivision(group.getGroupId()).getInfo().isLeaderReady()) continue;
            try {
                Thread.sleep(retryInterval);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        } while (!ready && Time.monotonicNow() - st < waitTimeout);
        if (!ready) {
            throw new IOException(String.format("Ratis group %s is not ready in %d ms", group.getGroupId(), waitTimeout));
        }
    }

    private static RaftServer.Builder newRaftServer(String scmId, ConfigurationSource conf) {
        RaftProperties serverProperties = RatisUtil.newRaftProperties(conf);
        return RaftServer.newBuilder().setServerId(RaftPeerId.getRaftPeerId((String)scmId)).setProperties(serverProperties);
    }

    @Override
    public void start() throws IOException {
        LOG.info("starting ratis server {}", (Object)this.server.getPeer().getAddress());
        this.server.start();
    }

    @Override
    public RaftServer.Division getDivision() {
        return this.division;
    }

    @VisibleForTesting
    public SCMStateMachine getStateMachine() {
        return this.stateMachine;
    }

    @Override
    public SCMStateMachine getSCMStateMachine() {
        return this.stateMachine;
    }

    @Override
    public void registerStateMachineHandler(SCMRatisProtocol.RequestType handlerType, Object handler) {
        this.stateMachine.registerHandler(handlerType, handler);
    }

    @Override
    public SCMRatisResponse submitRequest(SCMRatisRequest request) throws IOException, ExecutionException, InterruptedException, TimeoutException {
        RaftClientRequest raftClientRequest = RaftClientRequest.newBuilder().setClientId(this.clientId).setServerId(this.getDivision().getId()).setGroupId(this.getDivision().getGroup().getGroupId()).setCallId(this.nextCallId()).setMessage(request.encode()).setType(RaftClientRequest.writeRequestType()).build();
        RaftClientReply raftClientReply = (RaftClientReply)this.server.submitClientRequestAsync(raftClientRequest).get(this.requestTimeout, TimeUnit.MILLISECONDS);
        if (LOG.isDebugEnabled()) {
            LOG.info("request {} Reply {}", (Object)raftClientRequest, (Object)raftClientReply);
        }
        return SCMRatisResponse.decode(raftClientReply);
    }

    @Override
    public boolean triggerSnapshot() throws IOException {
        SnapshotManagementRequest req = SnapshotManagementRequest.newCreate((ClientId)this.clientId, (RaftPeerId)this.getDivision().getId(), (RaftGroupId)this.getDivision().getGroup().getGroupId(), (long)this.nextCallId(), (long)this.requestTimeout);
        RaftClientReply raftClientReply = this.server.snapshotManagement(req);
        if (!raftClientReply.isSuccess()) {
            LOG.warn("Snapshot request failed", (Throwable)raftClientReply.getException());
        }
        return raftClientReply.isSuccess();
    }

    private long nextCallId() {
        return this.callId.getAndIncrement() & Long.MAX_VALUE;
    }

    @Override
    public void stop() throws IOException {
        LOG.info("stopping ratis server {}", (Object)this.server.getPeer().getAddress());
        this.server.close();
        this.isStopped = true;
        this.getSCMStateMachine().close();
    }

    @Override
    public boolean isStopped() {
        return this.isStopped;
    }

    @Override
    public List<String> getRatisRoles() {
        Collection peers = this.division.getGroup().getPeers();
        RaftPeer leader = this.getLeader();
        ArrayList<String> ratisRoles = new ArrayList<String>();
        for (RaftPeer peer : peers) {
            InetAddress peerInetAddress = null;
            try {
                peerInetAddress = InetAddress.getByName((String)HddsUtils.getHostName((String)peer.getAddress()).get());
            }
            catch (IOException ex) {
                LOG.error("SCM Ratis PeerInetAddress {} is unresolvable", (Object)peer.getAddress());
            }
            ratisRoles.add(peer.getAddress() == null ? "" : peer.getAddress().concat(peer.equals((Object)leader) ? ":".concat(RaftProtos.RaftPeerRole.LEADER.toString()) : ":".concat(RaftProtos.RaftPeerRole.FOLLOWER.toString())).concat(":".concat(peer.getId().toString())).concat(":".concat(peerInetAddress == null ? "" : peerInetAddress.getHostAddress())));
        }
        return ratisRoles;
    }

    @Override
    public NotLeaderException triggerNotLeaderException() {
        ByteString leaderId = this.division.getInfo().getRoleInfoProto().getFollowerInfo().getLeaderInfo().getId().getId();
        RaftPeer suggestedLeader = leaderId.isEmpty() ? null : this.division.getRaftConf().getPeer(RaftPeerId.valueOf((ByteString)leaderId), new RaftProtos.RaftPeerRole[0]);
        return new NotLeaderException(this.division.getMemberId(), suggestedLeader, this.division.getGroup().getPeers());
    }

    @Override
    public boolean addSCM(AddSCMRequest request) throws IOException {
        ArrayList<RaftPeer> newRaftPeerList = new ArrayList<RaftPeer>(this.getDivision().getGroup().getPeers());
        RaftPeer raftPeer = RaftPeer.newBuilder().setId(request.getScmId()).setAddress(request.getRatisAddr()).build();
        newRaftPeerList.add(raftPeer);
        LOG.info("{}: Submitting SetConfiguration request to Ratis server with new SCM peers list: {}", (Object)this.scm.getScmId(), newRaftPeerList);
        SetConfigurationRequest configRequest = new SetConfigurationRequest(this.clientId, this.division.getPeer().getId(), this.division.getGroup().getGroupId(), this.nextCallId(), newRaftPeerList);
        try {
            RaftClientReply raftClientReply = this.division.getRaftServer().setConfiguration(configRequest);
            if (!raftClientReply.isSuccess()) {
                LOG.error("Failed to add new SCM: {}. Ratis reply: {}" + request.getScmId(), (Object)raftClientReply);
                throw new IOException((Throwable)raftClientReply.getException());
            }
            LOG.info("Successfully added new SCM: {}.", (Object)request.getScmId());
            return raftClientReply.isSuccess();
        }
        catch (IOException e) {
            LOG.error("Failed to update Ratis configuration and add new peer. Cannot add new SCM: {}.", (Object)this.scm.getScmId(), (Object)e);
            throw e;
        }
    }

    @Override
    public boolean removeSCM(RemoveSCMRequest request) throws IOException {
        ArrayList newRaftPeerList = new ArrayList(this.division.getGroup().getPeers());
        RaftPeer raftPeer = RaftPeer.newBuilder().setId(request.getScmId()).setAddress(request.getRatisAddr()).build();
        newRaftPeerList.remove(raftPeer);
        LOG.info("{}: Submitting SetConfiguration request to Ratis server with updated SCM peers list: {}", (Object)request.getScmId(), newRaftPeerList);
        SetConfigurationRequest configRequest = new SetConfigurationRequest(this.clientId, this.division.getPeer().getId(), this.division.getGroup().getGroupId(), this.nextCallId(), newRaftPeerList);
        try {
            RaftClientReply raftClientReply = this.server.setConfiguration(configRequest);
            if (!raftClientReply.isSuccess()) {
                LOG.error("Failed to remove SCM: {}. Ratis reply: {}" + request.getScmId(), (Object)raftClientReply);
                throw new IOException((Throwable)raftClientReply.getException());
            }
            LOG.info("Successfully removed SCM: {}.", (Object)request.getScmId());
            return raftClientReply.isSuccess();
        }
        catch (IOException e) {
            if (e instanceof NotLeaderException) {
                LOG.debug("Cannot remove peer: {}", (Object)request.getScmId(), (Object)e);
            } else {
                LOG.error("Failed to update Ratis configuration and remove peer. Cannot remove SCM: {}.", (Object)request.getScmId(), (Object)e);
            }
            throw e;
        }
    }

    private static RaftGroup buildRaftGroup(SCMNodeDetails details, String scmId, String clusterId) {
        Preconditions.checkNotNull((Object)scmId);
        RaftGroupId groupId = SCMRatisServerImpl.buildRaftGroupId(clusterId);
        RaftPeerId selfPeerId = SCMRatisServerImpl.getSelfPeerId(scmId);
        RaftPeer localRaftPeer = RaftPeer.newBuilder().setId(selfPeerId).setAddress(details.getRatisHostPortStr()).build();
        ArrayList<RaftPeer> raftPeers = new ArrayList<RaftPeer>();
        raftPeers.add(localRaftPeer);
        RaftGroup group = RaftGroup.valueOf((RaftGroupId)groupId, raftPeers);
        return group;
    }

    public static RaftPeerId getSelfPeerId(String scmId) {
        return RaftPeerId.getRaftPeerId((String)scmId);
    }

    @VisibleForTesting
    public static RaftGroupId buildRaftGroupId(String clusterId) {
        Preconditions.checkNotNull((Object)clusterId);
        return RaftGroupId.valueOf((UUID)UUID.fromString(clusterId.replace("CID-", "")));
    }

    public RaftPeer getLeader() {
        if (this.division.getInfo().isLeader()) {
            return this.division.getPeer();
        }
        ByteString leaderId = this.division.getInfo().getRoleInfoProto().getFollowerInfo().getLeaderInfo().getId().getId();
        return leaderId.isEmpty() ? null : this.division.getRaftConf().getPeer(RaftPeerId.valueOf((ByteString)leaderId), new RaftProtos.RaftPeerRole[0]);
    }
}

