/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.lookup.hash;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import javax.annotation.Nullable;
import org.apache.paimon.compression.BlockCompressionFactory;
import org.apache.paimon.io.CompressedPageFileOutput;
import org.apache.paimon.io.PageFileOutput;
import org.apache.paimon.lookup.LookupStoreFactory;
import org.apache.paimon.lookup.LookupStoreWriter;
import org.apache.paimon.lookup.hash.HashContext;
import org.apache.paimon.utils.BloomFilter;
import org.apache.paimon.utils.MurmurHashUtils;
import org.apache.paimon.utils.VarLengthIntUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HashLookupStoreWriter
implements LookupStoreWriter {
    private static final Logger LOG = LoggerFactory.getLogger((String)HashLookupStoreWriter.class.getName());
    private final double loadFactor;
    private final File tempFolder;
    private final File outputFile;
    private File[] indexFiles;
    private DataOutputStream[] indexStreams;
    private File[] dataFiles;
    private DataOutputStream[] dataStreams;
    private byte[][] lastValues;
    private int[] lastValuesLength;
    private long[] dataLengths;
    private int[] maxOffsetLengths;
    private int keyCount;
    private int[] keyCounts;
    private int valueCount;
    private int collisions;
    @Nullable
    private final BloomFilter.Builder bloomFilter;
    @Nullable
    private final BlockCompressionFactory compressionFactory;
    private final int compressPageSize;

    HashLookupStoreWriter(double loadFactor, File file, @Nullable BloomFilter.Builder bloomFilter, @Nullable BlockCompressionFactory compressionFactory, int compressPageSize) throws IOException {
        this.loadFactor = loadFactor;
        this.outputFile = file;
        this.compressionFactory = compressionFactory;
        this.compressPageSize = compressPageSize;
        if (loadFactor <= 0.0 || loadFactor >= 1.0) {
            throw new IllegalArgumentException("Illegal load factor = " + loadFactor + ", should be between 0.0 and 1.0.");
        }
        this.tempFolder = new File(file.getParentFile(), UUID.randomUUID().toString());
        if (!this.tempFolder.mkdir()) {
            throw new IOException("Can not create temp folder: " + this.tempFolder);
        }
        this.indexStreams = new DataOutputStream[0];
        this.dataStreams = new DataOutputStream[0];
        this.indexFiles = new File[0];
        this.dataFiles = new File[0];
        this.lastValues = new byte[0][];
        this.lastValuesLength = new int[0];
        this.dataLengths = new long[0];
        this.maxOffsetLengths = new int[0];
        this.keyCounts = new int[0];
        this.bloomFilter = bloomFilter;
    }

    @Override
    public void put(byte[] key, byte[] value) throws IOException {
        int keyLength = key.length;
        DataOutputStream indexStream = this.getIndexStream(keyLength);
        indexStream.write(key);
        byte[] lastValue = this.lastValues[keyLength];
        boolean sameValue = lastValue != null && Arrays.equals(value, lastValue);
        long dataLength = this.dataLengths[keyLength];
        if (sameValue) {
            dataLength -= (long)this.lastValuesLength[keyLength];
        }
        int offsetLength = VarLengthIntUtils.encodeLong(indexStream, dataLength);
        this.maxOffsetLengths[keyLength] = Math.max(offsetLength, this.maxOffsetLengths[keyLength]);
        if (!sameValue) {
            DataOutputStream dataStream = this.getDataStream(keyLength);
            int valueSize = VarLengthIntUtils.encodeInt(dataStream, value.length);
            dataStream.write(value);
            int n = keyLength;
            this.dataLengths[n] = this.dataLengths[n] + (long)(valueSize + value.length);
            this.lastValues[keyLength] = value;
            this.lastValuesLength[keyLength] = valueSize + value.length;
            ++this.valueCount;
        }
        ++this.keyCount;
        int n = keyLength;
        this.keyCounts[n] = this.keyCounts[n] + 1;
        if (this.bloomFilter != null) {
            this.bloomFilter.addHash(MurmurHashUtils.hashBytes(key));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public LookupStoreFactory.Context close() throws IOException {
        int i;
        for (DataOutputStream dos : this.dataStreams) {
            if (dos == null) continue;
            dos.close();
        }
        for (DataOutputStream dos : this.indexStreams) {
            if (dos == null) continue;
            dos.close();
        }
        LOG.info("Number of keys: {}", (Object)this.keyCount);
        LOG.info("Number of values: {}", (Object)this.valueCount);
        ArrayList<File> filesToMerge = new ArrayList<File>();
        int bloomFilterBytes = this.bloomFilter == null ? 0 : this.bloomFilter.getBuffer().size();
        HashContext context = new HashContext(this.bloomFilter != null, this.bloomFilter == null ? 0L : this.bloomFilter.expectedEntries(), bloomFilterBytes, new int[this.keyCounts.length], new int[this.keyCounts.length], new int[this.keyCounts.length], new int[this.keyCounts.length], new long[this.keyCounts.length], 0L, null);
        long indexesLength = bloomFilterBytes;
        long datasLength = 0L;
        for (i = 0; i < this.keyCounts.length; ++i) {
            int slots;
            if (this.keyCounts[i] <= 0) continue;
            context.keyCounts[i] = this.keyCounts[i];
            context.slots[i] = slots = (int)Math.round((double)this.keyCounts[i] / this.loadFactor);
            int offsetLength = this.maxOffsetLengths[i];
            context.slotSizes[i] = i + offsetLength;
            context.indexOffsets[i] = (int)indexesLength;
            indexesLength += (long)(i + offsetLength) * (long)slots;
            context.dataOffsets[i] = datasLength;
            datasLength += this.dataLengths[i];
        }
        for (i = 0; i < context.dataOffsets.length; ++i) {
            context.dataOffsets[i] = indexesLength + context.dataOffsets[i];
        }
        PageFileOutput output = PageFileOutput.create(this.outputFile, this.compressPageSize, this.compressionFactory);
        try {
            if (this.bloomFilter != null) {
                File bloomFilterFile = new File(this.tempFolder, "bloomfilter.dat");
                try (FileOutputStream bfOutputStream = new FileOutputStream(bloomFilterFile);){
                    bfOutputStream.write(this.bloomFilter.getBuffer().getArray());
                    LOG.info("Bloom filter size: {} bytes", (Object)this.bloomFilter.getBuffer().size());
                }
                filesToMerge.add(bloomFilterFile);
            }
            for (int i2 = 0; i2 < this.indexFiles.length; ++i2) {
                if (this.indexFiles[i2] == null) continue;
                filesToMerge.add(this.buildIndex(i2));
            }
            LOG.info("Number of collisions: {}", (Object)this.collisions);
            for (File dataFile : this.dataFiles) {
                if (dataFile == null) continue;
                filesToMerge.add(dataFile);
            }
            this.checkFreeDiskSpace(filesToMerge);
            this.mergeFiles(filesToMerge, output);
        }
        finally {
            this.cleanup(filesToMerge);
            output.close();
        }
        LOG.info("Compressed Total store size: {} Mb", (Object)new DecimalFormat("#,##0.0").format(this.outputFile.length() / 0x100000L));
        if (output instanceof CompressedPageFileOutput) {
            CompressedPageFileOutput compressedOutput = (CompressedPageFileOutput)output;
            context = context.copy(compressedOutput.uncompressBytes(), compressedOutput.pages());
        }
        return context;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private File buildIndex(int keyLength) throws IOException {
        long count = this.keyCounts[keyLength];
        int slots = (int)Math.round((double)count / this.loadFactor);
        int offsetLength = this.maxOffsetLengths[keyLength];
        int slotSize = keyLength + offsetLength;
        File indexFile = new File(this.tempFolder, "index" + keyLength + ".dat");
        try (RandomAccessFile indexAccessFile = new RandomAccessFile(indexFile, "rw");){
            indexAccessFile.setLength((long)slots * (long)slotSize);
            FileChannel indexChannel = indexAccessFile.getChannel();
            MappedByteBuffer byteBuffer = indexChannel.map(FileChannel.MapMode.READ_WRITE, 0L, indexAccessFile.length());
            File tempIndexFile = this.indexFiles[keyLength];
            DataInputStream tempIndexStream = new DataInputStream(new BufferedInputStream(new FileInputStream(tempIndexFile)));
            try {
                byte[] keyBuffer = new byte[keyLength];
                byte[] slotBuffer = new byte[slotSize];
                byte[] offsetBuffer = new byte[offsetLength];
                int i = 0;
                while ((long)i < count) {
                    tempIndexStream.readFully(keyBuffer);
                    long offset = VarLengthIntUtils.decodeLong(tempIndexStream);
                    long hash = MurmurHashUtils.hashBytesPositive(keyBuffer);
                    boolean collision = false;
                    int probe = 0;
                    while ((long)probe < count) {
                        int slot = (int)((hash + (long)probe) % (long)slots);
                        byteBuffer.position(slot * slotSize);
                        byteBuffer.get(slotBuffer);
                        long found = VarLengthIntUtils.decodeLong(slotBuffer, keyLength);
                        if (found == 0L) {
                            byteBuffer.position(slot * slotSize);
                            byteBuffer.put(keyBuffer);
                            int pos = VarLengthIntUtils.encodeLong(offsetBuffer, offset);
                            byteBuffer.put(offsetBuffer, 0, pos);
                            break;
                        }
                        collision = true;
                        if (Arrays.equals(keyBuffer, Arrays.copyOf(slotBuffer, keyLength))) {
                            throw new RuntimeException(String.format("A duplicate key has been found for for key bytes %s", Arrays.toString(keyBuffer)));
                        }
                        ++probe;
                    }
                    if (collision) {
                        ++this.collisions;
                    }
                    ++i;
                }
                String msg = "  Max offset length: " + offsetLength + " bytes\n  Slot size: " + slotSize + " bytes";
                LOG.info("Built index file {}\n" + msg, (Object)indexFile.getName());
            }
            finally {
                tempIndexStream.close();
                indexChannel.close();
                if (tempIndexFile.delete()) {
                    LOG.info("Temporary index file {} has been deleted", (Object)tempIndexFile.getName());
                }
            }
        }
        return indexFile;
    }

    private void checkFreeDiskSpace(List<File> inputFiles) {
        long usableSpace = 0L;
        long totalSize = 0L;
        for (File f : inputFiles) {
            if (!f.exists()) continue;
            totalSize += f.length();
            usableSpace = f.getUsableSpace();
        }
        LOG.info("Total expected store size is {} Mb", (Object)new DecimalFormat("#,##0.0").format(totalSize / 0x100000L));
        LOG.info("Usable free space on the system is {} Mb", (Object)new DecimalFormat("#,##0.0").format(usableSpace / 0x100000L));
        if ((double)totalSize / (double)usableSpace >= 0.66) {
            throw new RuntimeException("Aborting because there isn' enough free disk space");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void mergeFiles(List<File> inputFiles, PageFileOutput output) throws IOException {
        long startTime = System.nanoTime();
        for (File f : inputFiles) {
            if (f.exists()) {
                FileInputStream fileInputStream = new FileInputStream(f);
                BufferedInputStream bufferedInputStream = new BufferedInputStream(fileInputStream);
                try {
                    int length;
                    LOG.info("Merging {} size={}", (Object)f.getName(), (Object)f.length());
                    byte[] buffer = new byte[8192];
                    while ((length = bufferedInputStream.read(buffer)) > 0) {
                        output.write(buffer, 0, length);
                    }
                    continue;
                }
                finally {
                    bufferedInputStream.close();
                    fileInputStream.close();
                    continue;
                }
            }
            LOG.info("Skip merging file {} because it doesn't exist", (Object)f.getName());
        }
        LOG.info("Time to merge {} s", (Object)((double)(System.nanoTime() - startTime) / 1.0E9));
    }

    private void cleanup(List<File> inputFiles) {
        for (File f : inputFiles) {
            if (!f.exists() || !f.delete()) continue;
            LOG.info("Deleted temporary file {}", (Object)f.getName());
        }
        if (this.tempFolder.delete()) {
            LOG.info("Deleted temporary folder at {}", (Object)this.tempFolder.getAbsolutePath());
        }
    }

    private DataOutputStream getDataStream(int keyLength) throws IOException {
        DataOutputStream dos;
        if (this.dataStreams.length <= keyLength) {
            this.dataStreams = Arrays.copyOf(this.dataStreams, keyLength + 1);
            this.dataFiles = Arrays.copyOf(this.dataFiles, keyLength + 1);
        }
        if ((dos = this.dataStreams[keyLength]) == null) {
            File file;
            this.dataFiles[keyLength] = file = new File(this.tempFolder, "data" + keyLength + ".dat");
            this.dataStreams[keyLength] = dos = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(file)));
            dos.writeByte(0);
        }
        return dos;
    }

    private DataOutputStream getIndexStream(int keyLength) throws IOException {
        DataOutputStream dos;
        if (this.indexStreams.length <= keyLength) {
            this.indexStreams = Arrays.copyOf(this.indexStreams, keyLength + 1);
            this.indexFiles = Arrays.copyOf(this.indexFiles, keyLength + 1);
            this.keyCounts = Arrays.copyOf(this.keyCounts, keyLength + 1);
            this.maxOffsetLengths = Arrays.copyOf(this.maxOffsetLengths, keyLength + 1);
            this.lastValues = (byte[][])Arrays.copyOf(this.lastValues, keyLength + 1);
            this.lastValuesLength = Arrays.copyOf(this.lastValuesLength, keyLength + 1);
            this.dataLengths = Arrays.copyOf(this.dataLengths, keyLength + 1);
        }
        if ((dos = this.indexStreams[keyLength]) == null) {
            File file;
            this.indexFiles[keyLength] = file = new File(this.tempFolder, "temp_index" + keyLength + ".dat");
            this.indexStreams[keyLength] = dos = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(file)));
            int n = keyLength;
            this.dataLengths[n] = this.dataLengths[n] + 1L;
        }
        return dos;
    }
}

