/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.tools.ci.licensecheck;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.apache.paimon.tools.ci.utils.dependency.DependencyParser;
import org.apache.paimon.tools.ci.utils.deploy.DeployParser;
import org.apache.paimon.tools.ci.utils.notice.NoticeContents;
import org.apache.paimon.tools.ci.utils.notice.NoticeParser;
import org.apache.paimon.tools.ci.utils.shade.ShadeParser;
import org.apache.paimon.tools.ci.utils.shared.Dependency;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NoticeFileChecker {
    private static final Logger LOG = LoggerFactory.getLogger(NoticeFileChecker.class);
    private static final List<String> MODULES_DEFINING_EXCESS_DEPENDENCIES = NoticeFileChecker.loadFromResources("modules-defining-excess-dependencies.modulelist");
    private static final Pattern NOTICE_DEPENDENCY_PATTERN = Pattern.compile("- ([^ :]+):([^:]+):([^ ]+)($| )|.*bundles \"([^:]+):([^:]+):([^\"]+)\".*");

    static int run(File buildResult, Path root) throws IOException {
        Map<String, Set<Dependency>> modulesWithBundledDependencies = NoticeFileChecker.combineAndFilterPaimonDependencies(ShadeParser.parseShadeOutput(buildResult.toPath()), DependencyParser.parseDependencyCopyOutput(buildResult.toPath()));
        Set<String> deployedModules = DeployParser.parseDeployOutput(buildResult);
        LOG.info("Extracted " + deployedModules.size() + " modules that were deployed and " + modulesWithBundledDependencies.keySet().size() + " modules which bundle dependencies with a total of " + modulesWithBundledDependencies.values().size() + " dependencies");
        List<Path> noticeFiles = NoticeFileChecker.findNoticeFiles(root);
        LOG.info("Found {} NOTICE files to check", (Object)noticeFiles.size());
        Map<String, Optional<NoticeContents>> moduleToNotice = noticeFiles.stream().collect(Collectors.toMap(NoticeFileChecker::getModuleFromNoticeFile, noticeFile -> {
            try {
                return NoticeParser.parseNoticeFile(noticeFile);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }));
        return NoticeFileChecker.run(modulesWithBundledDependencies, deployedModules, moduleToNotice);
    }

    static int run(Map<String, Set<Dependency>> modulesWithBundledDependencies, Set<String> deployedModules, Map<String, Optional<NoticeContents>> noticeFiles) throws IOException {
        int severeIssueCount = 0;
        HashSet<String> modulesSkippingDeployment = new HashSet<String>(modulesWithBundledDependencies.keySet());
        modulesSkippingDeployment.removeAll(deployedModules);
        LOG.debug("The following {} modules are skipping deployment: {}", (Object)modulesSkippingDeployment.size(), (Object)modulesSkippingDeployment.stream().sorted().collect(Collectors.joining("\n\t", "\n\t", "")));
        for (String string : modulesSkippingDeployment) {
            boolean bundledByDeployedModule = modulesWithBundledDependencies.entrySet().stream().filter(entry -> ((Set)entry.getValue()).stream().map(Dependency::getArtifactId).anyMatch(artifactId -> artifactId.equals(moduleSkippingDeployment))).anyMatch(entry -> !modulesSkippingDeployment.contains(entry.getKey()));
            if (!bundledByDeployedModule) {
                modulesWithBundledDependencies.remove(string);
                continue;
            }
            LOG.debug("Including module {} in license checks, despite not being deployed, because it is bundled by another deployed module.", (Object)string);
        }
        severeIssueCount += NoticeFileChecker.ensureRequiredNoticeFiles(modulesWithBundledDependencies, noticeFiles.keySet());
        for (Map.Entry entry2 : noticeFiles.entrySet()) {
            severeIssueCount += NoticeFileChecker.checkNoticeFileAndLogProblems(modulesWithBundledDependencies, (String)entry2.getKey(), ((Optional)entry2.getValue()).orElse(null));
        }
        return severeIssueCount;
    }

    private static Map<String, Set<Dependency>> combineAndFilterPaimonDependencies(Map<String, Set<Dependency>> modulesWithBundledDependencies, Map<String, Set<Dependency>> modulesWithCopiedDependencies) {
        LinkedHashMap<String, Set<Dependency>> combinedAndFiltered = new LinkedHashMap<String, Set<Dependency>>();
        Stream.concat(modulesWithBundledDependencies.entrySet().stream(), modulesWithCopiedDependencies.entrySet().stream()).forEach(entry -> {
            Set dependencies = combinedAndFiltered.computeIfAbsent((String)entry.getKey(), ignored -> new LinkedHashSet());
            for (Dependency dependency : (Set)entry.getValue()) {
                if (dependency.getGroupId().contains("org.apache.paimon")) continue;
                dependencies.add(dependency);
            }
        });
        return combinedAndFiltered;
    }

    private static int ensureRequiredNoticeFiles(Map<String, Set<Dependency>> modulesWithShadedDependencies, Collection<String> modulesWithNoticeFile) {
        int severeIssueCount = 0;
        HashSet<String> shadingModules = new HashSet<String>(modulesWithShadedDependencies.keySet());
        shadingModules.removeAll(modulesWithNoticeFile);
        for (String moduleWithoutNotice : shadingModules) {
            if (!modulesWithShadedDependencies.get(moduleWithoutNotice).stream().anyMatch(dependency -> !dependency.getGroupId().equals("org.apache.paimon"))) continue;
            LOG.error("Module {} is missing a NOTICE file. It has shaded dependencies: {}", (Object)moduleWithoutNotice, (Object)modulesWithShadedDependencies.get(moduleWithoutNotice).stream().map(Dependency::toString).collect(Collectors.joining("\n\t", "\n\t", "")));
            ++severeIssueCount;
        }
        return severeIssueCount;
    }

    private static String getModuleFromNoticeFile(Path noticeFile) {
        Path moduleDirectory = noticeFile.getParent().getParent().getParent().getParent().getParent();
        return moduleDirectory.getFileName().toString();
    }

    private static int checkNoticeFileAndLogProblems(Map<String, Set<Dependency>> modulesWithShadedDependencies, String moduleName, @Nullable NoticeContents noticeContents) throws IOException {
        Map<Severity, List<String>> problemsBySeverity = NoticeFileChecker.checkNoticeFile(modulesWithShadedDependencies, moduleName, noticeContents);
        List<String> severeProblems = problemsBySeverity.getOrDefault((Object)Severity.CRITICAL, Collections.emptyList());
        if (!problemsBySeverity.isEmpty()) {
            List<String> toleratedProblems = problemsBySeverity.getOrDefault((Object)Severity.TOLERATED, Collections.emptyList());
            List<String> expectedProblems = problemsBySeverity.getOrDefault((Object)Severity.SUPPRESSED, Collections.emptyList());
            LOG.info("Problems were detected for a NOTICE file.\n\t{}:\n{}{}{}", new Object[]{moduleName, NoticeFileChecker.convertProblemsToIndentedString(severeProblems, "These issue are legally problematic and MUST be fixed:"), NoticeFileChecker.convertProblemsToIndentedString(toleratedProblems, "These issues are mistakes that aren't legally problematic. They SHOULD be fixed at some point, but we don't have to:"), NoticeFileChecker.convertProblemsToIndentedString(expectedProblems, "These issues are assumed to be false-positives:")});
        }
        return severeProblems.size();
    }

    static Map<Severity, List<String>> checkNoticeFile(Map<String, Set<Dependency>> modulesWithShadedDependencies, String moduleName, @Nullable NoticeContents noticeContents) {
        HashMap<Severity, List<String>> problemsBySeverity = new HashMap<Severity, List<String>>();
        if (noticeContents == null) {
            NoticeFileChecker.addProblem(problemsBySeverity, Severity.CRITICAL, "The NOTICE file was empty.");
        } else {
            if (!noticeContents.getNoticeModuleName().equals(moduleName)) {
                NoticeFileChecker.addProblem(problemsBySeverity, Severity.TOLERATED, String.format("First line does not start with module name. firstLine=%s", noticeContents.getNoticeModuleName()));
            }
            HashSet<Dependency> declaredDependencies = new HashSet<Dependency>();
            for (Dependency dependency2 : noticeContents.getDeclaredDependencies()) {
                if (declaredDependencies.add(dependency2)) continue;
                NoticeFileChecker.addProblem(problemsBySeverity, Severity.CRITICAL, String.format("Dependency %s is declared twice.", dependency2));
            }
            Collection expectedDependencies = modulesWithShadedDependencies.getOrDefault(moduleName, Collections.emptySet()).stream().filter(dependency -> !dependency.getGroupId().equals("org.apache.paimon")).collect(Collectors.toList());
            for (Dependency expectedDependency : expectedDependencies) {
                if (declaredDependencies.contains(expectedDependency)) continue;
                NoticeFileChecker.addProblem(problemsBySeverity, Severity.CRITICAL, String.format("Dependency %s is not listed.", expectedDependency));
            }
            boolean bl = MODULES_DEFINING_EXCESS_DEPENDENCIES.contains(moduleName);
            for (Dependency declaredDependency : declaredDependencies) {
                if (expectedDependencies.contains(declaredDependency)) continue;
                Severity severity = bl ? Severity.SUPPRESSED : Severity.TOLERATED;
                NoticeFileChecker.addProblem(problemsBySeverity, severity, String.format("Dependency %s is not bundled, but listed.", declaredDependency));
            }
        }
        return problemsBySeverity;
    }

    private static void addProblem(Map<Severity, List<String>> problemsBySeverity, Severity severity, String problem) {
        problemsBySeverity.computeIfAbsent(severity, ignored -> new ArrayList()).add(problem);
    }

    private static String convertProblemsToIndentedString(List<String> problems, String header) {
        return problems.isEmpty() ? "" : problems.stream().map(s -> "\t\t\t" + s).collect(Collectors.joining("\n", "\t\t " + header + " \n", "\n"));
    }

    private static List<Path> findNoticeFiles(Path root) throws IOException {
        return Files.walk(root, new FileVisitOption[0]).filter(file -> {
            int nameCount = file.getNameCount();
            return file.getName(nameCount - 3).toString().equals("resources") && file.getName(nameCount - 2).toString().equals("META-INF") && file.getName(nameCount - 1).toString().equals("NOTICE");
        }).collect(Collectors.toList());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static List<String> loadFromResources(String fileName) {
        try (BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(Objects.requireNonNull(NoticeFileChecker.class.getResourceAsStream("/" + fileName))));){
            List<String> result = bufferedReader.lines().filter(line -> !line.startsWith("#") && !line.isEmpty()).collect(Collectors.toList());
            LOG.debug("Loaded {} items from resource {}", (Object)result.size(), (Object)fileName);
            List<String> list = result;
            return list;
        }
        catch (Throwable e) {
            throw new RuntimeException("Error while loading resource", e);
        }
    }

    static enum Severity {
        CRITICAL,
        TOLERATED,
        SUPPRESSED;

    }
}

