/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.catalog;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.paimon.CoreOptions;
import org.apache.paimon.annotation.VisibleForTesting;
import org.apache.paimon.catalog.Catalog;
import org.apache.paimon.catalog.CatalogLockContext;
import org.apache.paimon.catalog.CatalogLockFactory;
import org.apache.paimon.catalog.Identifier;
import org.apache.paimon.factories.FactoryUtil;
import org.apache.paimon.fs.FileIO;
import org.apache.paimon.fs.Path;
import org.apache.paimon.lineage.LineageMetaFactory;
import org.apache.paimon.operation.FileStoreCommit;
import org.apache.paimon.operation.Lock;
import org.apache.paimon.options.CatalogOptions;
import org.apache.paimon.options.Options;
import org.apache.paimon.options.OptionsUtils;
import org.apache.paimon.schema.Schema;
import org.apache.paimon.schema.SchemaChange;
import org.apache.paimon.schema.TableSchema;
import org.apache.paimon.table.CatalogEnvironment;
import org.apache.paimon.table.FileStoreTable;
import org.apache.paimon.table.FileStoreTableFactory;
import org.apache.paimon.table.Table;
import org.apache.paimon.table.system.SystemTableLoader;
import org.apache.paimon.utils.Preconditions;
import org.apache.paimon.utils.StringUtils;

public abstract class AbstractCatalog
implements Catalog {
    public static final String DB_SUFFIX = ".db";
    protected static final String TABLE_DEFAULT_OPTION_PREFIX = "table-default.";
    protected static final String DB_LOCATION_PROP = "location";
    protected final FileIO fileIO;
    protected final Map<String, String> tableDefaultOptions;
    protected final Options catalogOptions;
    @Nullable
    protected final LineageMetaFactory lineageMetaFactory;

    protected AbstractCatalog(FileIO fileIO) {
        this.fileIO = fileIO;
        this.lineageMetaFactory = null;
        this.tableDefaultOptions = new HashMap<String, String>();
        this.catalogOptions = new Options();
    }

    protected AbstractCatalog(FileIO fileIO, Options options) {
        this.fileIO = fileIO;
        this.lineageMetaFactory = this.findAndCreateLineageMeta(options, AbstractCatalog.class.getClassLoader());
        this.tableDefaultOptions = OptionsUtils.convertToPropertiesPrefixKey(options.toMap(), TABLE_DEFAULT_OPTION_PREFIX);
        this.catalogOptions = options;
    }

    @Override
    public Map<String, String> options() {
        return this.catalogOptions.toMap();
    }

    @Override
    public FileIO fileIO() {
        return this.fileIO;
    }

    @Override
    public Optional<CatalogLockFactory> lockFactory() {
        if (!this.lockEnabled()) {
            return Optional.empty();
        }
        String lock = this.catalogOptions.get(CatalogOptions.LOCK_TYPE);
        if (lock == null) {
            return this.defaultLockFactory();
        }
        return Optional.of(FactoryUtil.discoverFactory(AbstractCatalog.class.getClassLoader(), CatalogLockFactory.class, lock));
    }

    public Optional<CatalogLockFactory> defaultLockFactory() {
        return Optional.empty();
    }

    @Override
    public Optional<CatalogLockContext> lockContext() {
        return Optional.of(CatalogLockContext.fromOptions(this.catalogOptions));
    }

    protected boolean lockEnabled() {
        return this.catalogOptions.get(CatalogOptions.LOCK_ENABLED);
    }

    @Override
    public boolean databaseExists(String databaseName) {
        if (this.isSystemDatabase(databaseName)) {
            return true;
        }
        return this.databaseExistsImpl(databaseName);
    }

    protected abstract boolean databaseExistsImpl(String var1);

    @Override
    public void createDatabase(String name, boolean ignoreIfExists, Map<String, String> properties) throws Catalog.DatabaseAlreadyExistException {
        if (this.isSystemDatabase(name)) {
            throw new Catalog.ProcessSystemDatabaseException();
        }
        if (this.databaseExists(name)) {
            if (ignoreIfExists) {
                return;
            }
            throw new Catalog.DatabaseAlreadyExistException(name);
        }
        this.createDatabaseImpl(name, properties);
    }

    @Override
    public Map<String, String> loadDatabaseProperties(String name) throws Catalog.DatabaseNotExistException {
        if (this.isSystemDatabase(name)) {
            return Collections.emptyMap();
        }
        if (!this.databaseExists(name)) {
            throw new Catalog.DatabaseNotExistException(name);
        }
        return this.loadDatabasePropertiesImpl(name);
    }

    protected abstract Map<String, String> loadDatabasePropertiesImpl(String var1);

    @Override
    public void dropPartition(Identifier identifier, Map<String, String> partitionSpec) throws Catalog.TableNotExistException {
        Table table = this.getTable(identifier);
        FileStoreTable fileStoreTable = (FileStoreTable)table;
        FileStoreCommit commit = fileStoreTable.store().newCommit(UUID.randomUUID().toString());
        commit.dropPartitions(Collections.singletonList(partitionSpec), Long.MAX_VALUE);
    }

    protected abstract void createDatabaseImpl(String var1, Map<String, String> var2);

    @Override
    public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade) throws Catalog.DatabaseNotExistException, Catalog.DatabaseNotEmptyException {
        if (this.isSystemDatabase(name)) {
            throw new Catalog.ProcessSystemDatabaseException();
        }
        if (!this.databaseExists(name)) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.DatabaseNotExistException(name);
        }
        if (!cascade && this.listTables(name).size() > 0) {
            throw new Catalog.DatabaseNotEmptyException(name);
        }
        this.dropDatabaseImpl(name);
    }

    protected abstract void dropDatabaseImpl(String var1);

    @Override
    public List<String> listTables(String databaseName) throws Catalog.DatabaseNotExistException {
        if (this.isSystemDatabase(databaseName)) {
            return SystemTableLoader.loadGlobalTableNames();
        }
        if (!this.databaseExists(databaseName)) {
            throw new Catalog.DatabaseNotExistException(databaseName);
        }
        return this.listTablesImpl(databaseName).stream().sorted().collect(Collectors.toList());
    }

    protected abstract List<String> listTablesImpl(String var1);

    @Override
    public void dropTable(Identifier identifier, boolean ignoreIfNotExists) throws Catalog.TableNotExistException {
        this.checkNotSystemTable(identifier, "dropTable");
        if (!this.tableExists(identifier)) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(identifier);
        }
        this.dropTableImpl(identifier);
    }

    protected abstract void dropTableImpl(Identifier var1);

    @Override
    public void createTable(Identifier identifier, Schema schema, boolean ignoreIfExists) throws Catalog.TableAlreadyExistException, Catalog.DatabaseNotExistException {
        this.checkNotSystemTable(identifier, "createTable");
        this.validateIdentifierNameCaseInsensitive(identifier);
        this.validateFieldNameCaseInsensitive(schema.rowType().getFieldNames());
        this.validateAutoCreateClose(schema.options());
        if (!this.databaseExists(identifier.getDatabaseName())) {
            throw new Catalog.DatabaseNotExistException(identifier.getDatabaseName());
        }
        if (this.tableExists(identifier)) {
            if (ignoreIfExists) {
                return;
            }
            throw new Catalog.TableAlreadyExistException(identifier);
        }
        this.copyTableDefaultOptions(schema.options());
        this.createTableImpl(identifier, schema);
    }

    protected abstract void createTableImpl(Identifier var1, Schema var2);

    @Override
    public void renameTable(Identifier fromTable, Identifier toTable, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.TableAlreadyExistException {
        this.checkNotSystemTable(fromTable, "renameTable");
        this.checkNotSystemTable(toTable, "renameTable");
        this.validateIdentifierNameCaseInsensitive(toTable);
        if (!this.tableExists(fromTable)) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(fromTable);
        }
        if (this.tableExists(toTable)) {
            throw new Catalog.TableAlreadyExistException(toTable);
        }
        this.renameTableImpl(fromTable, toTable);
    }

    protected abstract void renameTableImpl(Identifier var1, Identifier var2);

    @Override
    public void alterTable(Identifier identifier, List<SchemaChange> changes, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.ColumnAlreadyExistException, Catalog.ColumnNotExistException {
        this.checkNotSystemTable(identifier, "alterTable");
        this.validateIdentifierNameCaseInsensitive(identifier);
        this.validateFieldNameCaseInsensitiveInSchemaChange(changes);
        if (!this.tableExists(identifier)) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(identifier);
        }
        this.alterTableImpl(identifier, changes);
    }

    protected abstract void alterTableImpl(Identifier var1, List<SchemaChange> var2) throws Catalog.TableNotExistException, Catalog.ColumnAlreadyExistException, Catalog.ColumnNotExistException;

    @Nullable
    private LineageMetaFactory findAndCreateLineageMeta(Options options, ClassLoader classLoader) {
        return options.getOptional(CatalogOptions.LINEAGE_META).map(meta -> FactoryUtil.discoverFactory(classLoader, LineageMetaFactory.class, meta)).orElse(null);
    }

    @Override
    public Table getTable(Identifier identifier) throws Catalog.TableNotExistException {
        if (this.isSystemDatabase(identifier.getDatabaseName())) {
            String tableName = identifier.getObjectName();
            Table table = SystemTableLoader.loadGlobal(tableName, this.fileIO, this::allTablePaths, this.catalogOptions, this.lineageMetaFactory);
            if (table == null) {
                throw new Catalog.TableNotExistException(identifier);
            }
            return table;
        }
        if (AbstractCatalog.isSpecifiedSystemTable(identifier)) {
            String[] splits = this.tableAndSystemName(identifier);
            String tableName = splits[0];
            String type = splits[1];
            FileStoreTable originTable = this.getDataTable(new Identifier(identifier.getDatabaseName(), tableName));
            Table table = SystemTableLoader.load(type, this.fileIO, originTable);
            if (table == null) {
                throw new Catalog.TableNotExistException(identifier);
            }
            return table;
        }
        return this.getDataTable(identifier);
    }

    private FileStoreTable getDataTable(Identifier identifier) throws Catalog.TableNotExistException {
        TableSchema tableSchema = this.getDataTableSchema(identifier);
        return FileStoreTableFactory.create(this.fileIO, this.getDataTableLocation(identifier), tableSchema, new CatalogEnvironment(Lock.factory(this.lockFactory().orElse(null), this.lockContext().orElse(null), identifier), this.metastoreClientFactory(identifier).orElse(null), this.lineageMetaFactory));
    }

    public Path newDatabasePath(String database) {
        return AbstractCatalog.newDatabasePath(this.warehouse(), database);
    }

    public Map<String, Map<String, Path>> allTablePaths() {
        try {
            HashMap<String, Map<String, Path>> allPaths = new HashMap<String, Map<String, Path>>();
            for (String database : this.listDatabases()) {
                Map tableMap = allPaths.computeIfAbsent(database, d -> new HashMap());
                for (String table : this.listTables(database)) {
                    tableMap.put(table, this.getDataTableLocation(Identifier.create(database, table)));
                }
            }
            return allPaths;
        }
        catch (Catalog.DatabaseNotExistException e) {
            throw new RuntimeException("Database is deleted while listing", e);
        }
    }

    protected abstract TableSchema getDataTableSchema(Identifier var1) throws Catalog.TableNotExistException;

    @VisibleForTesting
    public Path getDataTableLocation(Identifier identifier) {
        return new Path(this.newDatabasePath(identifier.getDatabaseName()), identifier.getObjectName());
    }

    private static boolean isSpecifiedSystemTable(Identifier identifier) {
        return identifier.getObjectName().contains("$");
    }

    protected boolean isSystemTable(Identifier identifier) {
        return this.isSystemDatabase(identifier.getDatabaseName()) || AbstractCatalog.isSpecifiedSystemTable(identifier);
    }

    private void checkNotSystemTable(Identifier identifier, String method) {
        if (this.isSystemTable(identifier)) {
            throw new IllegalArgumentException(String.format("Cannot '%s' for system table '%s', please use data table.", method, identifier));
        }
    }

    public void copyTableDefaultOptions(Map<String, String> options) {
        this.tableDefaultOptions.forEach(options::putIfAbsent);
    }

    private String[] tableAndSystemName(Identifier identifier) {
        String[] splits = StringUtils.split(identifier.getObjectName(), "$");
        if (splits.length != 2) {
            throw new IllegalArgumentException("System table can only contain one '$' separator, but this is: " + identifier.getObjectName());
        }
        return splits;
    }

    public static Path newTableLocation(String warehouse, Identifier identifier) {
        if (AbstractCatalog.isSpecifiedSystemTable(identifier)) {
            throw new IllegalArgumentException(String.format("Table name[%s] cannot contain '%s' separator", identifier.getObjectName(), "$"));
        }
        return new Path(AbstractCatalog.newDatabasePath(warehouse, identifier.getDatabaseName()), identifier.getObjectName());
    }

    public static Path newDatabasePath(String warehouse, String database) {
        return new Path(warehouse, database + DB_SUFFIX);
    }

    private boolean isSystemDatabase(String database) {
        return "sys".equals(database);
    }

    public static void validateCaseInsensitive(boolean caseSensitive, String type, String ... names) {
        AbstractCatalog.validateCaseInsensitive(caseSensitive, type, Arrays.asList(names));
    }

    public static void validateCaseInsensitive(boolean caseSensitive, String type, List<String> names) {
        if (caseSensitive) {
            return;
        }
        List illegalNames = names.stream().filter(f -> !f.equals(f.toLowerCase())).collect(Collectors.toList());
        Preconditions.checkArgument(illegalNames.isEmpty(), String.format("%s name %s cannot contain upper case in the catalog.", type, illegalNames));
    }

    private void validateIdentifierNameCaseInsensitive(Identifier identifier) {
        AbstractCatalog.validateCaseInsensitive(this.caseSensitive(), "Database", identifier.getDatabaseName());
        AbstractCatalog.validateCaseInsensitive(this.caseSensitive(), "Table", identifier.getObjectName());
    }

    private void validateFieldNameCaseInsensitiveInSchemaChange(List<SchemaChange> changes) {
        ArrayList<String> fieldNames = new ArrayList<String>();
        for (SchemaChange change : changes) {
            if (change instanceof SchemaChange.AddColumn) {
                SchemaChange.AddColumn addColumn = (SchemaChange.AddColumn)change;
                fieldNames.add(addColumn.fieldName());
                continue;
            }
            if (!(change instanceof SchemaChange.RenameColumn)) continue;
            SchemaChange.RenameColumn rename = (SchemaChange.RenameColumn)change;
            fieldNames.add(rename.newName());
        }
        this.validateFieldNameCaseInsensitive(fieldNames);
    }

    private void validateFieldNameCaseInsensitive(List<String> fieldNames) {
        AbstractCatalog.validateCaseInsensitive(this.caseSensitive(), "Field", fieldNames);
    }

    private void validateAutoCreateClose(Map<String, String> options) {
        Preconditions.checkArgument(!Boolean.parseBoolean(options.getOrDefault(CoreOptions.AUTO_CREATE.key(), CoreOptions.AUTO_CREATE.defaultValue().toString())), String.format("The value of %s property should be %s.", CoreOptions.AUTO_CREATE.key(), Boolean.FALSE));
    }
}

