/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.utils;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.BinaryOperator;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.paimon.Snapshot;
import org.apache.paimon.fs.FileIO;
import org.apache.paimon.fs.FileStatus;
import org.apache.paimon.fs.Path;
import org.apache.paimon.utils.BranchManager;
import org.apache.paimon.utils.FileUtils;
import org.apache.paimon.utils.Filter;
import org.apache.paimon.utils.Preconditions;

public class SnapshotManager
implements Serializable {
    private static final long serialVersionUID = 1L;
    private static final String SNAPSHOT_PREFIX = "snapshot-";
    public static final String EARLIEST = "EARLIEST";
    public static final String LATEST = "LATEST";
    private static final int READ_HINT_RETRY_NUM = 3;
    private static final int READ_HINT_RETRY_INTERVAL = 1;
    private final FileIO fileIO;
    private final Path tablePath;

    public SnapshotManager(FileIO fileIO, Path tablePath) {
        this.fileIO = fileIO;
        this.tablePath = tablePath;
    }

    public FileIO fileIO() {
        return this.fileIO;
    }

    public Path tablePath() {
        return this.tablePath;
    }

    public Path snapshotDirectory() {
        return new Path(this.tablePath + "/snapshot");
    }

    public Path snapshotPath(long snapshotId) {
        return new Path(this.tablePath + "/snapshot/" + SNAPSHOT_PREFIX + snapshotId);
    }

    public Path branchSnapshotPath(String branchName, long snapshotId) {
        return new Path(BranchManager.getBranchPath(this.tablePath, branchName) + "/snapshot/" + SNAPSHOT_PREFIX + snapshotId);
    }

    public Snapshot snapshot(long snapshotId) {
        return Snapshot.fromPath(this.fileIO, this.snapshotPath(snapshotId));
    }

    public boolean snapshotExists(long snapshotId) {
        Path path = this.snapshotPath(snapshotId);
        try {
            return this.fileIO.exists(path);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to determine if snapshot #" + snapshotId + " exists in path " + path, e);
        }
    }

    @Nullable
    public Snapshot latestSnapshot() {
        Long snapshotId = this.latestSnapshotId();
        return snapshotId == null ? null : this.snapshot(snapshotId);
    }

    @Nullable
    public Long latestSnapshotId() {
        try {
            return this.findLatest();
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to find latest snapshot id", e);
        }
    }

    @Nullable
    public Snapshot earliestSnapshot() {
        Long snapshotId = this.earliestSnapshotId();
        return snapshotId == null ? null : this.snapshot(snapshotId);
    }

    @Nullable
    public Long earliestSnapshotId() {
        try {
            return this.findEarliest();
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to find earliest snapshot id", e);
        }
    }

    @Nullable
    public Long pickOrLatest(Predicate<Snapshot> predicate) {
        Long latestId = this.latestSnapshotId();
        Long earliestId = this.earliestSnapshotId();
        if (latestId == null || earliestId == null) {
            return null;
        }
        for (long snapshotId = latestId.longValue(); snapshotId >= earliestId; --snapshotId) {
            Snapshot snapshot;
            if (!this.snapshotExists(snapshotId) || !predicate.test(snapshot = this.snapshot(snapshotId))) continue;
            return snapshot.id();
        }
        return latestId;
    }

    @Nullable
    public Long earlierThanTimeMills(long timestampMills) {
        Long earliest = this.earliestSnapshotId();
        Long latest = this.latestSnapshotId();
        if (earliest == null || latest == null) {
            return null;
        }
        for (long i = latest.longValue(); i >= earliest; --i) {
            long commitTime = this.snapshot(i).timeMillis();
            if (commitTime >= timestampMills) continue;
            return i;
        }
        return earliest - 1L;
    }

    @Nullable
    public Snapshot earlierOrEqualTimeMills(long timestampMills) {
        Long earliest = this.earliestSnapshotId();
        Long latest = this.latestSnapshotId();
        if (earliest == null || latest == null) {
            return null;
        }
        if (this.snapshot(earliest).timeMillis() > timestampMills) {
            return null;
        }
        Snapshot finnalSnapshot = null;
        while (earliest <= latest) {
            long mid = earliest + (latest - earliest) / 2L;
            Snapshot snapshot = this.snapshot(mid);
            long commitTime = snapshot.timeMillis();
            if (commitTime > timestampMills) {
                latest = mid - 1L;
                continue;
            }
            if (commitTime < timestampMills) {
                earliest = mid + 1L;
                finnalSnapshot = snapshot;
                continue;
            }
            finnalSnapshot = snapshot;
            break;
        }
        return finnalSnapshot;
    }

    public long snapshotCount() throws IOException {
        return FileUtils.listVersionedFiles(this.fileIO, this.snapshotDirectory(), SNAPSHOT_PREFIX).count();
    }

    public Iterator<Snapshot> snapshots() throws IOException {
        return FileUtils.listVersionedFiles(this.fileIO, this.snapshotDirectory(), SNAPSHOT_PREFIX).map(this::snapshot).sorted(Comparator.comparingLong(Snapshot::id)).iterator();
    }

    public List<Snapshot> safelyGetAllSnapshots() throws IOException {
        List paths = FileUtils.listVersionedFiles(this.fileIO, this.snapshotDirectory(), SNAPSHOT_PREFIX).map(this::snapshotPath).collect(Collectors.toList());
        ArrayList<Snapshot> snapshots = new ArrayList<Snapshot>();
        for (Path path : paths) {
            Snapshot.safelyFromPath(this.fileIO, path).ifPresent(snapshots::add);
        }
        return snapshots;
    }

    public List<Path> tryGetNonSnapshotFiles(Predicate<FileStatus> fileStatusFilter) {
        try {
            FileStatus[] statuses = this.fileIO.listStatus(this.snapshotDirectory());
            if (statuses == null) {
                return Collections.emptyList();
            }
            return Arrays.stream(statuses).filter(fileStatusFilter).map(FileStatus::getPath).filter(this.nonSnapshotFileFilter()).collect(Collectors.toList());
        }
        catch (IOException ignored) {
            return Collections.emptyList();
        }
    }

    private Predicate<Path> nonSnapshotFileFilter() {
        return path -> {
            String name = path.getName();
            return !name.startsWith(SNAPSHOT_PREFIX) && !name.equals(EARLIEST) && !name.equals(LATEST);
        };
    }

    public Optional<Snapshot> latestSnapshotOfUser(String user) {
        Long latestId = this.latestSnapshotId();
        if (latestId == null) {
            return Optional.empty();
        }
        long earliestId = Preconditions.checkNotNull(this.earliestSnapshotId(), "Latest snapshot id is not null, but earliest snapshot id is null. This is unexpected.");
        for (long id = latestId.longValue(); id >= earliestId; --id) {
            Snapshot snapshot = this.snapshot(id);
            if (!user.equals(snapshot.commitUser())) continue;
            return Optional.of(snapshot);
        }
        return Optional.empty();
    }

    public List<Snapshot> findSnapshotsForIdentifiers(@Nonnull String user, List<Long> identifiers) {
        if (identifiers.isEmpty()) {
            return Collections.emptyList();
        }
        Long latestId = this.latestSnapshotId();
        if (latestId == null) {
            return Collections.emptyList();
        }
        long earliestId = Preconditions.checkNotNull(this.earliestSnapshotId(), "Latest snapshot id is not null, but earliest snapshot id is null. This is unexpected.");
        long minSearchedIdentifier = (Long)identifiers.stream().min(Long::compareTo).get();
        ArrayList<Snapshot> matchedSnapshots = new ArrayList<Snapshot>();
        HashSet<Long> remainingIdentifiers = new HashSet<Long>(identifiers);
        for (long id = latestId.longValue(); id >= earliestId && !remainingIdentifiers.isEmpty(); --id) {
            Snapshot snapshot = this.snapshot(id);
            if (!user.equals(snapshot.commitUser())) continue;
            if (remainingIdentifiers.remove(snapshot.commitIdentifier())) {
                matchedSnapshots.add(snapshot);
            }
            if (snapshot.commitIdentifier() <= minSearchedIdentifier) break;
        }
        return matchedSnapshots;
    }

    @Nullable
    public Snapshot traversalSnapshotsFromLatestSafely(Filter<Snapshot> checker) {
        Long latestId = this.latestSnapshotId();
        if (latestId == null) {
            return null;
        }
        Long earliestId = this.earliestSnapshotId();
        if (earliestId == null) {
            return null;
        }
        for (long id = latestId.longValue(); id >= earliestId; --id) {
            Snapshot snapshot;
            try {
                snapshot = this.snapshot(id);
            }
            catch (Exception e) {
                Long newEarliestId = this.earliestSnapshotId();
                if (newEarliestId == null) {
                    return null;
                }
                if (id >= newEarliestId) {
                    throw e;
                }
                return null;
            }
            if (!checker.test(snapshot)) continue;
            return snapshot;
        }
        return null;
    }

    @Nullable
    private Long findLatest() throws IOException {
        long nextSnapshot;
        Path snapshotDir = this.snapshotDirectory();
        if (!this.fileIO.exists(snapshotDir)) {
            return null;
        }
        Long snapshotId = this.readHint(LATEST);
        if (snapshotId != null && !this.snapshotExists(nextSnapshot = snapshotId + 1L)) {
            return snapshotId;
        }
        return this.findByListFiles(Math::max);
    }

    @Nullable
    private Long findEarliest() throws IOException {
        Path snapshotDir = this.snapshotDirectory();
        if (!this.fileIO.exists(snapshotDir)) {
            return null;
        }
        Long snapshotId = this.readHint(EARLIEST);
        if (snapshotId != null && this.snapshotExists(snapshotId)) {
            return snapshotId;
        }
        return this.findByListFiles(Math::min);
    }

    public Long readHint(String fileName) {
        Path snapshotDir = this.snapshotDirectory();
        Path path = new Path(snapshotDir, fileName);
        int retryNumber = 0;
        while (retryNumber++ < 3) {
            try {
                return Long.parseLong(this.fileIO.readFileUtf8(path));
            }
            catch (Exception exception) {
                try {
                    TimeUnit.MILLISECONDS.sleep(1L);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException(e);
                }
            }
        }
        return null;
    }

    private Long findByListFiles(BinaryOperator<Long> reducer2) throws IOException {
        Path snapshotDir = this.snapshotDirectory();
        return FileUtils.listVersionedFiles(this.fileIO, snapshotDir, SNAPSHOT_PREFIX).reduce(reducer2).orElse(null);
    }

    public void commitLatestHint(long snapshotId) throws IOException {
        this.commitHint(snapshotId, LATEST);
    }

    public void commitEarliestHint(long snapshotId) throws IOException {
        this.commitHint(snapshotId, EARLIEST);
    }

    private void commitHint(long snapshotId, String fileName) throws IOException {
        Path snapshotDir = this.snapshotDirectory();
        Path hintFile = new Path(snapshotDir, fileName);
        this.fileIO.delete(hintFile, false);
        this.fileIO.writeFileUtf8(hintFile, String.valueOf(snapshotId));
    }
}

