/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.utils;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.paimon.Snapshot;
import org.apache.paimon.fs.FileIO;
import org.apache.paimon.fs.Path;
import org.apache.paimon.schema.SchemaManager;
import org.apache.paimon.schema.TableSchema;
import org.apache.paimon.tag.Tag;
import org.apache.paimon.utils.FileUtils;
import org.apache.paimon.utils.Preconditions;
import org.apache.paimon.utils.SnapshotManager;
import org.apache.paimon.utils.StringUtils;
import org.apache.paimon.utils.TagManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BranchManager {
    private static final Logger LOG = LoggerFactory.getLogger(BranchManager.class);
    public static final String BRANCH_PREFIX = "branch-";
    public static final String DEFAULT_MAIN_BRANCH = "main";
    private final FileIO fileIO;
    private final Path tablePath;
    private final SnapshotManager snapshotManager;
    private final TagManager tagManager;
    private final SchemaManager schemaManager;

    public BranchManager(FileIO fileIO, Path path, SnapshotManager snapshotManager, TagManager tagManager, SchemaManager schemaManager) {
        this.fileIO = fileIO;
        this.tablePath = path;
        this.snapshotManager = snapshotManager;
        this.tagManager = tagManager;
        this.schemaManager = schemaManager;
    }

    public Path branchDirectory() {
        return new Path(this.tablePath + "/branch");
    }

    public static String normalizeBranch(String branch) {
        return StringUtils.isNullOrWhitespaceOnly(branch) ? DEFAULT_MAIN_BRANCH : branch;
    }

    public static boolean isMainBranch(String branch) {
        return branch.equals(DEFAULT_MAIN_BRANCH);
    }

    public static String branchPath(Path tablePath, String branch) {
        return BranchManager.isMainBranch(branch) ? tablePath.toString() : tablePath.toString() + "/branch/" + BRANCH_PREFIX + branch;
    }

    public Path branchPath(String branchName) {
        return new Path(BranchManager.branchPath(this.tablePath, branchName));
    }

    public void createBranch(String branchName) {
        this.validateBranch(branchName);
        try {
            TableSchema latestSchema = this.schemaManager.latest().get();
            this.copySchemasToBranch(branchName, latestSchema.id());
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Exception occurs when create branch '%s' (directory in %s).", branchName, BranchManager.branchPath(this.tablePath, branchName)), e);
        }
    }

    public void createBranch(String branchName, String tagName) {
        this.validateBranch(branchName);
        Preconditions.checkArgument(this.tagManager.tagExists(tagName), "Tag name '%s' not exists.", tagName);
        Snapshot snapshot = this.tagManager.taggedSnapshot(tagName);
        try {
            this.fileIO.copyFile(this.tagManager.tagPath(tagName), this.tagManager.copyWithBranch(branchName).tagPath(tagName), true);
            this.fileIO.copyFile(this.snapshotManager.snapshotPath(snapshot.id()), this.snapshotManager.copyWithBranch(branchName).snapshotPath(snapshot.id()), true);
            this.copySchemasToBranch(branchName, snapshot.schemaId());
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Exception occurs when create branch '%s' (directory in %s).", branchName, BranchManager.branchPath(this.tablePath, branchName)), e);
        }
    }

    public void deleteBranch(String branchName) {
        Preconditions.checkArgument(this.branchExists(branchName), "Branch name '%s' doesn't exist.", branchName);
        try {
            this.fileIO.delete(this.branchPath(branchName), true);
        }
        catch (IOException e) {
            LOG.info(String.format("Deleting the branch failed due to an exception in deleting the directory %s. Please try again.", BranchManager.branchPath(this.tablePath, branchName)), (Throwable)e);
        }
    }

    public boolean fileExists(Path path) {
        try {
            return this.fileIO.exists(path);
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Failed to determine if path '%s' exists.", path), e);
        }
    }

    public void fastForward(String branchName) {
        Preconditions.checkArgument(!branchName.equals(DEFAULT_MAIN_BRANCH), "Branch name '%s' do not use in fast-forward.", branchName);
        Preconditions.checkArgument(!StringUtils.isNullOrWhitespaceOnly(branchName), "Branch name '%s' is blank.", branchName);
        Preconditions.checkArgument(this.branchExists(branchName), "Branch name '%s' doesn't exist.", branchName);
        Long earliestSnapshotId = this.snapshotManager.copyWithBranch(branchName).earliestSnapshotId();
        Snapshot earliestSnapshot = this.snapshotManager.copyWithBranch(branchName).snapshot(earliestSnapshotId);
        long earliestSchemaId = earliestSnapshot.schemaId();
        try {
            List<Path> deleteSnapshotPaths = this.snapshotManager.snapshotPaths(id -> id >= earliestSnapshotId);
            List<Path> deleteSchemaPaths = this.schemaManager.schemaPaths(id -> id >= earliestSchemaId);
            List<Path> deleteTagPaths = this.tagManager.tagPaths(path -> Tag.fromPath(this.fileIO, path).id() >= earliestSnapshotId);
            List<Path> deletePaths = Stream.of(deleteSnapshotPaths, deleteSchemaPaths, deleteTagPaths).flatMap(Collection::stream).collect(Collectors.toList());
            this.snapshotManager.deleteLatestHint();
            this.fileIO.deleteFilesQuietly(deletePaths);
            this.fileIO.copyFiles(this.snapshotManager.copyWithBranch(branchName).snapshotDirectory(), this.snapshotManager.snapshotDirectory(), true);
            this.fileIO.copyFiles(this.schemaManager.copyWithBranch(branchName).schemaDirectory(), this.schemaManager.schemaDirectory(), true);
            this.fileIO.copyFiles(this.tagManager.copyWithBranch(branchName).tagDirectory(), this.tagManager.tagDirectory(), true);
            this.snapshotManager.invalidateCache();
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Exception occurs when fast forward '%s' (directory in %s).", branchName, BranchManager.branchPath(this.tablePath, branchName)), e);
        }
    }

    public boolean branchExists(String branchName) {
        Path branchPath = this.branchPath(branchName);
        return this.fileExists(branchPath);
    }

    public List<String> branches() {
        try {
            return FileUtils.listVersionedDirectories(this.fileIO, this.branchDirectory(), BRANCH_PREFIX).map(status -> status.getPath().getName().substring(BRANCH_PREFIX.length())).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void validateBranch(String branchName) {
        Preconditions.checkArgument(!BranchManager.isMainBranch(branchName), String.format("Branch name '%s' is the default branch and cannot be used.", DEFAULT_MAIN_BRANCH));
        Preconditions.checkArgument(!StringUtils.isNullOrWhitespaceOnly(branchName), "Branch name '%s' is blank.", branchName);
        Preconditions.checkArgument(!this.branchExists(branchName), "Branch name '%s' already exists.", branchName);
        Preconditions.checkArgument(!branchName.chars().allMatch(Character::isDigit), "Branch name cannot be pure numeric string but is '%s'.", branchName);
    }

    private void copySchemasToBranch(String branchName, long schemaId) throws IOException {
        int i = 0;
        while ((long)i <= schemaId) {
            if (this.schemaManager.schemaExists(i)) {
                this.fileIO.copyFile(this.schemaManager.toSchemaPath(i), this.schemaManager.copyWithBranch(branchName).toSchemaPath(i), true);
            }
            ++i;
        }
    }
}

