/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.options;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.stream.IntStream;
import org.apache.paimon.annotation.Public;
import org.apache.paimon.utils.Preconditions;

@Public
public class MemorySize
implements Serializable,
Comparable<MemorySize> {
    private static final long serialVersionUID = 1L;
    public static final MemorySize ZERO = new MemorySize(0L);
    public static final MemorySize MAX_VALUE = new MemorySize(Long.MAX_VALUE);
    private static final List<MemoryUnit> ORDERED_UNITS = Arrays.asList(MemoryUnit.BYTES, MemoryUnit.KILO_BYTES, MemoryUnit.MEGA_BYTES, MemoryUnit.GIGA_BYTES, MemoryUnit.TERA_BYTES);
    private final long bytes;
    private transient String stringified;
    private transient String humanReadableStr;

    public MemorySize(long bytes) {
        Preconditions.checkArgument(bytes >= 0L, "bytes must be >= 0");
        this.bytes = bytes;
    }

    public static MemorySize ofMebiBytes(long mebiBytes) {
        return new MemorySize(mebiBytes << 20);
    }

    public static MemorySize ofKibiBytes(long kibiBytes) {
        return new MemorySize(kibiBytes << 10);
    }

    public long getBytes() {
        return this.bytes;
    }

    public long getKibiBytes() {
        return this.bytes >> 10;
    }

    public int getMebiBytes() {
        return (int)(this.bytes >> 20);
    }

    public long getGibiBytes() {
        return this.bytes >> 30;
    }

    public long getTebiBytes() {
        return this.bytes >> 40;
    }

    public int hashCode() {
        return (int)(this.bytes ^ this.bytes >>> 32);
    }

    public boolean equals(Object obj) {
        return obj == this || obj != null && obj.getClass() == this.getClass() && ((MemorySize)obj).bytes == this.bytes;
    }

    public String toString() {
        if (this.stringified == null) {
            this.stringified = this.formatToString();
        }
        return this.stringified;
    }

    private String formatToString() {
        MemoryUnit highestIntegerUnit = IntStream.range(0, ORDERED_UNITS.size()).sequential().filter(idx -> this.bytes % ORDERED_UNITS.get(idx).getMultiplier() != 0L).boxed().findFirst().map(idx -> {
            if (idx == 0) {
                return ORDERED_UNITS.get(0);
            }
            return ORDERED_UNITS.get(idx - 1);
        }).orElse(MemoryUnit.BYTES);
        return String.format("%d %s", this.bytes / highestIntegerUnit.getMultiplier(), highestIntegerUnit.getUnits()[1]);
    }

    public String toHumanReadableString() {
        if (this.humanReadableStr == null) {
            this.humanReadableStr = this.formatToHumanReadableString();
        }
        return this.humanReadableStr;
    }

    private String formatToHumanReadableString() {
        MemoryUnit highestUnit = IntStream.range(0, ORDERED_UNITS.size()).sequential().filter(idx -> this.bytes > ORDERED_UNITS.get(idx).getMultiplier()).boxed().max(Comparator.naturalOrder()).map(ORDERED_UNITS::get).orElse(MemoryUnit.BYTES);
        if (highestUnit == MemoryUnit.BYTES) {
            return String.format("%d %s", this.bytes, MemoryUnit.BYTES.getUnits()[1]);
        }
        double approximate = 1.0 * (double)this.bytes / (double)highestUnit.getMultiplier();
        return String.format(Locale.ROOT, "%.3f%s (%d bytes)", approximate, highestUnit.getUnits()[1], this.bytes);
    }

    @Override
    public int compareTo(MemorySize that) {
        return Long.compare(this.bytes, that.bytes);
    }

    public MemorySize add(MemorySize that) {
        return new MemorySize(Math.addExact(this.bytes, that.bytes));
    }

    public MemorySize subtract(MemorySize that) {
        return new MemorySize(Math.subtractExact(this.bytes, that.bytes));
    }

    public MemorySize multiply(double multiplier) {
        Preconditions.checkArgument(multiplier >= 0.0, "multiplier must be >= 0");
        BigDecimal product = BigDecimal.valueOf(this.bytes).multiply(BigDecimal.valueOf(multiplier));
        if (product.compareTo(BigDecimal.valueOf(Long.MAX_VALUE)) > 0) {
            throw new ArithmeticException("long overflow");
        }
        return new MemorySize(product.longValue());
    }

    public MemorySize divide(long by) {
        Preconditions.checkArgument(by >= 0L, "divisor must be >= 0");
        return new MemorySize(this.bytes / by);
    }

    public static MemorySize parse(String text) throws IllegalArgumentException {
        return new MemorySize(MemorySize.parseBytes(text));
    }

    public static MemorySize parse(String text, MemoryUnit defaultUnit) throws IllegalArgumentException {
        if (!MemoryUnit.hasUnit(text)) {
            return MemorySize.parse(text + defaultUnit.getUnits()[0]);
        }
        return MemorySize.parse(text);
    }

    public static long parseBytes(String text) throws IllegalArgumentException {
        long value;
        char current;
        int pos;
        Preconditions.checkNotNull(text, "text");
        String trimmed = text.trim();
        Preconditions.checkArgument(!trimmed.isEmpty(), "argument is an empty- or whitespace-only string");
        int len = trimmed.length();
        for (pos = 0; pos < len && (current = trimmed.charAt(pos)) >= '0' && current <= '9'; ++pos) {
        }
        String number = trimmed.substring(0, pos);
        String unit = trimmed.substring(pos).trim().toLowerCase(Locale.US);
        if (number.isEmpty()) {
            throw new NumberFormatException("text does not start with a number");
        }
        try {
            value = Long.parseLong(number);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("The value '" + number + "' cannot be re represented as 64bit number (numeric overflow).");
        }
        long multiplier = MemorySize.parseUnit(unit).map(MemoryUnit::getMultiplier).orElse(1L);
        long result = value * multiplier;
        if (result / multiplier != value) {
            throw new IllegalArgumentException("The value '" + text + "' cannot be re represented as 64bit number of bytes (numeric overflow).");
        }
        return result;
    }

    private static Optional<MemoryUnit> parseUnit(String unit) {
        if (MemorySize.matchesAny(unit, MemoryUnit.BYTES)) {
            return Optional.of(MemoryUnit.BYTES);
        }
        if (MemorySize.matchesAny(unit, MemoryUnit.KILO_BYTES)) {
            return Optional.of(MemoryUnit.KILO_BYTES);
        }
        if (MemorySize.matchesAny(unit, MemoryUnit.MEGA_BYTES)) {
            return Optional.of(MemoryUnit.MEGA_BYTES);
        }
        if (MemorySize.matchesAny(unit, MemoryUnit.GIGA_BYTES)) {
            return Optional.of(MemoryUnit.GIGA_BYTES);
        }
        if (MemorySize.matchesAny(unit, MemoryUnit.TERA_BYTES)) {
            return Optional.of(MemoryUnit.TERA_BYTES);
        }
        if (!unit.isEmpty()) {
            throw new IllegalArgumentException("Memory size unit '" + unit + "' does not match any of the recognized units: " + MemoryUnit.getAllUnits());
        }
        return Optional.empty();
    }

    private static boolean matchesAny(String str, MemoryUnit unit) {
        for (String s : unit.getUnits()) {
            if (!s.equals(str)) continue;
            return true;
        }
        return false;
    }

    public static enum MemoryUnit {
        BYTES(new String[]{"b", "bytes"}, 1L),
        KILO_BYTES(new String[]{"k", "kb", "kibibytes"}, 1024L),
        MEGA_BYTES(new String[]{"m", "mb", "mebibytes"}, 0x100000L),
        GIGA_BYTES(new String[]{"g", "gb", "gibibytes"}, 0x40000000L),
        TERA_BYTES(new String[]{"t", "tb", "tebibytes"}, 0x10000000000L);

        private final String[] units;
        private final long multiplier;

        private MemoryUnit(String[] units, long multiplier) {
            this.units = units;
            this.multiplier = multiplier;
        }

        public String[] getUnits() {
            return this.units;
        }

        public long getMultiplier() {
            return this.multiplier;
        }

        public static String getAllUnits() {
            return MemoryUnit.concatenateUnits(BYTES.getUnits(), KILO_BYTES.getUnits(), MEGA_BYTES.getUnits(), GIGA_BYTES.getUnits(), TERA_BYTES.getUnits());
        }

        public static boolean hasUnit(String text) {
            char current;
            int pos;
            Preconditions.checkNotNull(text, "text");
            String trimmed = text.trim();
            Preconditions.checkArgument(!trimmed.isEmpty(), "argument is an empty- or whitespace-only string");
            int len = trimmed.length();
            for (pos = 0; pos < len && (current = trimmed.charAt(pos)) >= '0' && current <= '9'; ++pos) {
            }
            String unit = trimmed.substring(pos).trim().toLowerCase(Locale.US);
            return unit.length() > 0;
        }

        private static String concatenateUnits(String[] ... allUnits) {
            StringBuilder builder = new StringBuilder(128);
            for (String[] units : allUnits) {
                builder.append('(');
                for (String unit : units) {
                    builder.append(unit);
                    builder.append(" | ");
                }
                builder.setLength(builder.length() - 3);
                builder.append(") / ");
            }
            builder.setLength(builder.length() - 3);
            return builder.toString();
        }
    }
}

