/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.utils;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.paimon.CoreOptions;
import org.apache.paimon.Snapshot;
import org.apache.paimon.annotation.VisibleForTesting;
import org.apache.paimon.fs.FileIO;
import org.apache.paimon.fs.FileStatus;
import org.apache.paimon.fs.Path;
import org.apache.paimon.iceberg.IcebergCommitCallback;
import org.apache.paimon.manifest.ExpireFileEntry;
import org.apache.paimon.operation.TagDeletion;
import org.apache.paimon.table.sink.TagCallback;
import org.apache.paimon.tag.Tag;
import org.apache.paimon.tag.TagPeriodHandler;
import org.apache.paimon.tag.TagTimeExtractor;
import org.apache.paimon.utils.BranchManager;
import org.apache.paimon.utils.FileUtils;
import org.apache.paimon.utils.IOUtils;
import org.apache.paimon.utils.Pair;
import org.apache.paimon.utils.Preconditions;
import org.apache.paimon.utils.SnapshotManager;
import org.apache.paimon.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TagManager {
    private static final Logger LOG = LoggerFactory.getLogger(TagManager.class);
    private static final String TAG_PREFIX = "tag-";
    private final FileIO fileIO;
    private final Path tablePath;
    private final String branch;
    @Nullable
    private final TagPeriodHandler tagPeriodHandler;

    public TagManager(FileIO fileIO, Path tablePath) {
        this(fileIO, tablePath, "main", (TagPeriodHandler)null);
    }

    public TagManager(FileIO fileIO, Path tablePath, String branch) {
        this(fileIO, tablePath, branch, (TagPeriodHandler)null);
    }

    public TagManager(FileIO fileIO, Path tablePath, String branch, CoreOptions options) {
        this(fileIO, tablePath, branch, TagManager.createIfNecessary(options));
    }

    @Nullable
    private static TagPeriodHandler createIfNecessary(CoreOptions options) {
        return TagTimeExtractor.createForAutoTag(options) == null ? null : TagPeriodHandler.create(options);
    }

    private TagManager(FileIO fileIO, Path tablePath, String branch, @Nullable TagPeriodHandler tagPeriodHandler) {
        this.fileIO = fileIO;
        this.tablePath = tablePath;
        this.branch = BranchManager.normalizeBranch(branch);
        this.tagPeriodHandler = tagPeriodHandler;
    }

    public TagManager copyWithBranch(String branchName) {
        return new TagManager(this.fileIO, this.tablePath, branchName, (TagPeriodHandler)null);
    }

    public Path tagDirectory() {
        return new Path(BranchManager.branchPath(this.tablePath, this.branch) + "/tag");
    }

    public Path tagPath(String tagName) {
        return new Path(BranchManager.branchPath(this.tablePath, this.branch) + "/tag/" + TAG_PREFIX + tagName);
    }

    public List<Path> tagPaths(Predicate<Path> predicate) throws IOException {
        return FileUtils.listVersionedFileStatus(this.fileIO, this.tagDirectory(), TAG_PREFIX).map(FileStatus::getPath).filter(predicate).collect(Collectors.toList());
    }

    public void createTag(Snapshot snapshot, String tagName, Duration timeRetained, List<TagCallback> callbacks, boolean ignoreIfExists) {
        Preconditions.checkArgument(!StringUtils.isNullOrWhitespaceOnly(tagName), "Tag name shouldn't be blank.");
        if (this.tagExists(tagName)) {
            Preconditions.checkArgument(ignoreIfExists, "Tag '%s' already exists.", tagName);
            return;
        }
        this.createOrReplaceTag(snapshot, tagName, timeRetained, callbacks);
    }

    public void replaceTag(Snapshot snapshot, String tagName, Duration timeRetained, List<TagCallback> callbacks) {
        Preconditions.checkArgument(!StringUtils.isNullOrWhitespaceOnly(tagName), "Tag name shouldn't be blank.");
        Preconditions.checkArgument(this.tagExists(tagName), "Tag '%s' doesn't exist.", tagName);
        this.createOrReplaceTag(snapshot, tagName, timeRetained, callbacks.stream().filter(callback -> callback instanceof IcebergCommitCallback).collect(Collectors.toList()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void createOrReplaceTag(Snapshot snapshot, String tagName, @Nullable Duration timeRetained, @Nullable List<TagCallback> callbacks) {
        this.validateNoAutoTag(tagName, snapshot);
        String content = timeRetained != null ? Tag.fromSnapshotAndTagTtl(snapshot, timeRetained, LocalDateTime.now()).toJson() : snapshot.toJson();
        Path tagPath = this.tagPath(tagName);
        try {
            this.fileIO.overwriteFileUtf8(tagPath, content);
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Exception occurs when committing tag '%s' (path %s). Cannot clean up because we can't determine the success.", tagName, tagPath), e);
        }
        if (callbacks != null) {
            try {
                callbacks.forEach(callback -> callback.notifyCreation(tagName, snapshot.id()));
            }
            finally {
                for (TagCallback tagCallback : callbacks) {
                    IOUtils.closeQuietly(tagCallback);
                }
            }
        }
    }

    public void renameTag(String tagName, String targetTagName) {
        Preconditions.checkArgument(!StringUtils.isNullOrWhitespaceOnly(tagName), "Original tag name shouldn't be blank.");
        Preconditions.checkArgument(this.tagExists(tagName), "Tag '%s' doesn't exist.", tagName);
        Preconditions.checkArgument(!StringUtils.isNullOrWhitespaceOnly(targetTagName), "New tag name shouldn't be blank.");
        Preconditions.checkArgument(!this.tagExists(targetTagName), "Tag '%s' already exists.", tagName);
        try {
            this.fileIO.rename(this.tagPath(tagName), this.tagPath(targetTagName));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public void deleteAllTagsOfOneSnapshot(List<String> tagNames, TagDeletion tagDeletion, SnapshotManager snapshotManager) {
        Snapshot taggedSnapshot = this.getOrThrow(tagNames.get(0)).trimToSnapshot();
        if (snapshotManager.snapshotExists(taggedSnapshot.id())) {
            tagNames.forEach(tagName -> this.fileIO.deleteQuietly(this.tagPath((String)tagName)));
            return;
        }
        List<Snapshot> taggedSnapshots = this.taggedSnapshots();
        tagNames.forEach(tagName -> this.fileIO.deleteQuietly(this.tagPath((String)tagName)));
        this.doClean(taggedSnapshot, taggedSnapshots, snapshotManager, tagDeletion);
    }

    public void deleteTag(String tagName, TagDeletion tagDeletion, SnapshotManager snapshotManager, List<TagCallback> callbacks) {
        Preconditions.checkArgument(!StringUtils.isNullOrWhitespaceOnly(tagName), "Tag name shouldn't be blank.");
        Optional<Tag> tag = this.get(tagName);
        if (!tag.isPresent()) {
            LOG.warn("Tag '{}' doesn't exist.", (Object)tagName);
            return;
        }
        Snapshot taggedSnapshot = tag.get().trimToSnapshot();
        if (snapshotManager.copyWithBranch(this.branch).snapshotExists(taggedSnapshot.id())) {
            this.deleteTagMetaFile(tagName, callbacks);
            return;
        }
        SortedMap<Snapshot, List<String>> tags = this.tags();
        this.deleteTagMetaFile(tagName, callbacks);
        if (((List)tags.get(taggedSnapshot)).size() > 1) {
            return;
        }
        ArrayList<Snapshot> taggedSnapshots = new ArrayList<Snapshot>(tags.keySet());
        this.doClean(taggedSnapshot, taggedSnapshots, snapshotManager, tagDeletion);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void deleteTagMetaFile(String tagName, List<TagCallback> callbacks) {
        this.fileIO.deleteQuietly(this.tagPath(tagName));
        try {
            callbacks.forEach(callback -> callback.notifyDeletion(tagName));
        }
        finally {
            for (TagCallback tagCallback : callbacks) {
                IOUtils.closeQuietly(tagCallback);
            }
        }
    }

    private void doClean(Snapshot taggedSnapshot, List<Snapshot> taggedSnapshots, SnapshotManager snapshotManager, TagDeletion tagDeletion) {
        ArrayList<Snapshot> skippedSnapshots = new ArrayList<Snapshot>();
        int index = this.findIndex(taggedSnapshot, taggedSnapshots);
        if (index - 1 >= 0) {
            skippedSnapshots.add(taggedSnapshots.get(index - 1));
        }
        Snapshot right = snapshotManager.copyWithBranch(this.branch).earliestSnapshot();
        if (index + 1 < taggedSnapshots.size()) {
            Snapshot rightTag = taggedSnapshots.get(index + 1);
            right = right.id() < rightTag.id() ? right : rightTag;
        }
        skippedSnapshots.add(right);
        Predicate<ExpireFileEntry> dataFileSkipper = null;
        boolean success = true;
        try {
            dataFileSkipper = tagDeletion.dataFileSkipper(skippedSnapshots);
        }
        catch (Exception e) {
            LOG.info(String.format("Skip cleaning data files for tag of snapshot %s due to failed to build skipping set.", taggedSnapshot.id()), (Throwable)e);
            success = false;
        }
        if (success) {
            tagDeletion.cleanUnusedDataFiles(taggedSnapshot, dataFileSkipper);
            tagDeletion.cleanEmptyDirectories();
        }
        success = true;
        Set<String> manifestSkippingSet = null;
        try {
            manifestSkippingSet = tagDeletion.manifestSkippingSet(skippedSnapshots);
        }
        catch (Exception e) {
            LOG.info(String.format("Skip cleaning manifest files for tag of snapshot %s due to failed to build skipping set.", taggedSnapshot.id()), (Throwable)e);
            success = false;
        }
        if (success) {
            tagDeletion.cleanUnusedManifests(taggedSnapshot, manifestSkippingSet);
        }
    }

    public boolean tagExists(String tagName) {
        Path path = this.tagPath(tagName);
        try {
            return this.fileIO.exists(path);
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Failed to determine if tag '%s' exists in path %s.", tagName, path), e);
        }
    }

    public Optional<Tag> get(String tagName) {
        Preconditions.checkArgument(!StringUtils.isNullOrWhitespaceOnly(tagName), "Tag name shouldn't be blank.");
        try {
            return Optional.of(Tag.tryFromPath(this.fileIO, this.tagPath(tagName)));
        }
        catch (FileNotFoundException e) {
            return Optional.empty();
        }
    }

    public Tag getOrThrow(String tagName) {
        return this.get(tagName).orElseThrow(() -> new IllegalArgumentException("Tag '" + tagName + "' doesn't exist."));
    }

    public long tagCount() {
        try {
            return FileUtils.listVersionedFileStatus(this.fileIO, this.tagDirectory(), TAG_PREFIX).count();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public List<Snapshot> taggedSnapshots() {
        return new ArrayList<Snapshot>(this.tags().keySet());
    }

    public SortedMap<Snapshot, List<String>> tags() {
        return this.tags(tagName -> true);
    }

    public SortedMap<Snapshot, List<String>> tags(Predicate<String> filter) {
        TreeMap<Snapshot, List<String>> tags = new TreeMap<Snapshot, List<String>>(Comparator.comparingLong(Snapshot::id));
        try {
            List<Path> paths = this.tagPaths(path -> true);
            for (Path path2 : paths) {
                String tagName = path2.getName().substring(TAG_PREFIX.length());
                if (!filter.test(tagName)) continue;
                try {
                    Snapshot snapshot = Tag.tryFromPath(this.fileIO, path2).trimToSnapshot();
                    tags.computeIfAbsent(snapshot, s -> new ArrayList()).add(tagName);
                }
                catch (FileNotFoundException fileNotFoundException) {}
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return tags;
    }

    public List<Pair<Tag, String>> tagObjects() {
        try {
            List<Path> paths = this.tagPaths(path -> true);
            ArrayList<Pair<Tag, String>> tags = new ArrayList<Pair<Tag, String>>();
            for (Path path2 : paths) {
                String tagName = path2.getName().substring(TAG_PREFIX.length());
                try {
                    tags.add(Pair.of(Tag.tryFromPath(this.fileIO, path2), tagName));
                }
                catch (FileNotFoundException fileNotFoundException) {}
            }
            return tags;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public List<String> sortTagsOfOneSnapshot(List<String> tagNames) {
        return tagNames.stream().map(name -> {
            try {
                return this.fileIO.getFileStatus(this.tagPath((String)name));
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }).sorted(Comparator.comparingLong(FileStatus::getModificationTime)).map(fileStatus -> fileStatus.getPath().getName().substring(TAG_PREFIX.length())).collect(Collectors.toList());
    }

    @VisibleForTesting
    public List<String> allTagNames() {
        return this.tags().values().stream().flatMap(Collection::stream).collect(Collectors.toList());
    }

    private int findIndex(Snapshot taggedSnapshot, List<Snapshot> taggedSnapshots) {
        for (int i = 0; i < taggedSnapshots.size(); ++i) {
            if (taggedSnapshot.id() != taggedSnapshots.get(i).id()) continue;
            return i;
        }
        throw new RuntimeException(String.format("Didn't find tag with snapshot id '%s'. This is unexpected.", taggedSnapshot.id()));
    }

    private void validateNoAutoTag(String tagName, Snapshot snapshot) {
        if (this.tagPeriodHandler == null || !this.tagPeriodHandler.isAutoTag(tagName)) {
            return;
        }
        List autoTags = (List)this.tags(this.tagPeriodHandler::isAutoTag).get(snapshot);
        if (autoTags != null) {
            throw new RuntimeException(String.format("Snapshot %s is already auto-tagged with %s.", snapshot.id(), autoTags));
        }
    }
}

