/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.catalog;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.paimon.CoreOptions;
import org.apache.paimon.TableType;
import org.apache.paimon.catalog.Catalog;
import org.apache.paimon.catalog.CatalogLockContext;
import org.apache.paimon.catalog.CatalogLockFactory;
import org.apache.paimon.catalog.CatalogUtils;
import org.apache.paimon.catalog.Database;
import org.apache.paimon.catalog.Identifier;
import org.apache.paimon.catalog.PropertyChange;
import org.apache.paimon.factories.FactoryUtil;
import org.apache.paimon.fs.FileIO;
import org.apache.paimon.fs.FileStatus;
import org.apache.paimon.fs.Path;
import org.apache.paimon.metastore.MetastoreClient;
import org.apache.paimon.operation.FileStoreCommit;
import org.apache.paimon.operation.Lock;
import org.apache.paimon.options.CatalogOptions;
import org.apache.paimon.options.Options;
import org.apache.paimon.partition.Partition;
import org.apache.paimon.schema.Schema;
import org.apache.paimon.schema.SchemaChange;
import org.apache.paimon.schema.SchemaManager;
import org.apache.paimon.schema.TableSchema;
import org.apache.paimon.table.CatalogEnvironment;
import org.apache.paimon.table.FileStoreTable;
import org.apache.paimon.table.FileStoreTableFactory;
import org.apache.paimon.table.Table;
import org.apache.paimon.table.object.ObjectTable;
import org.apache.paimon.table.system.SystemTableLoader;
import org.apache.paimon.types.DataField;
import org.apache.paimon.types.RowType;
import org.apache.paimon.utils.Preconditions;

public abstract class AbstractCatalog
implements Catalog {
    protected final FileIO fileIO;
    protected final Map<String, String> tableDefaultOptions;
    protected final Options catalogOptions;

    protected AbstractCatalog(FileIO fileIO) {
        this.fileIO = fileIO;
        this.tableDefaultOptions = new HashMap<String, String>();
        this.catalogOptions = new Options();
    }

    protected AbstractCatalog(FileIO fileIO, Options options) {
        this.fileIO = fileIO;
        this.tableDefaultOptions = CatalogUtils.tableDefaultOptions(options.toMap());
        this.catalogOptions = options;
    }

    @Override
    public Map<String, String> options() {
        return this.catalogOptions.toMap();
    }

    @Override
    public FileIO fileIO() {
        return this.fileIO;
    }

    public Optional<CatalogLockFactory> lockFactory() {
        if (!this.lockEnabled()) {
            return Optional.empty();
        }
        String lock = this.catalogOptions.get(CatalogOptions.LOCK_TYPE);
        if (lock == null) {
            return this.defaultLockFactory();
        }
        return Optional.of(FactoryUtil.discoverFactory(AbstractCatalog.class.getClassLoader(), CatalogLockFactory.class, lock));
    }

    public Optional<CatalogLockFactory> defaultLockFactory() {
        return Optional.empty();
    }

    public Optional<CatalogLockContext> lockContext() {
        return Optional.of(CatalogLockContext.fromOptions(this.catalogOptions));
    }

    protected boolean lockEnabled() {
        return this.catalogOptions.getOptional(CatalogOptions.LOCK_ENABLED).orElse(this.fileIO.isObjectStore());
    }

    protected boolean allowCustomTablePath() {
        return false;
    }

    @Override
    public void createDatabase(String name, boolean ignoreIfExists, Map<String, String> properties) throws Catalog.DatabaseAlreadyExistException {
        this.checkNotSystemDatabase(name);
        try {
            this.getDatabase(name);
            if (ignoreIfExists) {
                return;
            }
            throw new Catalog.DatabaseAlreadyExistException(name);
        }
        catch (Catalog.DatabaseNotExistException databaseNotExistException) {
            this.createDatabaseImpl(name, properties);
            return;
        }
    }

    @Override
    public Database getDatabase(String name) throws Catalog.DatabaseNotExistException {
        if (AbstractCatalog.isSystemDatabase(name)) {
            return Database.of(name);
        }
        return this.getDatabaseImpl(name);
    }

    protected abstract Database getDatabaseImpl(String var1) throws Catalog.DatabaseNotExistException;

    @Override
    public void createPartition(Identifier identifier, Map<String, String> partitionSpec) throws Catalog.TableNotExistException {
        Identifier tableIdentifier = Identifier.create(identifier.getDatabaseName(), identifier.getTableName());
        FileStoreTable table = (FileStoreTable)this.getTable(tableIdentifier);
        if (table.partitionKeys().isEmpty() || !table.coreOptions().partitionedTableInMetastore()) {
            throw new UnsupportedOperationException("The table is not partitioned table in metastore.");
        }
        MetastoreClient.Factory metastoreFactory = table.catalogEnvironment().metastoreClientFactory();
        if (metastoreFactory == null) {
            throw new UnsupportedOperationException("The catalog must have metastore to create partition.");
        }
        try (MetastoreClient client = metastoreFactory.create();){
            client.addPartition(new LinkedHashMap<String, String>(partitionSpec));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void dropPartition(Identifier identifier, Map<String, String> partitionSpec) throws Catalog.TableNotExistException {
        AbstractCatalog.checkNotSystemTable(identifier, "dropPartition");
        Table table = this.getTable(identifier);
        FileStoreTable fileStoreTable = (FileStoreTable)table;
        try (FileStoreCommit commit = fileStoreTable.store().newCommit(CoreOptions.createCommitUser(fileStoreTable.coreOptions().toConfiguration()));){
            commit.dropPartitions(Collections.singletonList(partitionSpec), Long.MAX_VALUE);
        }
    }

    @Override
    public List<Partition> listPartitions(Identifier identifier) throws Catalog.TableNotExistException {
        return CatalogUtils.listPartitionsFromFileSystem(this.getTable(identifier));
    }

    protected abstract void createDatabaseImpl(String var1, Map<String, String> var2);

    @Override
    public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade) throws Catalog.DatabaseNotExistException, Catalog.DatabaseNotEmptyException {
        this.checkNotSystemDatabase(name);
        try {
            this.getDatabase(name);
        }
        catch (Catalog.DatabaseNotExistException e) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.DatabaseNotExistException(name);
        }
        if (!cascade && !this.listTables(name).isEmpty()) {
            throw new Catalog.DatabaseNotEmptyException(name);
        }
        this.dropDatabaseImpl(name);
    }

    protected abstract void dropDatabaseImpl(String var1);

    @Override
    public void alterDatabase(String name, List<PropertyChange> changes, boolean ignoreIfNotExists) throws Catalog.DatabaseNotExistException {
        this.checkNotSystemDatabase(name);
        try {
            if (changes == null || changes.isEmpty()) {
                return;
            }
            this.alterDatabaseImpl(name, changes);
        }
        catch (Catalog.DatabaseNotExistException e) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.DatabaseNotExistException(name);
        }
    }

    protected abstract void alterDatabaseImpl(String var1, List<PropertyChange> var2) throws Catalog.DatabaseNotExistException;

    @Override
    public List<String> listTables(String databaseName) throws Catalog.DatabaseNotExistException {
        if (AbstractCatalog.isSystemDatabase(databaseName)) {
            return SystemTableLoader.loadGlobalTableNames();
        }
        this.getDatabase(databaseName);
        return this.listTablesImpl(databaseName).stream().sorted().collect(Collectors.toList());
    }

    protected abstract List<String> listTablesImpl(String var1);

    @Override
    public void dropTable(Identifier identifier, boolean ignoreIfNotExists) throws Catalog.TableNotExistException {
        AbstractCatalog.checkNotBranch(identifier, "dropTable");
        AbstractCatalog.checkNotSystemTable(identifier, "dropTable");
        try {
            this.getTable(identifier);
        }
        catch (Catalog.TableNotExistException e) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(identifier);
        }
        this.dropTableImpl(identifier);
    }

    protected abstract void dropTableImpl(Identifier var1);

    @Override
    public void createTable(Identifier identifier, Schema schema, boolean ignoreIfExists) throws Catalog.TableAlreadyExistException, Catalog.DatabaseNotExistException {
        AbstractCatalog.checkNotBranch(identifier, "createTable");
        AbstractCatalog.checkNotSystemTable(identifier, "createTable");
        this.validateAutoCreateClose(schema.options());
        this.validateCustomTablePath(schema.options());
        this.getDatabase(identifier.getDatabaseName());
        try {
            this.getTable(identifier);
            if (ignoreIfExists) {
                return;
            }
            throw new Catalog.TableAlreadyExistException(identifier);
        }
        catch (Catalog.TableNotExistException tableNotExistException) {
            this.copyTableDefaultOptions(schema.options());
            switch (Options.fromMap(schema.options()).get(CoreOptions.TYPE)) {
                case TABLE: 
                case MATERIALIZED_TABLE: {
                    this.createTableImpl(identifier, schema);
                    break;
                }
                case OBJECT_TABLE: {
                    this.createObjectTable(identifier, schema);
                    break;
                }
                case FORMAT_TABLE: {
                    this.createFormatTable(identifier, schema);
                }
            }
            return;
        }
    }

    private void createObjectTable(Identifier identifier, Schema schema) {
        RowType rowType = schema.rowType();
        Preconditions.checkArgument(rowType.getFields().isEmpty() || new HashSet<DataField>(ObjectTable.SCHEMA.getFields()).containsAll(rowType.getFields()), "Schema of Object Table can be empty or %s, but is %s.", ObjectTable.SCHEMA, rowType);
        Preconditions.checkArgument(schema.options().containsKey(CoreOptions.OBJECT_LOCATION.key()), "Object table should have object-location option.");
        this.createTableImpl(identifier, schema.copy(ObjectTable.SCHEMA));
    }

    protected abstract void createTableImpl(Identifier var1, Schema var2);

    @Override
    public void renameTable(Identifier fromTable, Identifier toTable, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.TableAlreadyExistException {
        AbstractCatalog.checkNotBranch(fromTable, "renameTable");
        AbstractCatalog.checkNotBranch(toTable, "renameTable");
        AbstractCatalog.checkNotSystemTable(fromTable, "renameTable");
        AbstractCatalog.checkNotSystemTable(toTable, "renameTable");
        try {
            this.getTable(fromTable);
        }
        catch (Catalog.TableNotExistException e) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(fromTable);
        }
        try {
            this.getTable(toTable);
            throw new Catalog.TableAlreadyExistException(toTable);
        }
        catch (Catalog.TableNotExistException tableNotExistException) {
            this.renameTableImpl(fromTable, toTable);
            return;
        }
    }

    protected abstract void renameTableImpl(Identifier var1, Identifier var2);

    @Override
    public void alterTable(Identifier identifier, List<SchemaChange> changes, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.ColumnAlreadyExistException, Catalog.ColumnNotExistException {
        AbstractCatalog.checkNotSystemTable(identifier, "alterTable");
        try {
            this.getTable(identifier);
        }
        catch (Catalog.TableNotExistException e) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(identifier);
        }
        this.alterTableImpl(identifier, changes);
    }

    protected abstract void alterTableImpl(Identifier var1, List<SchemaChange> var2) throws Catalog.TableNotExistException, Catalog.ColumnAlreadyExistException, Catalog.ColumnNotExistException;

    @Override
    public Table getTable(Identifier identifier) throws Catalog.TableNotExistException {
        if (AbstractCatalog.isSystemDatabase(identifier.getDatabaseName())) {
            String tableName = identifier.getTableName();
            Table table = SystemTableLoader.loadGlobal(tableName, this.fileIO, this::allTablePaths, this.catalogOptions);
            if (table == null) {
                throw new Catalog.TableNotExistException(identifier);
            }
            return table;
        }
        if (identifier.isSystemTable()) {
            Table originTable = this.getDataOrFormatTable(new Identifier(identifier.getDatabaseName(), identifier.getTableName(), identifier.getBranchName(), null));
            if (!(originTable instanceof FileStoreTable)) {
                throw new UnsupportedOperationException(String.format("Only data table support system tables, but this table %s is %s.", identifier, originTable.getClass()));
            }
            Table table = SystemTableLoader.load(Preconditions.checkNotNull(identifier.getSystemTableName()), (FileStoreTable)originTable);
            if (table == null) {
                throw new Catalog.TableNotExistException(identifier);
            }
            return table;
        }
        return this.getDataOrFormatTable(identifier);
    }

    protected Table getDataOrFormatTable(Identifier identifier) throws Catalog.TableNotExistException {
        Preconditions.checkArgument(identifier.getSystemTableName() == null);
        TableMeta tableMeta = this.getDataTableMeta(identifier);
        FileStoreTable table = FileStoreTableFactory.create(this.fileIO, this.getTableLocation(identifier), tableMeta.schema, new CatalogEnvironment(identifier, tableMeta.uuid, Lock.factory(this.lockFactory().orElse(null), this.lockContext().orElse(null), identifier), this.metastoreClientFactory(identifier).orElse(null)));
        CoreOptions options = table.coreOptions();
        if (options.type() == TableType.OBJECT_TABLE) {
            String objectLocation = options.objectLocation();
            Preconditions.checkNotNull(objectLocation, "Object location should not be null for object table.");
            table = ObjectTable.builder().underlyingTable(table).objectLocation(objectLocation).objectFileIO(this.objectFileIO(objectLocation)).build();
        }
        return table;
    }

    protected FileIO objectFileIO(String objectLocation) {
        return this.fileIO;
    }

    public void createFormatTable(Identifier identifier, Schema schema) {
        throw new UnsupportedOperationException(this.getClass().getName() + " currently does not support format table");
    }

    public Path newDatabasePath(String database) {
        return AbstractCatalog.newDatabasePath(this.warehouse(), database);
    }

    public Map<String, Map<String, Path>> allTablePaths() {
        try {
            HashMap<String, Map<String, Path>> allPaths = new HashMap<String, Map<String, Path>>();
            for (String database : this.listDatabases()) {
                Map tableMap = allPaths.computeIfAbsent(database, d -> new HashMap());
                for (String table : this.listTables(database)) {
                    tableMap.put(table, this.getTableLocation(Identifier.create(database, table)));
                }
            }
            return allPaths;
        }
        catch (Catalog.DatabaseNotExistException e) {
            throw new RuntimeException("Database is deleted while listing", e);
        }
    }

    protected TableMeta getDataTableMeta(Identifier identifier) throws Catalog.TableNotExistException {
        return new TableMeta(this.getDataTableSchema(identifier), null);
    }

    protected abstract TableSchema getDataTableSchema(Identifier var1) throws Catalog.TableNotExistException;

    public Optional<MetastoreClient.Factory> metastoreClientFactory(Identifier identifier) {
        return Optional.empty();
    }

    public Path getTableLocation(Identifier identifier) {
        return new Path(this.newDatabasePath(identifier.getDatabaseName()), identifier.getTableName());
    }

    protected static void checkNotBranch(Identifier identifier, String method) {
        if (identifier.getBranchName() != null) {
            throw new IllegalArgumentException(String.format("Cannot '%s' for branch table '%s', please modify the table with the default branch.", method, identifier));
        }
    }

    protected void assertMainBranch(Identifier identifier) {
        if (identifier.getBranchName() != null && !"main".equals(identifier.getBranchName())) {
            throw new UnsupportedOperationException(this.getClass().getName() + " currently does not support table branches");
        }
    }

    protected static boolean isTableInSystemDatabase(Identifier identifier) {
        return AbstractCatalog.isSystemDatabase(identifier.getDatabaseName()) || identifier.isSystemTable();
    }

    protected static void checkNotSystemTable(Identifier identifier, String method) {
        if (AbstractCatalog.isTableInSystemDatabase(identifier)) {
            throw new IllegalArgumentException(String.format("Cannot '%s' for system table '%s', please use data table.", method, identifier));
        }
    }

    private void copyTableDefaultOptions(Map<String, String> options) {
        this.tableDefaultOptions.forEach(options::putIfAbsent);
    }

    public static Path newTableLocation(String warehouse, Identifier identifier) {
        AbstractCatalog.checkNotBranch(identifier, "newTableLocation");
        AbstractCatalog.checkNotSystemTable(identifier, "newTableLocation");
        return new Path(AbstractCatalog.newDatabasePath(warehouse, identifier.getDatabaseName()), identifier.getTableName());
    }

    public static Path newDatabasePath(String warehouse, String database) {
        return new Path(warehouse, database + ".db");
    }

    public static boolean isSystemDatabase(String database) {
        return "sys".equals(database);
    }

    protected void checkNotSystemDatabase(String database) {
        if (AbstractCatalog.isSystemDatabase(database)) {
            throw new Catalog.ProcessSystemDatabaseException();
        }
    }

    private void validateAutoCreateClose(Map<String, String> options) {
        Preconditions.checkArgument(!Boolean.parseBoolean(options.getOrDefault(CoreOptions.AUTO_CREATE.key(), CoreOptions.AUTO_CREATE.defaultValue().toString())), String.format("The value of %s property should be %s.", CoreOptions.AUTO_CREATE.key(), Boolean.FALSE));
    }

    private void validateCustomTablePath(Map<String, String> options) {
        if (!this.allowCustomTablePath() && options.containsKey(CoreOptions.PATH.key())) {
            throw new UnsupportedOperationException(String.format("The current catalog %s does not support specifying the table path when creating a table.", this.getClass().getSimpleName()));
        }
    }

    protected List<String> listDatabasesInFileSystem(Path warehouse) throws IOException {
        ArrayList<String> databases = new ArrayList<String>();
        for (FileStatus status : this.fileIO.listDirectories(warehouse)) {
            Path path = status.getPath();
            if (!status.isDir() || !path.getName().endsWith(".db")) continue;
            String fileName = path.getName();
            databases.add(fileName.substring(0, fileName.length() - ".db".length()));
        }
        return databases;
    }

    protected List<String> listTablesInFileSystem(Path databasePath) throws IOException {
        ArrayList<String> tables = new ArrayList<String>();
        for (FileStatus status : this.fileIO.listDirectories(databasePath)) {
            if (!status.isDir() || !this.tableExistsInFileSystem(status.getPath(), "main")) continue;
            tables.add(status.getPath().getName());
        }
        return tables;
    }

    protected boolean tableExistsInFileSystem(Path tablePath, String branchName) {
        SchemaManager schemaManager = new SchemaManager(this.fileIO, tablePath, branchName);
        boolean schemaZeroExists = schemaManager.schemaExists(0L);
        if (schemaZeroExists) {
            return true;
        }
        return !schemaManager.listAllIds().isEmpty();
    }

    public Optional<TableSchema> tableSchemaInFileSystem(Path tablePath, String branchName) {
        return new SchemaManager(this.fileIO, tablePath, branchName).latest().map(s -> {
            if (!"main".equals(branchName)) {
                Options branchOptions = new Options(s.options());
                branchOptions.set(CoreOptions.BRANCH, branchName);
                return s.copy(branchOptions.toMap());
            }
            return s;
        });
    }

    protected static class TableMeta {
        private final TableSchema schema;
        @Nullable
        private final String uuid;

        public TableMeta(TableSchema schema, @Nullable String uuid) {
            this.schema = schema;
            this.uuid = uuid;
        }

        public TableSchema schema() {
            return this.schema;
        }

        @Nullable
        public String uuid() {
            return this.uuid;
        }
    }
}

