/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.catalog;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import javax.annotation.Nullable;
import org.apache.paimon.catalog.Catalog;
import org.apache.paimon.catalog.Database;
import org.apache.paimon.catalog.DelegateCatalog;
import org.apache.paimon.catalog.Identifier;
import org.apache.paimon.catalog.PropertyChange;
import org.apache.paimon.fs.Path;
import org.apache.paimon.options.CatalogOptions;
import org.apache.paimon.options.MemorySize;
import org.apache.paimon.options.Options;
import org.apache.paimon.partition.Partition;
import org.apache.paimon.schema.SchemaChange;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Cache;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Caffeine;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Ticker;
import org.apache.paimon.table.FileStoreTable;
import org.apache.paimon.table.Table;
import org.apache.paimon.table.system.SystemTableLoader;
import org.apache.paimon.utils.Preconditions;
import org.apache.paimon.utils.SegmentsCache;

public class CachingCatalog
extends DelegateCatalog {
    private final Duration expirationInterval;
    private final int snapshotMaxNumPerTable;
    protected final Cache<String, Database> databaseCache;
    protected final Cache<Identifier, Table> tableCache;
    @Nullable
    protected final SegmentsCache<Path> manifestCache;
    @Nullable
    protected final Cache<Identifier, List<Partition>> partitionCache;

    public CachingCatalog(Catalog wrapped) {
        this(wrapped, CatalogOptions.CACHE_EXPIRATION_INTERVAL_MS.defaultValue(), CatalogOptions.CACHE_MANIFEST_SMALL_FILE_MEMORY.defaultValue(), CatalogOptions.CACHE_MANIFEST_SMALL_FILE_THRESHOLD.defaultValue().getBytes(), CatalogOptions.CACHE_PARTITION_MAX_NUM.defaultValue(), CatalogOptions.CACHE_SNAPSHOT_MAX_NUM_PER_TABLE.defaultValue());
    }

    public CachingCatalog(Catalog wrapped, Duration expirationInterval, MemorySize manifestMaxMemory, long manifestCacheThreshold, long cachedPartitionMaxNum, int snapshotMaxNumPerTable) {
        this(wrapped, expirationInterval, manifestMaxMemory, manifestCacheThreshold, cachedPartitionMaxNum, snapshotMaxNumPerTable, Ticker.systemTicker());
    }

    public CachingCatalog(Catalog wrapped, Duration expirationInterval, MemorySize manifestMaxMemory, long manifestCacheThreshold, long cachedPartitionMaxNum, int snapshotMaxNumPerTable, Ticker ticker) {
        super(wrapped);
        if (expirationInterval.isZero() || expirationInterval.isNegative()) {
            throw new IllegalArgumentException("When cache.expiration-interval is set to negative or 0, the catalog cache should be disabled.");
        }
        this.expirationInterval = expirationInterval;
        this.snapshotMaxNumPerTable = snapshotMaxNumPerTable;
        this.databaseCache = Caffeine.newBuilder().softValues().executor(Runnable::run).expireAfterAccess(expirationInterval).ticker(ticker).build();
        this.tableCache = Caffeine.newBuilder().softValues().executor(Runnable::run).expireAfterAccess(expirationInterval).ticker(ticker).build();
        this.manifestCache = SegmentsCache.create(manifestMaxMemory, manifestCacheThreshold);
        this.partitionCache = cachedPartitionMaxNum == 0L ? null : Caffeine.newBuilder().softValues().executor(Runnable::run).expireAfterAccess(expirationInterval).weigher((identifier, v) -> v.size()).maximumWeight(cachedPartitionMaxNum).ticker(ticker).build();
    }

    public static Catalog tryToCreate(Catalog catalog, Options options) {
        if (!options.get(CatalogOptions.CACHE_ENABLED).booleanValue()) {
            return catalog;
        }
        MemorySize manifestMaxMemory = options.get(CatalogOptions.CACHE_MANIFEST_SMALL_FILE_MEMORY);
        long manifestThreshold = options.get(CatalogOptions.CACHE_MANIFEST_SMALL_FILE_THRESHOLD).getBytes();
        Optional<MemorySize> maxMemory = options.getOptional(CatalogOptions.CACHE_MANIFEST_MAX_MEMORY);
        if (maxMemory.isPresent() && maxMemory.get().compareTo(manifestMaxMemory) > 0) {
            manifestMaxMemory = maxMemory.get();
            manifestThreshold = Long.MAX_VALUE;
        }
        return new CachingCatalog(catalog, options.get(CatalogOptions.CACHE_EXPIRATION_INTERVAL_MS), manifestMaxMemory, manifestThreshold, options.get(CatalogOptions.CACHE_PARTITION_MAX_NUM), options.get(CatalogOptions.CACHE_SNAPSHOT_MAX_NUM_PER_TABLE));
    }

    @Override
    public Database getDatabase(String databaseName) throws Catalog.DatabaseNotExistException {
        Database database = this.databaseCache.getIfPresent(databaseName);
        if (database != null) {
            return database;
        }
        database = super.getDatabase(databaseName);
        this.databaseCache.put(databaseName, database);
        return database;
    }

    @Override
    public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade) throws Catalog.DatabaseNotExistException, Catalog.DatabaseNotEmptyException {
        super.dropDatabase(name, ignoreIfNotExists, cascade);
        this.databaseCache.invalidate(name);
        if (cascade) {
            ArrayList<Identifier> tables = new ArrayList<Identifier>();
            for (Identifier identifier : this.tableCache.asMap().keySet()) {
                if (!identifier.getDatabaseName().equals(name)) continue;
                tables.add(identifier);
            }
            tables.forEach(this.tableCache::invalidate);
        }
    }

    @Override
    public void alterDatabase(String name, List<PropertyChange> changes, boolean ignoreIfNotExists) throws Catalog.DatabaseNotExistException {
        super.alterDatabase(name, changes, ignoreIfNotExists);
        this.databaseCache.invalidate(name);
    }

    @Override
    public void dropTable(Identifier identifier, boolean ignoreIfNotExists) throws Catalog.TableNotExistException {
        super.dropTable(identifier, ignoreIfNotExists);
        this.invalidateTable(identifier);
        for (Identifier i : this.tableCache.asMap().keySet()) {
            if (!identifier.getTableName().equals(i.getTableName()) || !identifier.getDatabaseName().equals(i.getDatabaseName())) continue;
            this.tableCache.invalidate(i);
        }
    }

    @Override
    public void renameTable(Identifier fromTable, Identifier toTable, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.TableAlreadyExistException {
        super.renameTable(fromTable, toTable, ignoreIfNotExists);
        this.invalidateTable(fromTable);
    }

    @Override
    public void alterTable(Identifier identifier, List<SchemaChange> changes, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.ColumnAlreadyExistException, Catalog.ColumnNotExistException {
        super.alterTable(identifier, changes, ignoreIfNotExists);
        this.invalidateTable(identifier);
    }

    @Override
    public Table getTable(Identifier identifier) throws Catalog.TableNotExistException {
        Table table = this.tableCache.getIfPresent(identifier);
        if (table != null) {
            return table;
        }
        if (identifier.isSystemTable()) {
            Identifier originIdentifier = new Identifier(identifier.getDatabaseName(), identifier.getTableName(), identifier.getBranchName(), null);
            Table originTable = this.getTable(originIdentifier);
            table = SystemTableLoader.load(Preconditions.checkNotNull(identifier.getSystemTableName()), (FileStoreTable)originTable);
            if (table == null) {
                throw new Catalog.TableNotExistException(identifier);
            }
            return table;
        }
        table = this.wrapped.getTable(identifier);
        this.putTableCache(identifier, table);
        return table;
    }

    private void putTableCache(Identifier identifier, Table table) {
        if (table instanceof FileStoreTable) {
            FileStoreTable storeTable = (FileStoreTable)table;
            storeTable.setSnapshotCache(Caffeine.newBuilder().softValues().expireAfterAccess(this.expirationInterval).maximumSize(this.snapshotMaxNumPerTable).executor(Runnable::run).build());
            storeTable.setStatsCache(Caffeine.newBuilder().softValues().expireAfterAccess(this.expirationInterval).maximumSize(5L).executor(Runnable::run).build());
            if (this.manifestCache != null) {
                storeTable.setManifestCache(this.manifestCache);
            }
        }
        this.tableCache.put(identifier, table);
    }

    @Override
    public List<Partition> listPartitions(Identifier identifier) throws Catalog.TableNotExistException {
        if (this.partitionCache == null) {
            return this.wrapped.listPartitions(identifier);
        }
        List<Partition> result = this.partitionCache.getIfPresent(identifier);
        if (result == null) {
            result = this.wrapped.listPartitions(identifier);
            this.partitionCache.put(identifier, result);
        }
        return result;
    }

    @Override
    public void dropPartition(Identifier identifier, Map<String, String> partitions) throws Catalog.TableNotExistException, Catalog.PartitionNotExistException {
        this.wrapped.dropPartition(identifier, partitions);
        if (this.partitionCache != null) {
            this.partitionCache.invalidate(identifier);
        }
    }

    @Override
    public void invalidateTable(Identifier identifier) {
        this.tableCache.invalidate(identifier);
        if (this.partitionCache != null) {
            this.partitionCache.invalidate(identifier);
        }
    }

    public void refreshPartitions(Identifier identifier) throws Catalog.TableNotExistException {
        if (this.partitionCache != null) {
            List<Partition> result = this.wrapped.listPartitions(identifier);
            this.partitionCache.put(identifier, result);
        }
    }

    public CacheSizes estimatedCacheSizes() {
        long databaseCacheSize = this.databaseCache.estimatedSize();
        long tableCacheSize = this.tableCache.estimatedSize();
        long manifestCacheSize = 0L;
        long manifestCacheBytes = 0L;
        if (this.manifestCache != null) {
            manifestCacheSize = this.manifestCache.estimatedSize();
            manifestCacheBytes = this.manifestCache.totalCacheBytes();
        }
        long partitionCacheSize = 0L;
        if (this.partitionCache != null) {
            for (Map.Entry entry : this.partitionCache.asMap().entrySet()) {
                partitionCacheSize += (long)((List)entry.getValue()).size();
            }
        }
        return new CacheSizes(databaseCacheSize, tableCacheSize, manifestCacheSize, manifestCacheBytes, partitionCacheSize);
    }

    public static class CacheSizes {
        private final long databaseCacheSize;
        private final long tableCacheSize;
        private final long manifestCacheSize;
        private final long manifestCacheBytes;
        private final long partitionCacheSize;

        public CacheSizes(long databaseCacheSize, long tableCacheSize, long manifestCacheSize, long manifestCacheBytes, long partitionCacheSize) {
            this.databaseCacheSize = databaseCacheSize;
            this.tableCacheSize = tableCacheSize;
            this.manifestCacheSize = manifestCacheSize;
            this.manifestCacheBytes = manifestCacheBytes;
            this.partitionCacheSize = partitionCacheSize;
        }

        public long databaseCacheSize() {
            return this.databaseCacheSize;
        }

        public long tableCacheSize() {
            return this.tableCacheSize;
        }

        public long manifestCacheSize() {
            return this.manifestCacheSize;
        }

        public long manifestCacheBytes() {
            return this.manifestCacheBytes;
        }

        public long partitionCacheSize() {
            return this.partitionCacheSize;
        }
    }
}

