/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.catalog;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import javax.annotation.Nullable;
import org.apache.paimon.catalog.AbstractCatalog;
import org.apache.paimon.catalog.Catalog;
import org.apache.paimon.catalog.DelegateCatalog;
import org.apache.paimon.catalog.Identifier;
import org.apache.paimon.fs.Path;
import org.apache.paimon.options.CatalogOptions;
import org.apache.paimon.options.MemorySize;
import org.apache.paimon.options.Options;
import org.apache.paimon.schema.SchemaChange;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Cache;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Caffeine;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.RemovalCause;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.RemovalListener;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Ticker;
import org.apache.paimon.table.FileStoreTable;
import org.apache.paimon.table.Table;
import org.apache.paimon.table.system.SystemTableLoader;
import org.apache.paimon.utils.Preconditions;
import org.apache.paimon.utils.SegmentsCache;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CachingCatalog
extends DelegateCatalog {
    private static final Logger LOG = LoggerFactory.getLogger(CachingCatalog.class);
    protected final Cache<String, Map<String, String>> databaseCache;
    protected final Cache<Identifier, Table> tableCache;
    @Nullable
    protected final SegmentsCache<Path> manifestCache;

    public CachingCatalog(Catalog wrapped) {
        this(wrapped, CatalogOptions.CACHE_EXPIRATION_INTERVAL_MS.defaultValue(), CatalogOptions.CACHE_MANIFEST_SMALL_FILE_MEMORY.defaultValue(), CatalogOptions.CACHE_MANIFEST_SMALL_FILE_THRESHOLD.defaultValue().getBytes());
    }

    public CachingCatalog(Catalog wrapped, Duration expirationInterval, MemorySize manifestMaxMemory, long manifestCacheThreshold) {
        this(wrapped, expirationInterval, manifestMaxMemory, manifestCacheThreshold, Ticker.systemTicker());
    }

    public CachingCatalog(Catalog wrapped, Duration expirationInterval, MemorySize manifestMaxMemory, long manifestCacheThreshold, Ticker ticker) {
        super(wrapped);
        if (expirationInterval.isZero() || expirationInterval.isNegative()) {
            throw new IllegalArgumentException("When cache.expiration-interval is set to negative or 0, the catalog cache should be disabled.");
        }
        this.databaseCache = Caffeine.newBuilder().softValues().executor(Runnable::run).expireAfterAccess(expirationInterval).ticker(ticker).build();
        this.tableCache = Caffeine.newBuilder().softValues().removalListener(new TableInvalidatingRemovalListener()).executor(Runnable::run).expireAfterAccess(expirationInterval).ticker(ticker).build();
        this.manifestCache = SegmentsCache.create(manifestMaxMemory, manifestCacheThreshold);
    }

    public static Catalog tryToCreate(Catalog catalog, Options options) {
        if (!options.get(CatalogOptions.CACHE_ENABLED).booleanValue()) {
            return catalog;
        }
        MemorySize manifestMaxMemory = options.get(CatalogOptions.CACHE_MANIFEST_SMALL_FILE_MEMORY);
        long manifestThreshold = options.get(CatalogOptions.CACHE_MANIFEST_SMALL_FILE_THRESHOLD).getBytes();
        Optional<MemorySize> maxMemory = options.getOptional(CatalogOptions.CACHE_MANIFEST_MAX_MEMORY);
        if (maxMemory.isPresent() && maxMemory.get().compareTo(manifestMaxMemory) > 0) {
            manifestMaxMemory = maxMemory.get();
            manifestThreshold = Long.MAX_VALUE;
        }
        return new CachingCatalog(catalog, options.get(CatalogOptions.CACHE_EXPIRATION_INTERVAL_MS), manifestMaxMemory, manifestThreshold);
    }

    @Override
    public Map<String, String> loadDatabaseProperties(String databaseName) throws Catalog.DatabaseNotExistException {
        Map<String, String> properties = this.databaseCache.getIfPresent(databaseName);
        if (properties != null) {
            return properties;
        }
        properties = super.loadDatabaseProperties(databaseName);
        this.databaseCache.put(databaseName, properties);
        return properties;
    }

    @Override
    public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade) throws Catalog.DatabaseNotExistException, Catalog.DatabaseNotEmptyException {
        super.dropDatabase(name, ignoreIfNotExists, cascade);
        this.databaseCache.invalidate(name);
        if (cascade) {
            ArrayList<Identifier> tables = new ArrayList<Identifier>();
            for (Identifier identifier : this.tableCache.asMap().keySet()) {
                if (!identifier.getDatabaseName().equals(name)) continue;
                tables.add(identifier);
            }
            tables.forEach(this.tableCache::invalidate);
        }
    }

    @Override
    public void dropTable(Identifier identifier, boolean ignoreIfNotExists) throws Catalog.TableNotExistException {
        super.dropTable(identifier, ignoreIfNotExists);
        this.invalidateTable(identifier);
    }

    @Override
    public void renameTable(Identifier fromTable, Identifier toTable, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.TableAlreadyExistException {
        super.renameTable(fromTable, toTable, ignoreIfNotExists);
        this.invalidateTable(fromTable);
    }

    @Override
    public void alterTable(Identifier identifier, List<SchemaChange> changes, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.ColumnAlreadyExistException, Catalog.ColumnNotExistException {
        super.alterTable(identifier, changes, ignoreIfNotExists);
        this.invalidateTable(identifier);
    }

    @Override
    public Table getTable(Identifier identifier) throws Catalog.TableNotExistException {
        Table table = this.tableCache.getIfPresent(identifier);
        if (table != null) {
            return table;
        }
        if (AbstractCatalog.isSpecifiedSystemTable(identifier)) {
            Identifier originIdentifier = new Identifier(identifier.getDatabaseName(), identifier.getTableName(), identifier.getBranchName(), null);
            Table originTable = this.tableCache.getIfPresent(originIdentifier);
            if (originTable == null) {
                originTable = this.wrapped.getTable(originIdentifier);
                this.putTableCache(originIdentifier, originTable);
            }
            if ((table = SystemTableLoader.load(Preconditions.checkNotNull(identifier.getSystemTableName()), (FileStoreTable)originTable)) == null) {
                throw new Catalog.TableNotExistException(identifier);
            }
            this.putTableCache(identifier, table);
            return table;
        }
        table = this.wrapped.getTable(identifier);
        this.putTableCache(identifier, table);
        return table;
    }

    private void putTableCache(Identifier identifier, Table table) {
        if (this.manifestCache != null && table instanceof FileStoreTable) {
            ((FileStoreTable)table).setManifestCache(this.manifestCache);
        }
        this.tableCache.put(identifier, table);
    }

    @Override
    public void invalidateTable(Identifier identifier) {
        this.tableCache.invalidate(identifier);
        this.tryInvalidateSysTables(identifier);
    }

    private void tryInvalidateSysTables(Identifier identifier) {
        if (!AbstractCatalog.isSpecifiedSystemTable(identifier)) {
            this.tableCache.invalidateAll(CachingCatalog.allSystemTables(identifier));
        }
    }

    private static Iterable<Identifier> allSystemTables(Identifier ident) {
        ArrayList<Identifier> tables = new ArrayList<Identifier>();
        for (String type : SystemTableLoader.SYSTEM_TABLES) {
            tables.add(Identifier.fromString(ident.getFullName() + "$" + type));
        }
        return tables;
    }

    private class TableInvalidatingRemovalListener
    implements RemovalListener<Identifier, Table> {
        private TableInvalidatingRemovalListener() {
        }

        @Override
        public void onRemoval(Identifier identifier, Table table, @NonNull RemovalCause cause) {
            LOG.debug("Evicted {} from the table cache ({})", (Object)identifier, (Object)cause);
            if (RemovalCause.EXPIRED.equals((Object)cause)) {
                CachingCatalog.this.tryInvalidateSysTables(identifier);
            }
        }
    }
}

