/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.catalog;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.paimon.CoreOptions;
import org.apache.paimon.catalog.Catalog;
import org.apache.paimon.catalog.CatalogLockContext;
import org.apache.paimon.catalog.CatalogLockFactory;
import org.apache.paimon.catalog.Identifier;
import org.apache.paimon.factories.FactoryUtil;
import org.apache.paimon.fs.FileIO;
import org.apache.paimon.fs.FileStatus;
import org.apache.paimon.fs.Path;
import org.apache.paimon.lineage.LineageMetaFactory;
import org.apache.paimon.operation.FileStoreCommit;
import org.apache.paimon.operation.Lock;
import org.apache.paimon.options.CatalogOptions;
import org.apache.paimon.options.Options;
import org.apache.paimon.schema.Schema;
import org.apache.paimon.schema.SchemaChange;
import org.apache.paimon.schema.SchemaManager;
import org.apache.paimon.schema.TableSchema;
import org.apache.paimon.table.CatalogEnvironment;
import org.apache.paimon.table.FileStoreTable;
import org.apache.paimon.table.FileStoreTableFactory;
import org.apache.paimon.table.FormatTable;
import org.apache.paimon.table.Table;
import org.apache.paimon.table.system.SystemTableLoader;
import org.apache.paimon.utils.Preconditions;

public abstract class AbstractCatalog
implements Catalog {
    protected final FileIO fileIO;
    protected final Map<String, String> tableDefaultOptions;
    protected final Options catalogOptions;
    @Nullable
    protected final LineageMetaFactory lineageMetaFactory;

    protected AbstractCatalog(FileIO fileIO) {
        this.fileIO = fileIO;
        this.lineageMetaFactory = null;
        this.tableDefaultOptions = new HashMap<String, String>();
        this.catalogOptions = new Options();
    }

    protected AbstractCatalog(FileIO fileIO, Options options) {
        this.fileIO = fileIO;
        this.lineageMetaFactory = this.findAndCreateLineageMeta(options, AbstractCatalog.class.getClassLoader());
        this.tableDefaultOptions = Catalog.tableDefaultOptions(options.toMap());
        this.catalogOptions = options;
    }

    @Override
    public Map<String, String> options() {
        return this.catalogOptions.toMap();
    }

    @Override
    public FileIO fileIO() {
        return this.fileIO;
    }

    @Override
    public Optional<CatalogLockFactory> lockFactory() {
        if (!this.lockEnabled()) {
            return Optional.empty();
        }
        String lock = this.catalogOptions.get(CatalogOptions.LOCK_TYPE);
        if (lock == null) {
            return this.defaultLockFactory();
        }
        return Optional.of(FactoryUtil.discoverFactory(AbstractCatalog.class.getClassLoader(), CatalogLockFactory.class, lock));
    }

    public Optional<CatalogLockFactory> defaultLockFactory() {
        return Optional.empty();
    }

    @Override
    public Optional<CatalogLockContext> lockContext() {
        return Optional.of(CatalogLockContext.fromOptions(this.catalogOptions));
    }

    protected boolean lockEnabled() {
        return this.catalogOptions.getOptional(CatalogOptions.LOCK_ENABLED).orElse(this.fileIO.isObjectStore());
    }

    @Override
    public boolean allowUpperCase() {
        return this.catalogOptions.getOptional(CatalogOptions.ALLOW_UPPER_CASE).orElse(true);
    }

    @Override
    public void createDatabase(String name, boolean ignoreIfExists, Map<String, String> properties) throws Catalog.DatabaseAlreadyExistException {
        this.checkNotSystemDatabase(name);
        if (this.databaseExists(name)) {
            if (ignoreIfExists) {
                return;
            }
            throw new Catalog.DatabaseAlreadyExistException(name);
        }
        this.createDatabaseImpl(name, properties);
    }

    @Override
    public Map<String, String> loadDatabaseProperties(String name) throws Catalog.DatabaseNotExistException {
        if (AbstractCatalog.isSystemDatabase(name)) {
            return Collections.emptyMap();
        }
        return this.loadDatabasePropertiesImpl(name);
    }

    protected abstract Map<String, String> loadDatabasePropertiesImpl(String var1) throws Catalog.DatabaseNotExistException;

    @Override
    public void dropPartition(Identifier identifier, Map<String, String> partitionSpec) throws Catalog.TableNotExistException {
        AbstractCatalog.checkNotSystemTable(identifier, "dropPartition");
        Table table = this.getTable(identifier);
        FileStoreTable fileStoreTable = (FileStoreTable)table;
        try (FileStoreCommit commit = fileStoreTable.store().newCommit(CoreOptions.createCommitUser(fileStoreTable.coreOptions().toConfiguration()));){
            commit.dropPartitions(Collections.singletonList(partitionSpec), Long.MAX_VALUE);
        }
    }

    protected abstract void createDatabaseImpl(String var1, Map<String, String> var2);

    @Override
    public void dropDatabase(String name, boolean ignoreIfNotExists, boolean cascade) throws Catalog.DatabaseNotExistException, Catalog.DatabaseNotEmptyException {
        this.checkNotSystemDatabase(name);
        if (!this.databaseExists(name)) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.DatabaseNotExistException(name);
        }
        if (!cascade && !this.listTables(name).isEmpty()) {
            throw new Catalog.DatabaseNotEmptyException(name);
        }
        this.dropDatabaseImpl(name);
    }

    protected abstract void dropDatabaseImpl(String var1);

    @Override
    public List<String> listTables(String databaseName) throws Catalog.DatabaseNotExistException {
        if (AbstractCatalog.isSystemDatabase(databaseName)) {
            return SystemTableLoader.loadGlobalTableNames();
        }
        if (!this.databaseExists(databaseName)) {
            throw new Catalog.DatabaseNotExistException(databaseName);
        }
        return this.listTablesImpl(databaseName).stream().sorted().collect(Collectors.toList());
    }

    protected abstract List<String> listTablesImpl(String var1);

    @Override
    public void dropTable(Identifier identifier, boolean ignoreIfNotExists) throws Catalog.TableNotExistException {
        AbstractCatalog.checkNotBranch(identifier, "dropTable");
        AbstractCatalog.checkNotSystemTable(identifier, "dropTable");
        if (!this.tableExists(identifier)) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(identifier);
        }
        this.dropTableImpl(identifier);
    }

    protected abstract void dropTableImpl(Identifier var1);

    @Override
    public void createTable(Identifier identifier, Schema schema, boolean ignoreIfExists) throws Catalog.TableAlreadyExistException, Catalog.DatabaseNotExistException {
        AbstractCatalog.checkNotBranch(identifier, "createTable");
        AbstractCatalog.checkNotSystemTable(identifier, "createTable");
        this.validateIdentifierNameCaseInsensitive(identifier);
        this.validateFieldNameCaseInsensitive(schema.rowType().getFieldNames());
        this.validateAutoCreateClose(schema.options());
        if (!this.databaseExists(identifier.getDatabaseName())) {
            throw new Catalog.DatabaseNotExistException(identifier.getDatabaseName());
        }
        if (this.tableExists(identifier)) {
            if (ignoreIfExists) {
                return;
            }
            throw new Catalog.TableAlreadyExistException(identifier);
        }
        this.copyTableDefaultOptions(schema.options());
        this.createTableImpl(identifier, schema);
    }

    protected abstract void createTableImpl(Identifier var1, Schema var2);

    @Override
    public void renameTable(Identifier fromTable, Identifier toTable, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.TableAlreadyExistException {
        AbstractCatalog.checkNotBranch(fromTable, "renameTable");
        AbstractCatalog.checkNotBranch(toTable, "renameTable");
        AbstractCatalog.checkNotSystemTable(fromTable, "renameTable");
        AbstractCatalog.checkNotSystemTable(toTable, "renameTable");
        this.validateIdentifierNameCaseInsensitive(toTable);
        if (!this.tableExists(fromTable)) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(fromTable);
        }
        if (this.tableExists(toTable)) {
            throw new Catalog.TableAlreadyExistException(toTable);
        }
        this.renameTableImpl(fromTable, toTable);
    }

    protected abstract void renameTableImpl(Identifier var1, Identifier var2);

    @Override
    public void alterTable(Identifier identifier, List<SchemaChange> changes, boolean ignoreIfNotExists) throws Catalog.TableNotExistException, Catalog.ColumnAlreadyExistException, Catalog.ColumnNotExistException {
        AbstractCatalog.checkNotSystemTable(identifier, "alterTable");
        this.validateIdentifierNameCaseInsensitive(identifier);
        this.validateFieldNameCaseInsensitiveInSchemaChange(changes);
        if (!this.tableExists(identifier)) {
            if (ignoreIfNotExists) {
                return;
            }
            throw new Catalog.TableNotExistException(identifier);
        }
        this.alterTableImpl(identifier, changes);
    }

    protected abstract void alterTableImpl(Identifier var1, List<SchemaChange> var2) throws Catalog.TableNotExistException, Catalog.ColumnAlreadyExistException, Catalog.ColumnNotExistException;

    @Nullable
    private LineageMetaFactory findAndCreateLineageMeta(Options options, ClassLoader classLoader) {
        return options.getOptional(CatalogOptions.LINEAGE_META).map(meta -> FactoryUtil.discoverFactory(classLoader, LineageMetaFactory.class, meta)).orElse(null);
    }

    @Override
    public Table getTable(Identifier identifier) throws Catalog.TableNotExistException {
        if (AbstractCatalog.isSystemDatabase(identifier.getDatabaseName())) {
            String tableName = identifier.getTableName();
            Table table = SystemTableLoader.loadGlobal(tableName, this.fileIO, this::allTablePaths, this.catalogOptions, this.lineageMetaFactory);
            if (table == null) {
                throw new Catalog.TableNotExistException(identifier);
            }
            return table;
        }
        if (AbstractCatalog.isSpecifiedSystemTable(identifier)) {
            FileStoreTable originTable = this.getDataTable(new Identifier(identifier.getDatabaseName(), identifier.getTableName(), identifier.getBranchName(), null));
            Table table = SystemTableLoader.load(Preconditions.checkNotNull(identifier.getSystemTableName()), originTable);
            if (table == null) {
                throw new Catalog.TableNotExistException(identifier);
            }
            return table;
        }
        try {
            return this.getDataTable(identifier);
        }
        catch (Catalog.TableNotExistException e) {
            return this.getFormatTable(identifier);
        }
    }

    private FileStoreTable getDataTable(Identifier identifier) throws Catalog.TableNotExistException {
        Preconditions.checkArgument(identifier.getSystemTableName() == null);
        TableSchema tableSchema = this.getDataTableSchema(identifier);
        return FileStoreTableFactory.create(this.fileIO, this.getTableLocation(identifier), tableSchema, new CatalogEnvironment(identifier, Lock.factory(this.lockFactory().orElse(null), this.lockContext().orElse(null), identifier), this.metastoreClientFactory(identifier).orElse(null), this.lineageMetaFactory));
    }

    public FormatTable getFormatTable(Identifier identifier) throws Catalog.TableNotExistException {
        throw new Catalog.TableNotExistException(identifier);
    }

    public Path newDatabasePath(String database) {
        return AbstractCatalog.newDatabasePath(this.warehouse(), database);
    }

    public Map<String, Map<String, Path>> allTablePaths() {
        try {
            HashMap<String, Map<String, Path>> allPaths = new HashMap<String, Map<String, Path>>();
            for (String database : this.listDatabases()) {
                Map tableMap = allPaths.computeIfAbsent(database, d -> new HashMap());
                for (String table : this.listTables(database)) {
                    tableMap.put(table, this.getTableLocation(Identifier.create(database, table)));
                }
            }
            return allPaths;
        }
        catch (Catalog.DatabaseNotExistException e) {
            throw new RuntimeException("Database is deleted while listing", e);
        }
    }

    protected abstract TableSchema getDataTableSchema(Identifier var1) throws Catalog.TableNotExistException;

    @Override
    public Path getTableLocation(Identifier identifier) {
        return new Path(this.newDatabasePath(identifier.getDatabaseName()), identifier.getTableName());
    }

    protected static void checkNotBranch(Identifier identifier, String method) {
        if (identifier.getBranchName() != null) {
            throw new IllegalArgumentException(String.format("Cannot '%s' for branch table '%s', please modify the table with the default branch.", method, identifier));
        }
    }

    protected void assertMainBranch(Identifier identifier) {
        if (identifier.getBranchName() != null && !"main".equals(identifier.getBranchName())) {
            throw new UnsupportedOperationException(this.getClass().getName() + " currently does not support table branches");
        }
    }

    public static boolean isSpecifiedSystemTable(Identifier identifier) {
        return identifier.getSystemTableName() != null;
    }

    protected static boolean isSystemTable(Identifier identifier) {
        return AbstractCatalog.isSystemDatabase(identifier.getDatabaseName()) || AbstractCatalog.isSpecifiedSystemTable(identifier);
    }

    protected static void checkNotSystemTable(Identifier identifier, String method) {
        if (AbstractCatalog.isSystemTable(identifier)) {
            throw new IllegalArgumentException(String.format("Cannot '%s' for system table '%s', please use data table.", method, identifier));
        }
    }

    private void copyTableDefaultOptions(Map<String, String> options) {
        this.tableDefaultOptions.forEach(options::putIfAbsent);
    }

    public static Path newTableLocation(String warehouse, Identifier identifier) {
        AbstractCatalog.checkNotBranch(identifier, "newTableLocation");
        AbstractCatalog.checkNotSystemTable(identifier, "newTableLocation");
        return new Path(AbstractCatalog.newDatabasePath(warehouse, identifier.getDatabaseName()), identifier.getTableName());
    }

    public static Path newDatabasePath(String warehouse, String database) {
        return new Path(warehouse, database + ".db");
    }

    public static boolean isSystemDatabase(String database) {
        return "sys".equals(database);
    }

    protected void checkNotSystemDatabase(String database) {
        if (AbstractCatalog.isSystemDatabase(database)) {
            throw new Catalog.ProcessSystemDatabaseException();
        }
    }

    protected void validateIdentifierNameCaseInsensitive(Identifier identifier) {
        Catalog.validateCaseInsensitive(this.allowUpperCase(), "Database", identifier.getDatabaseName());
        Catalog.validateCaseInsensitive(this.allowUpperCase(), "Table", identifier.getObjectName());
    }

    private void validateFieldNameCaseInsensitiveInSchemaChange(List<SchemaChange> changes) {
        ArrayList<String> fieldNames = new ArrayList<String>();
        for (SchemaChange change : changes) {
            if (change instanceof SchemaChange.AddColumn) {
                SchemaChange.AddColumn addColumn = (SchemaChange.AddColumn)change;
                fieldNames.add(addColumn.fieldName());
                continue;
            }
            if (!(change instanceof SchemaChange.RenameColumn)) continue;
            SchemaChange.RenameColumn rename = (SchemaChange.RenameColumn)change;
            fieldNames.add(rename.newName());
        }
        this.validateFieldNameCaseInsensitive(fieldNames);
    }

    protected void validateFieldNameCaseInsensitive(List<String> fieldNames) {
        Catalog.validateCaseInsensitive(this.allowUpperCase(), "Field", fieldNames);
    }

    private void validateAutoCreateClose(Map<String, String> options) {
        Preconditions.checkArgument(!Boolean.parseBoolean(options.getOrDefault(CoreOptions.AUTO_CREATE.key(), CoreOptions.AUTO_CREATE.defaultValue().toString())), String.format("The value of %s property should be %s.", CoreOptions.AUTO_CREATE.key(), Boolean.FALSE));
    }

    protected List<String> listDatabasesInFileSystem(Path warehouse) throws IOException {
        ArrayList<String> databases = new ArrayList<String>();
        for (FileStatus status : this.fileIO.listDirectories(warehouse)) {
            Path path = status.getPath();
            if (!status.isDir() || !path.getName().endsWith(".db")) continue;
            String fileName = path.getName();
            databases.add(fileName.substring(0, fileName.length() - ".db".length()));
        }
        return databases;
    }

    protected List<String> listTablesInFileSystem(Path databasePath) throws IOException {
        ArrayList<String> tables = new ArrayList<String>();
        for (FileStatus status : this.fileIO.listDirectories(databasePath)) {
            if (!status.isDir() || !this.tableExistsInFileSystem(status.getPath(), "main")) continue;
            tables.add(status.getPath().getName());
        }
        return tables;
    }

    protected boolean tableExistsInFileSystem(Path tablePath, String branchName) {
        return !new SchemaManager(this.fileIO, tablePath, branchName).listAllIds().isEmpty();
    }

    public Optional<TableSchema> tableSchemaInFileSystem(Path tablePath, String branchName) {
        return new SchemaManager(this.fileIO, tablePath, branchName).latest().map(s -> {
            if (!"main".equals(branchName)) {
                Options branchOptions = new Options(s.options());
                branchOptions.set(CoreOptions.BRANCH, branchName);
                return s.copy(branchOptions.toMap());
            }
            return s;
        });
    }
}

