/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.vfs;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import javax.annotation.Nullable;
import org.apache.paimon.CoreOptions;
import org.apache.paimon.TableType;
import org.apache.paimon.annotation.VisibleForTesting;
import org.apache.paimon.catalog.CatalogContext;
import org.apache.paimon.catalog.Identifier;
import org.apache.paimon.fs.FileIO;
import org.apache.paimon.fs.Path;
import org.apache.paimon.options.CatalogOptions;
import org.apache.paimon.options.Options;
import org.apache.paimon.rest.RESTApi;
import org.apache.paimon.rest.RESTTokenFileIO;
import org.apache.paimon.rest.exceptions.AlreadyExistsException;
import org.apache.paimon.rest.exceptions.BadRequestException;
import org.apache.paimon.rest.exceptions.ForbiddenException;
import org.apache.paimon.rest.exceptions.NoSuchResourceException;
import org.apache.paimon.rest.exceptions.NotImplementedException;
import org.apache.paimon.rest.responses.GetDatabaseResponse;
import org.apache.paimon.rest.responses.GetTableResponse;
import org.apache.paimon.schema.Schema;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Cache;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Caffeine;
import org.apache.paimon.shade.caffeine2.com.github.benmanes.caffeine.cache.Ticker;
import org.apache.paimon.vfs.VFSCatalogIdentifier;
import org.apache.paimon.vfs.VFSDatabaseIdentifier;
import org.apache.paimon.vfs.VFSIdentifier;
import org.apache.paimon.vfs.VFSTableInfo;
import org.apache.paimon.vfs.VFSTableObjectIdentifier;
import org.apache.paimon.vfs.VFSTableRootIdentifier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VFSOperations {
    private static final Logger LOG = LoggerFactory.getLogger(VFSOperations.class);
    private final RESTApi api;
    private final CatalogContext context;
    @Nullable
    private Cache<Identifier, VFSTableInfo> tableCache;

    public VFSOperations(Options options) {
        this.api = new RESTApi(options);
        if (((Boolean)options.get(CatalogOptions.CACHE_ENABLED)).booleanValue()) {
            Duration expireAfterAccess = (Duration)options.get(CatalogOptions.CACHE_EXPIRE_AFTER_ACCESS);
            if (expireAfterAccess.isZero() || expireAfterAccess.isNegative()) {
                throw new IllegalArgumentException("When 'cache.expire-after-access' is set to negative or 0, the catalog cache should be disabled.");
            }
            Duration expireAfterWrite = (Duration)options.get(CatalogOptions.CACHE_EXPIRE_AFTER_WRITE);
            if (expireAfterWrite.isZero() || expireAfterWrite.isNegative()) {
                throw new IllegalArgumentException("When 'cache.expire-after-write' is set to negative or 0, the catalog cache should be disabled.");
            }
            LOG.info("Initialize virtual file system with table cache enabled, expireAfterAccess={}, expireAfterWrite={}", (Object)expireAfterAccess, (Object)expireAfterWrite);
            this.tableCache = Caffeine.newBuilder().softValues().executor(Runnable::run).expireAfterAccess(expireAfterAccess).expireAfterWrite(expireAfterWrite).ticker(Ticker.systemTicker()).build();
        } else {
            LOG.info("Initialize virtual file system with table cache disabled");
        }
        this.context = CatalogContext.create((Options)this.api.options());
    }

    public VFSIdentifier getVFSIdentifier(String virtualPath) throws IOException {
        if (virtualPath.startsWith("/")) {
            virtualPath = virtualPath.substring(1);
        }
        String[] parts = virtualPath.split("/");
        if (virtualPath.isEmpty() || parts.length == 0) {
            return new VFSCatalogIdentifier();
        }
        if (parts.length == 1) {
            return new VFSDatabaseIdentifier(parts[0]);
        }
        String databaseName = parts[0];
        String tableName = parts[1];
        String relativePath = null;
        if (parts.length > 2) {
            relativePath = String.join((CharSequence)"/", Arrays.copyOfRange(parts, 2, parts.length));
        }
        Identifier identifier = new Identifier(databaseName, tableName);
        try {
            VFSTableInfo tableInfo = this.getTableInfo(identifier);
            return relativePath == null ? new VFSTableRootIdentifier(databaseName, tableName, tableInfo) : new VFSTableObjectIdentifier(databaseName, tableName, relativePath, tableInfo);
        }
        catch (FileNotFoundException e) {
            return relativePath == null ? new VFSTableRootIdentifier(databaseName, tableName) : new VFSTableObjectIdentifier(databaseName, tableName, relativePath);
        }
    }

    public GetDatabaseResponse getDatabase(String databaseName) throws IOException {
        try {
            return this.api.getDatabase(databaseName);
        }
        catch (NoSuchResourceException e) {
            throw new FileNotFoundException("Database " + databaseName + " not found");
        }
        catch (ForbiddenException e) {
            throw new IOException("No permission to access database " + databaseName);
        }
    }

    public List<String> listDatabases() {
        return this.api.listDatabases();
    }

    public void createDatabase(String databaseName) throws IOException {
        try {
            this.api.createDatabase(databaseName, Collections.emptyMap());
        }
        catch (AlreadyExistsException e) {
            LOG.info("Database {} already exist, no need to create", (Object)databaseName);
        }
        catch (ForbiddenException e) {
            throw new IOException("No permission to create database " + databaseName);
        }
        catch (BadRequestException e) {
            throw new IOException("Bad request when creating database " + databaseName, e);
        }
    }

    public void dropDatabase(String databaseName, boolean recursive) throws IOException {
        try {
            if (!recursive && !this.api.listTables(databaseName).isEmpty()) {
                throw new IOException("Database " + databaseName + " is not empty, set recursive to true to drop it");
            }
            this.api.dropDatabase(databaseName);
            if (this.isCacheEnabled()) {
                ArrayList<Identifier> tables = new ArrayList<Identifier>();
                for (Identifier identifier : this.tableCache.asMap().keySet()) {
                    if (!identifier.getDatabaseName().equals(databaseName)) continue;
                    tables.add(identifier);
                }
                tables.forEach(arg_0 -> this.tableCache.invalidate(arg_0));
            }
        }
        catch (NoSuchResourceException e) {
            throw new FileNotFoundException("Database " + databaseName + " not found");
        }
        catch (ForbiddenException e) {
            throw new IOException("No permission to drop database " + databaseName);
        }
    }

    public List<String> listTables(String databaseName) throws IOException {
        try {
            return this.api.listTables(databaseName);
        }
        catch (NoSuchResourceException e) {
            throw new FileNotFoundException("Database " + databaseName + " not found");
        }
        catch (ForbiddenException e) {
            throw new IOException("No permission to access database " + databaseName);
        }
    }

    public void createObjectTable(String databaseName, String tableName) throws IOException {
        Identifier identifier = Identifier.create((String)databaseName, (String)tableName);
        Schema schema = Schema.newBuilder().option(CoreOptions.TYPE.key(), TableType.OBJECT_TABLE.toString()).build();
        try {
            this.tryCreateObjectTable(identifier, schema);
        }
        catch (FileNotFoundException e) {
            this.createDatabase(databaseName);
            this.tryCreateObjectTable(identifier, schema);
        }
    }

    public void dropTable(String databaseName, String tableName) throws IOException {
        Identifier identifier = Identifier.create((String)databaseName, (String)tableName);
        try {
            this.api.dropTable(identifier);
            if (this.isCacheEnabled()) {
                this.tableCache.invalidate((Object)identifier);
            }
        }
        catch (NoSuchResourceException e) {
            throw new FileNotFoundException("Table " + identifier + " not found");
        }
        catch (ForbiddenException e) {
            throw new IOException("No permission to drop table " + identifier);
        }
    }

    public void renameTable(String databaseName, String srcTableName, String dstTableName) throws IOException {
        Identifier srcIdentifier = Identifier.create((String)databaseName, (String)srcTableName);
        Identifier dstIdentifier = Identifier.create((String)databaseName, (String)dstTableName);
        try {
            this.api.renameTable(srcIdentifier, dstIdentifier);
            if (this.isCacheEnabled()) {
                this.tableCache.invalidate((Object)srcIdentifier);
                this.tableCache.invalidate((Object)dstIdentifier);
            }
        }
        catch (NoSuchResourceException e) {
            throw new FileNotFoundException("Source table " + srcIdentifier + " not found");
        }
        catch (ForbiddenException e) {
            throw new IOException("No permission to rename table " + srcIdentifier + " to " + dstIdentifier);
        }
        catch (AlreadyExistsException e) {
            throw new FileAlreadyExistsException("Target table " + dstIdentifier + " already exist");
        }
        catch (BadRequestException e) {
            throw new IOException("Bad request when renaming table " + srcIdentifier + " to " + dstIdentifier, e);
        }
    }

    private void tryCreateObjectTable(Identifier identifier, Schema schema) throws IOException {
        try {
            this.api.createTable(identifier, schema);
        }
        catch (AlreadyExistsException e) {
            LOG.info("Table {} already exist, no need to create", (Object)identifier);
        }
        catch (NotImplementedException e) {
            throw new IOException("Create object table not implemented");
        }
        catch (NoSuchResourceException e) {
            throw new FileNotFoundException("Database not found");
        }
        catch (BadRequestException e) {
            throw new IOException("Bad request when creating table " + identifier, e);
        }
        catch (IllegalArgumentException e) {
            throw new IOException("Illegal argument when creating table " + identifier, e);
        }
        catch (Exception e) {
            throw new IOException(e);
        }
    }

    private GetTableResponse loadTableMetadata(Identifier identifier) throws IOException {
        GetTableResponse response;
        Identifier loadTableIdentifier = identifier.isSystemTable() ? new Identifier(identifier.getDatabaseName(), identifier.getTableName(), identifier.getBranchName()) : identifier;
        try {
            response = this.api.getTable(loadTableIdentifier);
        }
        catch (NoSuchResourceException e) {
            throw new FileNotFoundException("Table not found");
        }
        catch (ForbiddenException e) {
            throw new IOException("No permission to access table " + identifier);
        }
        return response;
    }

    private VFSTableInfo loadTableInfo(Identifier identifier) throws IOException {
        GetTableResponse table = this.loadTableMetadata(identifier);
        if (table.isExternal()) {
            throw new IOException("Do not support visiting external table " + identifier);
        }
        Path tablePath = new Path(table.getPath());
        RESTTokenFileIO fileIO = new RESTTokenFileIO(this.context, this.api, identifier, tablePath);
        return new VFSTableInfo(table.getId(), tablePath, (FileIO)fileIO);
    }

    private VFSTableInfo getTableInfo(Identifier identifier) throws IOException {
        if (!this.isCacheEnabled()) {
            return this.loadTableInfo(identifier);
        }
        VFSTableInfo vfsTableInfo = (VFSTableInfo)this.tableCache.getIfPresent((Object)identifier);
        if (vfsTableInfo != null) {
            return vfsTableInfo;
        }
        vfsTableInfo = this.loadTableInfo(identifier);
        this.tableCache.put((Object)identifier, (Object)vfsTableInfo);
        return vfsTableInfo;
    }

    @VisibleForTesting
    public boolean isCacheEnabled() {
        return this.tableCache != null;
    }
}

