/*
 * Decompiled with CFR 0.152.
 */
package org.apache.parquet.bytes;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Random;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.apache.parquet.bytes.ByteBufferAllocator;
import org.apache.parquet.bytes.ByteBufferInputStream;
import org.apache.parquet.bytes.ByteBufferReleaser;
import org.apache.parquet.bytes.BytesInput;
import org.apache.parquet.bytes.BytesUtils;
import org.apache.parquet.bytes.CapacityByteArrayOutputStream;
import org.apache.parquet.bytes.ConcatenatingByteBufferCollector;
import org.apache.parquet.bytes.DirectByteBufferAllocator;
import org.apache.parquet.bytes.HeapByteBufferAllocator;
import org.apache.parquet.bytes.TrackingByteBufferAllocator;
import org.apache.parquet.util.AutoCloseables;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.mockito.Matchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

@RunWith(value=Parameterized.class)
public class TestBytesInput {
    private static final Random RANDOM = new Random(202402201628L);
    private TrackingByteBufferAllocator allocator;
    private final ByteBufferAllocator innerAllocator;

    @Parameterized.Parameters(name="{0}")
    public static Object[][] parameters() {
        return new Object[][]{{new HeapByteBufferAllocator(){

            public String toString() {
                return "heap-allocator";
            }
        }}, {new DirectByteBufferAllocator(){

            public String toString() {
                return "direct-allocator";
            }
        }}};
    }

    public TestBytesInput(ByteBufferAllocator innerAllocator) {
        this.innerAllocator = innerAllocator;
    }

    @Before
    public void initAllocator() {
        this.allocator = TrackingByteBufferAllocator.wrap((ByteBufferAllocator)this.innerAllocator);
    }

    @After
    public void closeAllocator() {
        this.allocator.close();
    }

    @Test
    public void testFromSingleByteBuffer() throws IOException {
        byte[] data = new byte[1000];
        RANDOM.nextBytes(data);
        Supplier<BytesInput> factory = () -> BytesInput.from((ByteBuffer[])new ByteBuffer[]{this.toByteBuffer(data)});
        this.validate(data, factory);
        this.validateToByteBufferIsInternal(factory);
    }

    @Test
    public void testFromMultipleByteBuffers() throws IOException {
        byte[] data = new byte[1000];
        RANDOM.nextBytes(data);
        Supplier<BytesInput> factory = () -> BytesInput.from((ByteBuffer[])new ByteBuffer[]{this.toByteBuffer(data, 0, 250), this.toByteBuffer(data, 250, 250), this.toByteBuffer(data, 500, 250), this.toByteBuffer(data, 750, 250)});
        this.validate(data, factory);
    }

    @Test
    public void testFromByteArray() throws IOException {
        byte[] data = new byte[1000];
        RANDOM.nextBytes(data);
        byte[] input = new byte[data.length + 20];
        RANDOM.nextBytes(input);
        System.arraycopy(data, 0, input, 10, data.length);
        Supplier<BytesInput> factory = () -> BytesInput.from((byte[])input, (int)10, (int)data.length);
        this.validate(data, factory);
    }

    @Test
    public void testFromInputStream() throws IOException {
        byte[] data = new byte[1000];
        RANDOM.nextBytes(data);
        byte[] input = new byte[data.length + 10];
        RANDOM.nextBytes(input);
        System.arraycopy(data, 0, input, 0, data.length);
        Supplier<BytesInput> factory = () -> BytesInput.from((InputStream)new ByteArrayInputStream(input), (int)1000);
        this.validate(data, factory);
    }

    @Test
    public void testFromByteArrayOutputStream() throws IOException {
        byte[] data = new byte[1000];
        RANDOM.nextBytes(data);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        baos.write(data);
        Supplier<BytesInput> factory = () -> BytesInput.from((ByteArrayOutputStream)baos);
        this.validate(data, factory);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFromCapacityByteArrayOutputStreamOneSlab() throws IOException {
        byte[] data = new byte[1000];
        RANDOM.nextBytes(data);
        ArrayList toClose = new ArrayList();
        Supplier<BytesInput> factory = () -> {
            CapacityByteArrayOutputStream cbaos = new CapacityByteArrayOutputStream(10, 1000, (ByteBufferAllocator)this.allocator);
            toClose.add(cbaos);
            try {
                cbaos.write(data);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            return BytesInput.from((CapacityByteArrayOutputStream)cbaos);
        };
        try {
            this.validate(data, factory);
            this.validateToByteBufferIsInternal(factory);
        }
        finally {
            AutoCloseables.uncheckedClose(toClose);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFromCapacityByteArrayOutputStreamMultipleSlabs() throws IOException {
        byte[] data = new byte[1000];
        RANDOM.nextBytes(data);
        ArrayList toClose = new ArrayList();
        Supplier<BytesInput> factory = () -> {
            CapacityByteArrayOutputStream cbaos = new CapacityByteArrayOutputStream(10, 1000, (ByteBufferAllocator)this.allocator);
            toClose.add(cbaos);
            for (byte b : data) {
                cbaos.write((int)b);
            }
            return BytesInput.from((CapacityByteArrayOutputStream)cbaos);
        };
        try {
            this.validate(data, factory);
        }
        finally {
            AutoCloseables.uncheckedClose(toClose);
        }
    }

    @Test
    public void testFromInt() throws IOException {
        int value = RANDOM.nextInt();
        ByteArrayOutputStream baos = new ByteArrayOutputStream(4);
        BytesUtils.writeIntLittleEndian((OutputStream)baos, (int)value);
        byte[] data = baos.toByteArray();
        Supplier<BytesInput> factory = () -> BytesInput.fromInt((int)value);
        this.validate(data, factory);
    }

    @Test
    public void testFromUnsignedVarInt() throws IOException {
        int value = RANDOM.nextInt(Short.MAX_VALUE);
        ByteArrayOutputStream baos = new ByteArrayOutputStream(2);
        BytesUtils.writeUnsignedVarInt((int)value, (OutputStream)baos);
        byte[] data = baos.toByteArray();
        Supplier<BytesInput> factory = () -> BytesInput.fromUnsignedVarInt((int)value);
        this.validate(data, factory);
    }

    @Test
    public void testFromUnsignedVarLong() throws IOException {
        long value = RANDOM.nextInt(Integer.MAX_VALUE);
        ByteArrayOutputStream baos = new ByteArrayOutputStream(4);
        BytesUtils.writeUnsignedVarLong((long)value, (OutputStream)baos);
        byte[] data = baos.toByteArray();
        Supplier<BytesInput> factory = () -> BytesInput.fromUnsignedVarLong((long)value);
        this.validate(data, factory);
    }

    @Test
    public void testFromZigZagVarInt() throws IOException {
        int value = RANDOM.nextInt() % Short.MAX_VALUE;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        BytesUtils.writeZigZagVarInt((int)value, (OutputStream)baos);
        byte[] data = baos.toByteArray();
        Supplier<BytesInput> factory = () -> BytesInput.fromZigZagVarInt((int)value);
        this.validate(data, factory);
    }

    @Test
    public void testFromZigZagVarLong() throws IOException {
        long value = RANDOM.nextInt();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        BytesUtils.writeZigZagVarLong((long)value, (OutputStream)baos);
        byte[] data = baos.toByteArray();
        Supplier<BytesInput> factory = () -> BytesInput.fromZigZagVarLong((long)value);
        this.validate(data, factory);
    }

    @Test
    public void testEmpty() throws IOException {
        byte[] data = new byte[]{};
        Supplier<BytesInput> factory = () -> BytesInput.empty();
        this.validate(data, factory);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testConcatenatingByteBufferCollectorOneSlab() throws IOException {
        byte[] data = new byte[1000];
        RANDOM.nextBytes(data);
        ArrayList toClose = new ArrayList();
        Supplier<BytesInput> factory = () -> {
            ConcatenatingByteBufferCollector collector = new ConcatenatingByteBufferCollector((ByteBufferAllocator)this.allocator);
            toClose.add(collector);
            collector.collect(BytesInput.from((ByteBuffer[])new ByteBuffer[]{this.toByteBuffer(data)}));
            return collector;
        };
        try {
            this.validate(data, factory);
            this.validateToByteBufferIsInternal(factory);
        }
        finally {
            AutoCloseables.uncheckedClose(toClose);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testConcatenatingByteBufferCollectorMultipleSlabs() throws IOException {
        byte[] data = new byte[1000];
        RANDOM.nextBytes(data);
        ArrayList toClose = new ArrayList();
        Supplier<BytesInput> factory = () -> {
            ConcatenatingByteBufferCollector collector = new ConcatenatingByteBufferCollector((ByteBufferAllocator)this.allocator);
            toClose.add(collector);
            collector.collect(BytesInput.from((ByteBuffer[])new ByteBuffer[]{this.toByteBuffer(data, 0, 250)}));
            collector.collect(BytesInput.from((ByteBuffer[])new ByteBuffer[]{this.toByteBuffer(data, 250, 250)}));
            collector.collect(BytesInput.from((ByteBuffer[])new ByteBuffer[]{this.toByteBuffer(data, 500, 250)}));
            collector.collect(BytesInput.from((ByteBuffer[])new ByteBuffer[]{this.toByteBuffer(data, 750, 250)}));
            return collector;
        };
        try {
            this.validate(data, factory);
        }
        finally {
            AutoCloseables.uncheckedClose(toClose);
        }
    }

    @Test
    public void testConcat() throws IOException {
        byte[] data = new byte[1000];
        RANDOM.nextBytes(data);
        Supplier<BytesInput> factory = () -> BytesInput.concat((BytesInput[])new BytesInput[]{BytesInput.from((ByteBuffer[])new ByteBuffer[]{this.toByteBuffer(data, 0, 250)}), BytesInput.empty(), BytesInput.from((ByteBuffer[])new ByteBuffer[]{this.toByteBuffer(data, 250, 250)}), BytesInput.from((byte[])data, (int)500, (int)250), BytesInput.from((InputStream)new ByteArrayInputStream(data, 750, 250), (int)250)});
        this.validate(data, factory);
    }

    private ByteBuffer toByteBuffer(byte[] data) {
        return this.toByteBuffer(data, 0, data.length);
    }

    private ByteBuffer toByteBuffer(byte[] data, int offset, int length) {
        ByteBuffer buf = this.innerAllocator.allocate(length);
        buf.put(data, offset, length);
        buf.flip();
        return buf;
    }

    private void validate(byte[] data, Supplier<BytesInput> factory) throws IOException {
        Assert.assertEquals((long)data.length, (long)factory.get().size());
        this.validateToByteBuffer(data, factory);
        this.validateCopy(data, factory);
        this.validateToInputStream(data, factory);
        this.validateWriteAllTo(data, factory);
    }

    private void validateToByteBuffer(byte[] data, Supplier<BytesInput> factory) {
        BytesInput bi = factory.get();
        try (ByteBufferReleaser releaser = new ByteBufferReleaser((ByteBufferAllocator)this.allocator);){
            ByteBuffer buf = bi.toByteBuffer(releaser);
            int index = 0;
            while (buf.hasRemaining()) {
                if (buf.get() == data[index++]) continue;
                Assert.fail((String)("Data mismatch at position " + index));
            }
        }
    }

    private void validateCopy(byte[] data, Supplier<BytesInput> factory) throws IOException {
        BytesInput bi = factory.get();
        try (ByteBufferReleaser releaser = new ByteBufferReleaser((ByteBufferAllocator)this.allocator);
             ByteBufferInputStream is = bi.copy(releaser).toInputStream();){
            this.assertContentEquals(data, (InputStream)is);
        }
    }

    private void validateToInputStream(byte[] data, Supplier<BytesInput> factory) throws IOException {
        BytesInput bi = factory.get();
        try (ByteBufferInputStream is = bi.toInputStream();){
            this.assertContentEquals(data, (InputStream)is);
        }
    }

    private void assertContentEquals(byte[] expected, InputStream is) throws IOException {
        byte[] actual = new byte[expected.length];
        is.read(actual);
        Assert.assertArrayEquals((byte[])expected, (byte[])actual);
    }

    private void validateWriteAllTo(byte[] data, Supplier<BytesInput> factory) throws IOException {
        BytesInput bi = factory.get();
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            bi.writeAllTo((OutputStream)baos);
            Assert.assertArrayEquals((byte[])data, (byte[])baos.toByteArray());
        }
    }

    private void validateToByteBufferIsInternal(Supplier<BytesInput> factory) {
        ByteBufferAllocator allocatorMock = (ByteBufferAllocator)Mockito.mock(ByteBufferAllocator.class);
        Mockito.when((Object)allocatorMock.isDirect()).thenReturn((Object)this.innerAllocator.isDirect());
        Consumer callbackMock = (Consumer)Mockito.mock(Consumer.class);
        factory.get().toByteBuffer(allocatorMock, callbackMock);
        ((ByteBufferAllocator)Mockito.verify((Object)allocatorMock, (VerificationMode)Mockito.never())).allocate(Matchers.anyInt());
        ((Consumer)Mockito.verify((Object)callbackMock, (VerificationMode)Mockito.never())).accept(Matchers.anyObject());
    }
}

