/*
 * Decompiled with CFR 0.152.
 */
package org.apache.parquet.bytes;

import java.nio.ByteBuffer;
import java.nio.InvalidMarkException;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.function.Function;
import org.apache.parquet.bytes.ByteBufferAllocator;
import org.apache.parquet.bytes.ByteBufferReleaser;
import org.apache.parquet.bytes.DirectByteBufferAllocator;
import org.apache.parquet.bytes.HeapByteBufferAllocator;
import org.apache.parquet.bytes.ReusingByteBufferAllocator;
import org.apache.parquet.bytes.TrackingByteBufferAllocator;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class TestReusingByteBufferAllocator {
    private static final Random RANDOM = new Random(202402220951L);
    private TrackingByteBufferAllocator allocator;
    @Parameterized.Parameter
    public ByteBufferAllocator innerAllocator;
    @Parameterized.Parameter(value=1)
    public AllocatorType type;

    @Parameterized.Parameters(name="{0} {1}")
    public static List<Object[]> parameters() {
        ArrayList<Object[]> params = new ArrayList<Object[]>();
        for (Object allocator : new Object[]{new HeapByteBufferAllocator(){

            public String toString() {
                return "HEAP";
            }
        }, new DirectByteBufferAllocator(){

            public String toString() {
                return "DIRECT";
            }
        }}) {
            for (AllocatorType type : AllocatorType.values()) {
                params.add(new Object[]{allocator, type});
            }
        }
        return params;
    }

    @Before
    public void initAllocator() {
        this.allocator = TrackingByteBufferAllocator.wrap((ByteBufferAllocator)this.innerAllocator);
    }

    @After
    public void closeAllocator() {
        this.allocator.close();
    }

    @Test
    public void normalUseCase() {
        try (ReusingByteBufferAllocator reusingAllocator = this.type.create((ByteBufferAllocator)this.allocator);){
            Assert.assertEquals((Object)this.innerAllocator.isDirect(), (Object)reusingAllocator.isDirect());
            for (int i = 0; i < 10; ++i) {
                try (ByteBufferReleaser releaser = reusingAllocator.getReleaser();){
                    int size = RANDOM.nextInt(1024);
                    ByteBuffer buf = reusingAllocator.allocate(size);
                    releaser.releaseLater(buf);
                    this.validateBuffer(buf, size);
                    buf.position(buf.capacity() / 2);
                    buf.mark();
                    buf.position(buf.limit());
                    continue;
                }
            }
            reusingAllocator.allocate(1025);
        }
    }

    private void validateBuffer(ByteBuffer buf, int size) {
        Assert.assertEquals((long)0L, (long)buf.position());
        Assert.assertEquals((long)size, (long)buf.capacity());
        Assert.assertEquals((long)size, (long)buf.remaining());
        Assert.assertEquals((Object)this.allocator.isDirect(), (Object)buf.isDirect());
        Assert.assertThrows(InvalidMarkException.class, buf::reset);
    }

    @Test
    public void validateExceptions() {
        try (ByteBufferReleaser releaser = new ByteBufferReleaser((ByteBufferAllocator)this.allocator);
             ReusingByteBufferAllocator reusingAllocator = this.type.create((ByteBufferAllocator)this.allocator);){
            ByteBuffer fromOther = this.allocator.allocate(10);
            releaser.releaseLater(fromOther);
            Assert.assertThrows(IllegalStateException.class, () -> reusingAllocator.release(fromOther));
            ByteBuffer fromReusing = reusingAllocator.allocate(10);
            Assert.assertThrows(IllegalArgumentException.class, () -> reusingAllocator.release(fromOther));
            switch (this.type) {
                case STRICT: {
                    Assert.assertThrows(IllegalStateException.class, () -> reusingAllocator.allocate(5));
                    break;
                }
                case UNSAFE: {
                    fromReusing = reusingAllocator.allocate(5);
                    this.validateBuffer(fromReusing, 5);
                }
            }
            reusingAllocator.release(fromReusing);
            ByteBuffer fromReusingFinal = fromReusing;
            Assert.assertThrows(IllegalStateException.class, () -> reusingAllocator.release(fromOther));
            Assert.assertThrows(IllegalStateException.class, () -> reusingAllocator.release(fromReusingFinal));
        }
    }

    private static enum AllocatorType {
        STRICT(ReusingByteBufferAllocator::strict),
        UNSAFE(ReusingByteBufferAllocator::unsafe);

        private final Function<ByteBufferAllocator, ReusingByteBufferAllocator> factory;

        private AllocatorType(Function<ByteBufferAllocator, ReusingByteBufferAllocator> factory) {
            this.factory = factory;
        }

        public ReusingByteBufferAllocator create(ByteBufferAllocator allocator) {
            return this.factory.apply(allocator);
        }
    }
}

