/*
 * Decompiled with CFR 0.152.
 */
package org.apache.parquet.bytes;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.util.Arrays;
import java.util.List;
import org.apache.parquet.Log;
import org.apache.parquet.bytes.ByteBufferInputStream;
import org.apache.parquet.bytes.BytesUtils;
import org.apache.parquet.bytes.CapacityByteArrayOutputStream;

public abstract class BytesInput {
    private static final Log LOG = Log.getLog(BytesInput.class);
    private static final boolean DEBUG = false;
    private static final EmptyBytesInput EMPTY_BYTES_INPUT = new EmptyBytesInput();

    public static BytesInput concat(BytesInput ... inputs) {
        return new SequenceBytesIn(Arrays.asList(inputs));
    }

    public static BytesInput concat(List<BytesInput> inputs) {
        return new SequenceBytesIn(inputs);
    }

    public static BytesInput from(InputStream in, int bytes) {
        return new StreamBytesInput(in, bytes);
    }

    public static BytesInput from(ByteBuffer buffer, int offset, int length) {
        return new ByteBufferBytesInput(buffer, offset, length);
    }

    public static BytesInput from(byte[] in) {
        return new ByteArrayBytesInput(in, 0, in.length);
    }

    public static BytesInput from(byte[] in, int offset, int length) {
        return new ByteArrayBytesInput(in, offset, length);
    }

    public static BytesInput fromInt(int intValue) {
        return new IntBytesInput(intValue);
    }

    public static BytesInput fromUnsignedVarInt(int intValue) {
        return new UnsignedVarIntBytesInput(intValue);
    }

    public static BytesInput fromZigZagVarInt(int intValue) {
        int zigZag = intValue << 1 ^ intValue >> 31;
        return new UnsignedVarIntBytesInput(zigZag);
    }

    public static BytesInput fromUnsignedVarLong(long longValue) {
        return new UnsignedVarLongBytesInput(longValue);
    }

    public static BytesInput fromZigZagVarLong(long longValue) {
        long zigZag = longValue << 1 ^ longValue >> 63;
        return new UnsignedVarLongBytesInput(zigZag);
    }

    public static BytesInput from(CapacityByteArrayOutputStream arrayOut) {
        return new CapacityBAOSBytesInput(arrayOut);
    }

    public static BytesInput from(ByteArrayOutputStream baos) {
        return new BAOSBytesInput(baos);
    }

    public static BytesInput empty() {
        return EMPTY_BYTES_INPUT;
    }

    public static BytesInput copy(BytesInput bytesInput) throws IOException {
        return BytesInput.from(bytesInput.toByteArray());
    }

    public abstract void writeAllTo(OutputStream var1) throws IOException;

    public byte[] toByteArray() throws IOException {
        BAOS baos = new BAOS((int)this.size());
        this.writeAllTo(baos);
        return baos.getBuf();
    }

    public ByteBuffer toByteBuffer() throws IOException {
        return ByteBuffer.wrap(this.toByteArray());
    }

    public InputStream toInputStream() throws IOException {
        return new ByteBufferInputStream(this.toByteBuffer());
    }

    public abstract long size();

    private static class ByteBufferBytesInput
    extends BytesInput {
        private final ByteBuffer byteBuf;
        private final int length;
        private final int offset;

        private ByteBufferBytesInput(ByteBuffer byteBuf, int offset, int length) {
            this.byteBuf = byteBuf;
            this.offset = offset;
            this.length = length;
        }

        @Override
        public void writeAllTo(OutputStream out) throws IOException {
            WritableByteChannel outputChannel = Channels.newChannel(out);
            this.byteBuf.position(this.offset);
            ByteBuffer tempBuf = this.byteBuf.slice();
            tempBuf.limit(this.length);
            outputChannel.write(tempBuf);
        }

        @Override
        public ByteBuffer toByteBuffer() throws IOException {
            this.byteBuf.position(this.offset);
            ByteBuffer buf = this.byteBuf.slice();
            buf.limit(this.length);
            return buf;
        }

        @Override
        public long size() {
            return this.length;
        }
    }

    private static class ByteArrayBytesInput
    extends BytesInput {
        private final byte[] in;
        private final int offset;
        private final int length;

        private ByteArrayBytesInput(byte[] in, int offset, int length) {
            this.in = in;
            this.offset = offset;
            this.length = length;
        }

        @Override
        public void writeAllTo(OutputStream out) throws IOException {
            out.write(this.in, this.offset, this.length);
        }

        @Override
        public ByteBuffer toByteBuffer() throws IOException {
            return ByteBuffer.wrap(this.in, this.offset, this.length);
        }

        @Override
        public long size() {
            return this.length;
        }
    }

    private static class BAOSBytesInput
    extends BytesInput {
        private final ByteArrayOutputStream arrayOut;

        private BAOSBytesInput(ByteArrayOutputStream arrayOut) {
            this.arrayOut = arrayOut;
        }

        @Override
        public void writeAllTo(OutputStream out) throws IOException {
            this.arrayOut.writeTo(out);
        }

        @Override
        public long size() {
            return this.arrayOut.size();
        }
    }

    private static class CapacityBAOSBytesInput
    extends BytesInput {
        private final CapacityByteArrayOutputStream arrayOut;

        private CapacityBAOSBytesInput(CapacityByteArrayOutputStream arrayOut) {
            this.arrayOut = arrayOut;
        }

        @Override
        public void writeAllTo(OutputStream out) throws IOException {
            this.arrayOut.writeTo(out);
        }

        @Override
        public long size() {
            return this.arrayOut.size();
        }
    }

    private static class EmptyBytesInput
    extends BytesInput {
        private EmptyBytesInput() {
        }

        @Override
        public void writeAllTo(OutputStream out) throws IOException {
        }

        @Override
        public long size() {
            return 0L;
        }

        @Override
        public ByteBuffer toByteBuffer() throws IOException {
            return ByteBuffer.allocate(0);
        }
    }

    private static class UnsignedVarLongBytesInput
    extends BytesInput {
        private final long longValue;

        public UnsignedVarLongBytesInput(long longValue) {
            this.longValue = longValue;
        }

        @Override
        public void writeAllTo(OutputStream out) throws IOException {
            BytesUtils.writeUnsignedVarLong(this.longValue, out);
        }

        @Override
        public long size() {
            int s = (70 - Long.numberOfLeadingZeros(this.longValue)) / 7;
            return s == 0 ? 1L : (long)s;
        }
    }

    private static class UnsignedVarIntBytesInput
    extends BytesInput {
        private final int intValue;

        public UnsignedVarIntBytesInput(int intValue) {
            this.intValue = intValue;
        }

        @Override
        public void writeAllTo(OutputStream out) throws IOException {
            BytesUtils.writeUnsignedVarInt(this.intValue, out);
        }

        @Override
        public ByteBuffer toByteBuffer() throws IOException {
            ByteBuffer ret = ByteBuffer.allocate((int)this.size());
            BytesUtils.writeUnsignedVarInt(this.intValue, ret);
            return ret;
        }

        @Override
        public long size() {
            int s = (38 - Integer.numberOfLeadingZeros(this.intValue)) / 7;
            return s == 0 ? 1L : (long)s;
        }
    }

    private static class IntBytesInput
    extends BytesInput {
        private final int intValue;

        public IntBytesInput(int intValue) {
            this.intValue = intValue;
        }

        @Override
        public void writeAllTo(OutputStream out) throws IOException {
            BytesUtils.writeIntLittleEndian(out, this.intValue);
        }

        @Override
        public ByteBuffer toByteBuffer() throws IOException {
            return ByteBuffer.allocate(4).putInt(0, this.intValue);
        }

        @Override
        public long size() {
            return 4L;
        }
    }

    private static class SequenceBytesIn
    extends BytesInput {
        private static final Log LOG = Log.getLog(SequenceBytesIn.class);
        private final List<BytesInput> inputs;
        private final long size;

        private SequenceBytesIn(List<BytesInput> inputs) {
            this.inputs = inputs;
            long total = 0L;
            for (BytesInput input : inputs) {
                total += input.size();
            }
            this.size = total;
        }

        @Override
        public void writeAllTo(OutputStream out) throws IOException {
            for (BytesInput input : this.inputs) {
                input.writeAllTo(out);
            }
        }

        @Override
        public long size() {
            return this.size;
        }
    }

    private static class StreamBytesInput
    extends BytesInput {
        private static final Log LOG = Log.getLog(StreamBytesInput.class);
        private final InputStream in;
        private final int byteCount;

        private StreamBytesInput(InputStream in, int byteCount) {
            this.in = in;
            this.byteCount = byteCount;
        }

        @Override
        public void writeAllTo(OutputStream out) throws IOException {
            out.write(this.toByteArray());
        }

        @Override
        public byte[] toByteArray() throws IOException {
            byte[] buf = new byte[this.byteCount];
            new DataInputStream(this.in).readFully(buf);
            return buf;
        }

        @Override
        public long size() {
            return this.byteCount;
        }
    }

    private static final class BAOS
    extends ByteArrayOutputStream {
        private BAOS(int size) {
            super(size);
        }

        public byte[] getBuf() {
            return this.buf;
        }
    }
}

