/*
 * Decompiled with CFR 0.152.
 */
package org.apache.parquet.hadoop.util.wrapped.io;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.ByteBufferPool;
import org.apache.hadoop.io.ElasticByteBufferPool;
import org.apache.parquet.bytes.ByteBufferAllocator;
import org.apache.parquet.bytes.DirectByteBufferAllocator;
import org.apache.parquet.hadoop.util.wrapped.io.FutureIO;
import org.apache.parquet.hadoop.util.wrapped.io.VectorIoBridge;
import org.apache.parquet.io.ParquetFileRange;
import org.junit.After;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;

public class TestVectorIoBridge {
    private static final int DATASET_LEN = 65536;
    private static final byte[] DATASET = TestVectorIoBridge.dataset(65536, 97, 32);
    private static final String VECTORED_READ_FILE_NAME = "target/test/vectored_file.txt";
    private static final int VECTORED_READ_OPERATION_TEST_TIMEOUT_SECONDS = 300;
    private final Path vectoredPath = new Path("target/test/vectored_file.txt");
    private final ByteBufferPool pool = new ElasticByteBufferPool();
    private final ByteBufferAllocator allocate = new ByteBufferAllocator(){

        public ByteBuffer allocate(int size) {
            return TestVectorIoBridge.this.pool.getBuffer(false, size);
        }

        public void release(ByteBuffer b) {
            TestVectorIoBridge.this.pool.putBuffer(b);
        }

        public boolean isDirect() {
            return false;
        }
    };
    private FileSystem fileSystem;
    private Path testFilePath;
    private VectorIoBridge vectorIOBridge;
    private long initialVectorReadCount;
    private long initialBlocksRead;
    private long initialBytesRead;

    @Before
    public void setUp() throws IOException {
        Assume.assumeTrue((String)"Bridge not available", (boolean)VectorIoBridge.instance().available());
        this.fileSystem = FileSystem.getLocal((Configuration)new Configuration());
        this.testFilePath = this.fileSystem.makeQualified(this.vectoredPath);
        TestVectorIoBridge.createFile(this.fileSystem, this.testFilePath, DATASET);
        this.vectorIOBridge = VectorIoBridge.availableInstance();
        this.initialVectorReadCount = this.vectorIOBridge.getVectorReads();
        this.initialBlocksRead = this.vectorIOBridge.getBlocksRead();
        this.initialBytesRead = this.vectorIOBridge.getBytesRead();
    }

    @After
    public void tearDown() throws IOException {
        if (this.fileSystem != null) {
            this.fileSystem.delete(this.testFilePath, false);
        }
    }

    public FileSystem getFileSystem() {
        return this.fileSystem;
    }

    @Test
    public void testVectorIOBridgeAvailable() throws Throwable {
        Assert.assertTrue((String)"VectorIoBridge not available", (boolean)VectorIoBridge.bridgeAvailable());
    }

    private static byte[] dataset(int len, int base, int modulo) {
        byte[] dataset = new byte[len];
        for (int i = 0; i < len; ++i) {
            dataset[i] = (byte)(base + i % modulo);
        }
        return dataset;
    }

    public static void createFile(FileSystem fs, Path path, byte[] data) throws IOException {
        try (FSDataOutputStream stream = fs.create(path, true);){
            if (data != null && data.length > 0) {
                stream.write(data);
            }
        }
    }

    private FSDataInputStream openTestFile() throws IOException {
        return this.getFileSystem().open(this.testFilePath);
    }

    @Test
    public void testVectoredReadMultipleRanges() throws Exception {
        ArrayList<ParquetFileRange> fileRanges = new ArrayList<ParquetFileRange>();
        for (int i = 0; i < 10; ++i) {
            fileRanges.add(this.range(i * 100, 100));
        }
        try (FSDataInputStream in = this.openTestFile();){
            this.readVectored(in, fileRanges);
            CompletableFuture[] completableFutures = new CompletableFuture[fileRanges.size()];
            int i = 0;
            for (ParquetFileRange res : fileRanges) {
                completableFutures[i++] = res.getDataReadFuture();
            }
            CompletableFuture<Void> combinedFuture = CompletableFuture.allOf(completableFutures);
            combinedFuture.get();
            this.validateVectoredReadResult(fileRanges, DATASET);
        }
    }

    @Test
    public void testVectoredReadAndReadFully() throws Exception {
        int offset = 100;
        int length = 256;
        List<ParquetFileRange> fileRanges = this.ranges(100, 256);
        try (FSDataInputStream in = this.openTestFile();){
            this.readVectored(in, fileRanges);
            byte[] readFullRes = new byte[256];
            in.readFully(100L, readFullRes);
            ByteBuffer vecRes = (ByteBuffer)FutureIO.awaitFuture((Future)fileRanges.get(0).getDataReadFuture(), (long)300L, (TimeUnit)TimeUnit.SECONDS);
            TestVectorIoBridge.assertDatasetEquals(0, "readFully", vecRes, 256, readFullRes);
        }
    }

    @Test
    public void testDisjointRanges() throws Exception {
        List<ParquetFileRange> fileRanges = this.ranges(0, 100, 4101, 100, 16101, 100);
        try (FSDataInputStream in = this.openTestFile();){
            this.readVectored(in, fileRanges);
            this.validateVectoredReadResult(fileRanges, DATASET);
        }
    }

    @Test
    public void testStreamImplementsReadVectored() throws Exception {
        try (FSDataInputStream in = this.openTestFile();){
            boolean streamDoesNativeVectorIo = VectorIoBridge.instance().hasCapability(in, "in:readvectored");
            Assert.assertTrue((String)("capability in:readvectored not supported by " + in), (boolean)streamDoesNativeVectorIo);
        }
    }

    @Test
    public void testAllRangesMergedIntoOne() throws Exception {
        List<ParquetFileRange> fileRanges = this.ranges(0, 100, 4101, 100, 16101, 100);
        try (FSDataInputStream in = this.openTestFile();){
            this.readVectored(in, fileRanges);
            this.validateVectoredReadResult(fileRanges, DATASET);
        }
    }

    @Test
    public void testSomeRangesMergedSomeUnmerged() throws Exception {
        List<ParquetFileRange> fileRanges = this.ranges(8192, 100, 14336, 100, 10240, 100, 1947, 100, 40960, 1024);
        try (FSDataInputStream in = this.openTestFile();){
            this.readVectored(in, fileRanges);
            this.validateVectoredReadResult(fileRanges, DATASET);
        }
    }

    @Test
    public void testOverlappingRanges() throws Exception {
        this.verifyExceptionalVectoredRead(this.getSampleOverlappingRanges(), IllegalArgumentException.class);
    }

    @Test
    public void testSameRanges() throws Exception {
        this.verifyExceptionalVectoredRead(this.getSampleSameRanges(), IllegalArgumentException.class);
    }

    @Test
    public void testNullRangeList() throws Exception {
        this.verifyExceptionalVectoredRead(null, NullPointerException.class);
    }

    @Test
    public void testEmptyRangeList() throws Exception {
        ArrayList<ParquetFileRange> fileRanges = new ArrayList<ParquetFileRange>();
        try (FSDataInputStream in = this.openTestFile();){
            this.readVectored(in, fileRanges);
        }
    }

    @Test
    public void testSomeRandomNonOverlappingRanges() throws Exception {
        List<ParquetFileRange> fileRanges = this.ranges(500, 100, 1000, 200, 50, 10, 10, 5);
        try (FSDataInputStream in = this.openTestFile();){
            this.readVectored(in, fileRanges);
            this.validateVectoredReadResult(fileRanges, DATASET);
        }
    }

    @Test
    public void testConsecutiveRanges() throws Exception {
        List<ParquetFileRange> fileRanges = this.getConsecutiveRanges();
        try (FSDataInputStream in = this.openTestFile();){
            this.readVectored(in, fileRanges);
            this.validateVectoredReadResult(fileRanges, DATASET);
        }
    }

    @Test
    public void testNegativeLengthRange() throws Exception {
        this.verifyExceptionalVectoredRead(this.ranges(1, -50), IllegalArgumentException.class);
    }

    @Test
    public void testNegativeOffsetRange() throws Exception {
        this.verifyExceptionalVectoredRead(this.ranges(-1, 50), IllegalArgumentException.class);
    }

    @Test
    public void testNormalReadAfterVectoredRead() throws Exception {
        List<ParquetFileRange> fileRanges = this.getSampleNonOverlappingRanges();
        try (FSDataInputStream in = this.openTestFile();){
            this.readVectored(in, fileRanges);
            byte[] res = new byte[200];
            in.read(res, 0, 200);
            ByteBuffer buffer = ByteBuffer.wrap(res);
            TestVectorIoBridge.assertDatasetEquals(0, "normal_read", buffer, 200, DATASET);
            Assert.assertEquals((String)"Vectored read shouldn't change file pointer.", (long)200L, (long)in.getPos());
            this.validateVectoredReadResult(fileRanges, DATASET);
        }
    }

    @Test
    public void testVectoredReadAfterNormalRead() throws Exception {
        List<ParquetFileRange> fileRanges = this.getSampleNonOverlappingRanges();
        try (FSDataInputStream in = this.openTestFile();){
            byte[] res = new byte[200];
            in.read(res, 0, 200);
            ByteBuffer buffer = ByteBuffer.wrap(res);
            TestVectorIoBridge.assertDatasetEquals(0, "normal_read", buffer, 200, DATASET);
            Assert.assertEquals((String)"Vectored read shouldn't change file pointer.", (long)200L, (long)in.getPos());
            this.readVectored(in, fileRanges);
            this.validateVectoredReadResult(fileRanges, DATASET);
        }
    }

    @Test
    public void testMultipleVectoredReads() throws Exception {
        List<ParquetFileRange> fileRanges1 = this.getSampleNonOverlappingRanges();
        List<ParquetFileRange> fileRanges2 = this.getSampleNonOverlappingRanges();
        try (FSDataInputStream in = this.openTestFile();){
            this.readVectored(in, fileRanges1);
            this.readVectored(in, fileRanges2);
            this.validateVectoredReadResult(fileRanges2, DATASET);
            this.validateVectoredReadResult(fileRanges1, DATASET);
        }
    }

    @Test
    public void testDirectBufferReadRejected() throws Exception {
        this.verifyExceptionalVectoredRead(this.getSampleNonOverlappingRanges(), (ByteBufferAllocator)DirectByteBufferAllocator.getInstance(), UnsupportedOperationException.class);
    }

    @Test
    public void testDirectBufferReadReportedAsUnavailable() throws Exception {
        try (FSDataInputStream in = this.openTestFile();){
            Assert.assertFalse((String)"Direct buffer read should not be available", (boolean)VectorIoBridge.instance().readVectoredAvailable(in, (ByteBufferAllocator)DirectByteBufferAllocator.getInstance()));
        }
    }

    private void readVectored(FSDataInputStream in, List<ParquetFileRange> fileRanges) throws IOException {
        VectorIoBridge.instance().readVectoredRanges(in, fileRanges, this.allocate);
    }

    private ParquetFileRange range(long offset, int length) {
        return new ParquetFileRange(offset, length);
    }

    private List<ParquetFileRange> ranges(int ... args) {
        int len = args.length;
        Assert.assertEquals((String)("range argument length of " + len + " is not even"), (long)0L, (long)(len & 1));
        ArrayList<ParquetFileRange> fileRanges = new ArrayList<ParquetFileRange>();
        for (int i = 0; i < len; i += 2) {
            fileRanges.add(this.range(args[i], args[i + 1]));
        }
        return fileRanges;
    }

    protected List<ParquetFileRange> getSampleNonOverlappingRanges() {
        return this.ranges(0, 100, 110, 50);
    }

    protected List<ParquetFileRange> getSampleOverlappingRanges() {
        return this.ranges(100, 500, 400, 500);
    }

    protected List<ParquetFileRange> getConsecutiveRanges() {
        return this.ranges(100, 500, 600, 500);
    }

    protected List<ParquetFileRange> getSampleSameRanges() {
        return this.ranges(8000, 1000, 8000, 1000, 8000, 1000);
    }

    public static void assertDatasetEquals(int readOffset, String operation, ByteBuffer data, int length, byte[] originalData) {
        for (int i = 0; i < length; ++i) {
            int o = readOffset + i;
            Assert.assertEquals((String)(operation + " with read offset " + readOffset + ": data[" + i + "] != DATASET[" + o + "]"), (long)originalData[o], (long)data.get());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void validateVectoredReadResult(List<ParquetFileRange> fileRanges, byte[] originalData) throws IOException, TimeoutException {
        CompletableFuture[] completableFutures = new CompletableFuture[fileRanges.size()];
        int i = 0;
        for (ParquetFileRange res : fileRanges) {
            completableFutures[i++] = res.getDataReadFuture();
        }
        CompletableFuture<Void> combinedFuture = CompletableFuture.allOf(completableFutures);
        FutureIO.awaitFuture(combinedFuture, (long)300L, (TimeUnit)TimeUnit.SECONDS);
        for (ParquetFileRange res : fileRanges) {
            CompletableFuture data = res.getDataReadFuture();
            ByteBuffer buffer = (ByteBuffer)FutureIO.awaitFuture((Future)data, (long)300L, (TimeUnit)TimeUnit.SECONDS);
            try {
                TestVectorIoBridge.assertDatasetEquals((int)res.getOffset(), "vecRead", buffer, res.getLength(), originalData);
            }
            finally {
                this.pool.putBuffer(buffer);
            }
        }
    }

    protected <T extends Throwable> T verifyExceptionalVectoredRead(List<ParquetFileRange> fileRanges, Class<T> clazz) throws IOException {
        return this.verifyExceptionalVectoredRead(fileRanges, this.allocate, clazz);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private <T extends Throwable> T verifyExceptionalVectoredRead(List<ParquetFileRange> fileRanges, ByteBufferAllocator allocator, Class<T> clazz) throws IOException {
        try (FSDataInputStream in = this.openTestFile();){
            VectorIoBridge.instance().readVectoredRanges(in, fileRanges, allocator);
            Assert.fail((String)("expected error reading " + in));
            T t = null;
            return t;
        }
        catch (AssertionError e) {
            throw e;
        }
        catch (Exception e) {
            if (clazz.isAssignableFrom(e.getClass())) return (T)e;
            throw e;
        }
    }
}

