/*
 * Copyright (c) 2010, 2019 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

package org.apache.hbase.thirdparty.org.glassfish.jersey.server.internal.inject;

import java.util.List;
import java.util.function.Function;

import org.apache.hbase.thirdparty.javax.ws.rs.MatrixParam;
import org.apache.hbase.thirdparty.javax.ws.rs.core.PathSegment;

import javax.inject.Provider;
import javax.inject.Singleton;

import org.apache.hbase.thirdparty.org.glassfish.jersey.internal.inject.ExtractorException;
import org.apache.hbase.thirdparty.org.glassfish.jersey.server.ContainerRequest;
import org.apache.hbase.thirdparty.org.glassfish.jersey.server.ParamException;
import org.apache.hbase.thirdparty.org.glassfish.jersey.server.model.Parameter;

/**
 * Value supplier provider supporting the {@link MatrixParam &#64;MatrixParam} injection annotation.
 *
 * @author Paul Sandoz
 * @author Marek Potociar
 */
@Singleton
final class MatrixParamValueParamProvider extends AbstractValueParamProvider {

    /**
     * Injection constructor.
     *
     * @param mpep multivalued map parameter extractor provider.
     */
    public MatrixParamValueParamProvider(Provider<MultivaluedParameterExtractorProvider> mpep) {
        super(mpep, Parameter.Source.MATRIX);
    }

    @Override
    public Function<ContainerRequest, ?> createValueProvider(Parameter parameter) {
        String parameterName = parameter.getSourceName();
        if (parameterName == null || parameterName.length() == 0) {
            // Invalid header parameter name
            return null;
        }

        MultivaluedParameterExtractor e = get(parameter);
        if (e == null) {
            return null;
        }

        return new MatrixParamValueProvider(e, !parameter.isEncoded());
    }

    private static final class MatrixParamValueProvider implements Function<ContainerRequest, Object> {

        private final MultivaluedParameterExtractor<?> extractor;
        private final boolean decode;

        MatrixParamValueProvider(MultivaluedParameterExtractor<?> extractor, boolean decode) {
            this.extractor = extractor;
            this.decode = decode;
        }

        @Override
        public Object apply(ContainerRequest containerRequest) {
            List<PathSegment> l = containerRequest.getUriInfo().getPathSegments(decode);
            PathSegment p = l.get(l.size() - 1);
            try {
                return extractor.extract(p.getMatrixParameters());
            } catch (ExtractorException e) {
                throw new ParamException.MatrixParamException(e.getCause(),
                        extractor.getName(), extractor.getDefaultValueString());
            }
        }
    }
}
