/* Generic definitions */
/* Assertions (useful to generate conditional code) */
/* Current type and class (and size, if applicable) */
/* Value methods */
/* Interfaces (keys) */
/* Interfaces (values) */
/* Abstract implementations (keys) */
/* Abstract implementations (values) */
/* Static containers (keys) */
/* Static containers (values) */
/* Implementations */
/* Synchronized wrappers */
/* Unmodifiable wrappers */
/* Other wrappers */
/* Methods (keys) */
/* Methods (values) */
/* Methods (keys/values) */
/* Methods that have special names depending on keys (but the special names depend on values) */
/* Equality */
/* Object/Reference-only definitions (keys) */
/* Primitive-type-only definitions (keys) */
/* Object/Reference-only definitions (values) */
/* Primitive-type-only definitions (values) */
/*		 
 * Copyright (C) 2002-2015 Sebastiano Vigna
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License. 
 */
package it.unimi.dsi.fastutil.floats;

import it.unimi.dsi.fastutil.Hash;
import it.unimi.dsi.fastutil.HashCommon;
import static it.unimi.dsi.fastutil.HashCommon.arraySize;
import static it.unimi.dsi.fastutil.HashCommon.maxFill;
import java.util.Map;
import java.util.Arrays;
import java.util.NoSuchElementException;
import it.unimi.dsi.fastutil.chars.CharCollection;
import it.unimi.dsi.fastutil.chars.AbstractCharCollection;
import it.unimi.dsi.fastutil.chars.CharIterator;
import java.util.Comparator;
import it.unimi.dsi.fastutil.chars.CharListIterator;
import it.unimi.dsi.fastutil.objects.AbstractObjectSortedSet;
import it.unimi.dsi.fastutil.objects.ObjectListIterator;
import it.unimi.dsi.fastutil.objects.ObjectBidirectionalIterator;
import it.unimi.dsi.fastutil.objects.ObjectSortedSet;

/** A type-specific linked hash map with with a fast, small-footprint implementation.
 *
 * <P>Instances of this class use a hash table to represent a map. The table is filled up to a specified <em>load factor</em>, and then doubled in size to accommodate new entries. If the table is
 * emptied below <em>one fourth</em> of the load factor, it is halved in size. However, halving is not performed when deleting entries from an iterator, as it would interfere with the iteration
 * process.
 *
 * <p>Note that {@link #clear()} does not modify the hash table size. Rather, a family of {@linkplain #trim() trimming methods} lets you control the size of the table; this is particularly useful if
 * you reuse instances of this class.
 *
 * <P>Iterators generated by this map will enumerate pairs in the same order in which they have been added to the map (addition of pairs whose key is already present in the set does not change the
 * iteration order). Note that this order has nothing in common with the natural order of the keys. The order is kept by means of a doubly linked list, represented <i>via</i> an array of longs
 * parallel to the table.
 *
 * <P>This class implements the interface of a sorted map, so to allow easy access of the iteration order: for instance, you can get the first key in iteration order with {@link #firstKey()} without
 * having to create an iterator; however, this class partially violates the {@link java.util.SortedMap} contract because all submap methods throw an exception and {@link #comparator()} returns always
 * <code>null</code>.
 *
 * <p>Additional methods, such as <code>getAndMoveToFirst()</code>, make it easy to use instances of this class as a cache (e.g., with LRU policy).
 *
 * <P>The iterators provided by the views of this class using are type-specific {@linkplain java.util.ListIterator list iterators}, and can be started at any element <em>which is a key of the map</em>,
 * or a {@link NoSuchElementException} exception will be thrown. If, however, the provided element is not the first or last key in the set, the first access to the list index will require linear time,
 * as in the worst case the entire key set must be scanned in iteration order to retrieve the positional index of the starting key. If you use just the methods of a type-specific
 * {@link it.unimi.dsi.fastutil.BidirectionalIterator}, however, all operations will be performed in constant time.
 *
 * @see Hash
 * @see HashCommon */
public class Float2CharLinkedOpenHashMap extends AbstractFloat2CharSortedMap implements java.io.Serializable, Cloneable, Hash {
	private static final long serialVersionUID = 0L;
	private static final boolean ASSERTS = false;
	/** The array of keys. */
	protected transient float[] key;
	/** The array of values. */
	protected transient char[] value;
	/** The mask for wrapping a position counter. */
	protected transient int mask;
	/** Whether this set contains the key zero. */
	protected transient boolean containsNullKey;
	/** The index of the first entry in iteration order. It is valid iff {@link #size} is nonzero; otherwise, it contains -1. */
	protected transient int first = -1;
	/** The index of the last entry in iteration order. It is valid iff {@link #size} is nonzero; otherwise, it contains -1. */
	protected transient int last = -1;
	/** For each entry, the next and the previous entry in iteration order, stored as <code>((prev & 0xFFFFFFFFL) << 32) | (next & 0xFFFFFFFFL)</code>. The first entry contains predecessor -1, and the
	 * last entry contains successor -1. */
	protected transient long[] link;
	/** The current table size. */
	protected transient int n;
	/** Threshold after which we rehash. It must be the table size times {@link #f}. */
	protected transient int maxFill;
	/** Number of entries in the set (including the key zero, if present). */
	protected int size;
	/** The acceptable load factor. */
	protected final float f;
	/** Cached set of entries. */
	protected transient volatile FastSortedEntrySet entries;
	/** Cached set of keys. */
	protected transient volatile FloatSortedSet keys;
	/** Cached collection of values. */
	protected transient volatile CharCollection values;

	/** Creates a new hash map.
	 *
	 * <p>The actual table size will be the least power of two greater than <code>expected</code>/<code>f</code>.
	 *
	 * @param expected the expected number of elements in the hash set.
	 * @param f the load factor. */

	public Float2CharLinkedOpenHashMap( final int expected, final float f ) {
		if ( f <= 0 || f > 1 ) throw new IllegalArgumentException( "Load factor must be greater than 0 and smaller than or equal to 1" );
		if ( expected < 0 ) throw new IllegalArgumentException( "The expected number of elements must be nonnegative" );
		this.f = f;
		n = arraySize( expected, f );
		mask = n - 1;
		maxFill = maxFill( n, f );
		key = new float[ n + 1 ];
		value = new char[ n + 1 ];
		link = new long[ n + 1 ];
	}

	/** Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor.
	 *
	 * @param expected the expected number of elements in the hash map. */
	public Float2CharLinkedOpenHashMap( final int expected ) {
		this( expected, DEFAULT_LOAD_FACTOR );
	}

	/** Creates a new hash map with initial expected {@link Hash#DEFAULT_INITIAL_SIZE} entries and {@link Hash#DEFAULT_LOAD_FACTOR} as load factor. */
	public Float2CharLinkedOpenHashMap() {
		this( DEFAULT_INITIAL_SIZE, DEFAULT_LOAD_FACTOR );
	}

	/** Creates a new hash map copying a given one.
	 *
	 * @param m a {@link Map} to be copied into the new hash map.
	 * @param f the load factor. */
	public Float2CharLinkedOpenHashMap( final Map<? extends Float, ? extends Character> m, final float f ) {
		this( m.size(), f );
		putAll( m );
	}

	/** Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor copying a given one.
	 *
	 * @param m a {@link Map} to be copied into the new hash map. */
	public Float2CharLinkedOpenHashMap( final Map<? extends Float, ? extends Character> m ) {
		this( m, DEFAULT_LOAD_FACTOR );
	}

	/** Creates a new hash map copying a given type-specific one.
	 *
	 * @param m a type-specific map to be copied into the new hash map.
	 * @param f the load factor. */
	public Float2CharLinkedOpenHashMap( final Float2CharMap m, final float f ) {
		this( m.size(), f );
		putAll( m );
	}

	/** Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor copying a given type-specific one.
	 *
	 * @param m a type-specific map to be copied into the new hash map. */
	public Float2CharLinkedOpenHashMap( final Float2CharMap m ) {
		this( m, DEFAULT_LOAD_FACTOR );
	}

	/** Creates a new hash map using the elements of two parallel arrays.
	 *
	 * @param k the array of keys of the new hash map.
	 * @param v the array of corresponding values in the new hash map.
	 * @param f the load factor.
	 * @throws IllegalArgumentException if <code>k</code> and <code>v</code> have different lengths. */
	public Float2CharLinkedOpenHashMap( final float[] k, final char[] v, final float f ) {
		this( k.length, f );
		if ( k.length != v.length ) throw new IllegalArgumentException( "The key array and the value array have different lengths (" + k.length + " and " + v.length + ")" );
		for ( int i = 0; i < k.length; i++ )
			this.put( k[ i ], v[ i ] );
	}

	/** Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor using the elements of two parallel arrays.
	 *
	 * @param k the array of keys of the new hash map.
	 * @param v the array of corresponding values in the new hash map.
	 * @throws IllegalArgumentException if <code>k</code> and <code>v</code> have different lengths. */
	public Float2CharLinkedOpenHashMap( final float[] k, final char[] v ) {
		this( k, v, DEFAULT_LOAD_FACTOR );
	}

	private int realSize() {
		return containsNullKey ? size - 1 : size;
	}

	private void ensureCapacity( final int capacity ) {
		final int needed = arraySize( capacity, f );
		if ( needed > n ) rehash( needed );
	}

	private void tryCapacity( final long capacity ) {
		final int needed = (int)Math.min( 1 << 30, Math.max( 2, HashCommon.nextPowerOfTwo( (long)Math.ceil( capacity / f ) ) ) );
		if ( needed > n ) rehash( needed );
	}

	private char removeEntry( final int pos ) {
		final char oldValue = value[ pos ];
		size--;
		fixPointers( pos );
		shiftKeys( pos );
		if ( size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE ) rehash( n / 2 );
		return oldValue;
	}

	private char removeNullEntry() {
		containsNullKey = false;
		final char oldValue = value[ n ];
		size--;
		fixPointers( n );
		if ( size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE ) rehash( n / 2 );
		return oldValue;
	}

	/** {@inheritDoc} */
	public void putAll( Map<? extends Float, ? extends Character> m ) {
		if ( f <= .5 ) ensureCapacity( m.size() ); // The resulting map will be sized for m.size() elements
		else tryCapacity( size() + m.size() ); // The resulting map will be tentatively sized for size() + m.size() elements
		super.putAll( m );
	}

	private int insert( final float k, final char v ) {
		int pos;
		if ( ( Float.floatToIntBits( k ) == 0 ) ) {
			if ( containsNullKey ) return n;
			containsNullKey = true;
			pos = n;
		}
		else {
			float curr;
			final float[] key = this.key;
			// The starting point.
			if ( !( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) {
				if ( ( Float.floatToIntBits( curr ) == Float.floatToIntBits( k ) ) ) return pos;
				while ( !( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) )
					if ( ( Float.floatToIntBits( curr ) == Float.floatToIntBits( k ) ) ) return pos;
			}
			key[ pos ] = k;
		}
		value[ pos ] = v;
		if ( size == 0 ) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER( link[ pos ], -1, -1 );
			link[ pos ] = -1L;
		}
		else {
			link[ last ] ^= ( ( link[ last ] ^ ( pos & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
			link[ pos ] = ( ( last & 0xFFFFFFFFL ) << 32 ) | ( -1 & 0xFFFFFFFFL );
			last = pos;
		}
		if ( size++ >= maxFill ) rehash( arraySize( size + 1, f ) );
		if ( ASSERTS ) checkTable();
		return -1;
	}

	public char put( final float k, final char v ) {
		final int pos = insert( k, v );
		if ( pos < 0 ) return defRetValue;
		final char oldValue = value[ pos ];
		value[ pos ] = v;
		return oldValue;
	}

	public Character put( final Float ok, final Character ov ) {
		final char v = ( ( ov ).charValue() );
		final int pos = insert( ( ( ok ).floatValue() ), v );
		if ( pos < 0 ) return ( null );
		final char oldValue = value[ pos ];
		value[ pos ] = v;
		return ( Character.valueOf( oldValue ) );
	}

	/** Shifts left entries with the specified hash code, starting at the specified position, and empties the resulting free entry.
	 *
	 * @param pos a starting position. */
	protected final void shiftKeys( int pos ) {
		// Shift entries with the same hash.
		int last, slot;
		float curr;
		final float[] key = this.key;
		for ( ;; ) {
			pos = ( ( last = pos ) + 1 ) & mask;
			for ( ;; ) {
				if ( ( Float.floatToIntBits( curr = key[ pos ] ) == 0 ) ) {
					key[ last ] = ( 0 );
					return;
				}
				slot = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( curr ) ) & mask;
				if ( last <= pos ? last >= slot || slot > pos : last >= slot && slot > pos ) break;
				pos = ( pos + 1 ) & mask;
			}
			key[ last ] = curr;
			value[ last ] = value[ pos ];
			fixPointers( pos, last );
		}
	}

	public char remove( final float k ) {
		if ( ( Float.floatToIntBits( k ) == 0 ) ) {
			if ( containsNullKey ) return removeNullEntry();
			return defRetValue;
		}
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ( ( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) return defRetValue;
		if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) return removeEntry( pos );
		while ( true ) {
			if ( ( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) ) return defRetValue;
			if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) return removeEntry( pos );
		}
	}

	public Character remove( final Object ok ) {
		final float k = ( ( ( (Float)( ok ) ).floatValue() ) );
		if ( ( Float.floatToIntBits( k ) == 0 ) ) {
			if ( containsNullKey ) return ( Character.valueOf( removeNullEntry() ) );
			return ( null );
		}
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ( ( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) return ( null );
		if ( ( Float.floatToIntBits( curr ) == Float.floatToIntBits( k ) ) ) return ( Character.valueOf( removeEntry( pos ) ) );
		while ( true ) {
			if ( ( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) ) return ( null );
			if ( ( Float.floatToIntBits( curr ) == Float.floatToIntBits( k ) ) ) return ( Character.valueOf( removeEntry( pos ) ) );
		}
	}

	private char setValue( final int pos, final char v ) {
		final char oldValue = value[ pos ];
		value[ pos ] = v;
		return oldValue;
	}

	/** Removes the mapping associated with the first key in iteration order.
	 * 
	 * @return the value previously associated with the first key in iteration order.
	 * @throws NoSuchElementException is this map is empty. */
	public char removeFirstChar() {
		if ( size == 0 ) throw new NoSuchElementException();
		final int pos = first;
		// Abbreviated version of fixPointers(pos)
		first = (int)link[ pos ];
		if ( 0 <= first ) {
			// Special case of SET_PREV( link[ first ], -1 )
			link[ first ] |= ( -1 & 0xFFFFFFFFL ) << 32;
		}
		size--;
		final char v = value[ pos ];
		if ( pos == n ) {
			containsNullKey = false;
		}
		else shiftKeys( pos );
		if ( size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE ) rehash( n / 2 );
		return v;
	}

	/** Removes the mapping associated with the last key in iteration order.
	 * 
	 * @return the value previously associated with the last key in iteration order.
	 * @throws NoSuchElementException is this map is empty. */
	public char removeLastChar() {
		if ( size == 0 ) throw new NoSuchElementException();
		final int pos = last;
		// Abbreviated version of fixPointers(pos)
		last = (int)( link[ pos ] >>> 32 );
		if ( 0 <= last ) {
			// Special case of SET_NEXT( link[ last ], -1 )
			link[ last ] |= -1 & 0xFFFFFFFFL;
		}
		size--;
		final char v = value[ pos ];
		if ( pos == n ) {
			containsNullKey = false;
		}
		else shiftKeys( pos );
		if ( size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE ) rehash( n / 2 );
		return v;
	}

	private void moveIndexToFirst( final int i ) {
		if ( size == 1 || first == i ) return;
		if ( last == i ) {
			last = (int)( link[ i ] >>> 32 );
			// Special case of SET_NEXT( link[ last ], -1 );
			link[ last ] |= -1 & 0xFFFFFFFFL;
		}
		else {
			final long linki = link[ i ];
			final int prev = (int)( linki >>> 32 );
			final int next = (int)linki;
			link[ prev ] ^= ( ( link[ prev ] ^ ( linki & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
			link[ next ] ^= ( ( link[ next ] ^ ( linki & 0xFFFFFFFF00000000L ) ) & 0xFFFFFFFF00000000L );
		}
		link[ first ] ^= ( ( link[ first ] ^ ( ( i & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
		link[ i ] = ( ( -1 & 0xFFFFFFFFL ) << 32 ) | ( first & 0xFFFFFFFFL );
		first = i;
	}

	private void moveIndexToLast( final int i ) {
		if ( size == 1 || last == i ) return;
		if ( first == i ) {
			first = (int)link[ i ];
			// Special case of SET_PREV( link[ first ], -1 );
			link[ first ] |= ( -1 & 0xFFFFFFFFL ) << 32;
		}
		else {
			final long linki = link[ i ];
			final int prev = (int)( linki >>> 32 );
			final int next = (int)linki;
			link[ prev ] ^= ( ( link[ prev ] ^ ( linki & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
			link[ next ] ^= ( ( link[ next ] ^ ( linki & 0xFFFFFFFF00000000L ) ) & 0xFFFFFFFF00000000L );
		}
		link[ last ] ^= ( ( link[ last ] ^ ( i & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
		link[ i ] = ( ( last & 0xFFFFFFFFL ) << 32 ) | ( -1 & 0xFFFFFFFFL );
		last = i;
	}

	/** Returns the value to which the given key is mapped; if the key is present, it is moved to the first position of the iteration order.
	 *
	 * @param k the key.
	 * @return the corresponding value, or the {@linkplain #defaultReturnValue() default return value} if no value was present for the given key. */
	public char getAndMoveToFirst( final float k ) {
		if ( ( Float.floatToIntBits( k ) == 0 ) ) {
			if ( containsNullKey ) {
				moveIndexToFirst( n );
				return value[ n ];
			}
			return defRetValue;
		}
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ( ( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) return defRetValue;
		if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) {
			moveIndexToFirst( pos );
			return value[ pos ];
		}
		// There's always an unused entry.
		while ( true ) {
			if ( ( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) ) return defRetValue;
			if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) {
				moveIndexToFirst( pos );
				return value[ pos ];
			}
		}
	}

	/** Returns the value to which the given key is mapped; if the key is present, it is moved to the last position of the iteration order.
	 *
	 * @param k the key.
	 * @return the corresponding value, or the {@linkplain #defaultReturnValue() default return value} if no value was present for the given key. */
	public char getAndMoveToLast( final float k ) {
		if ( ( Float.floatToIntBits( k ) == 0 ) ) {
			if ( containsNullKey ) {
				moveIndexToLast( n );
				return value[ n ];
			}
			return defRetValue;
		}
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ( ( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) return defRetValue;
		if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) {
			moveIndexToLast( pos );
			return value[ pos ];
		}
		// There's always an unused entry.
		while ( true ) {
			if ( ( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) ) return defRetValue;
			if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) {
				moveIndexToLast( pos );
				return value[ pos ];
			}
		}
	}

	/** Adds a pair to the map; if the key is already present, it is moved to the first position of the iteration order.
	 *
	 * @param k the key.
	 * @param v the value.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value was present for the given key. */
	public char putAndMoveToFirst( final float k, final char v ) {
		int pos;
		if ( ( Float.floatToIntBits( k ) == 0 ) ) {
			if ( containsNullKey ) {
				moveIndexToFirst( n );
				return setValue( n, v );
			}
			containsNullKey = true;
			pos = n;
		}
		else {
			float curr;
			final float[] key = this.key;
			// The starting point.
			if ( !( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) {
				if ( ( Float.floatToIntBits( curr ) == Float.floatToIntBits( k ) ) ) {
					moveIndexToFirst( pos );
					return setValue( pos, v );
				}
				while ( !( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) )
					if ( ( Float.floatToIntBits( curr ) == Float.floatToIntBits( k ) ) ) {
						moveIndexToFirst( pos );
						return setValue( pos, v );
					}
			}
			key[ pos ] = k;
		}
		value[ pos ] = v;
		if ( size == 0 ) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER( link[ pos ], -1, -1 );
			link[ pos ] = -1L;
		}
		else {
			link[ first ] ^= ( ( link[ first ] ^ ( ( pos & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
			link[ pos ] = ( ( -1 & 0xFFFFFFFFL ) << 32 ) | ( first & 0xFFFFFFFFL );
			first = pos;
		}
		if ( size++ >= maxFill ) rehash( arraySize( size, f ) );
		if ( ASSERTS ) checkTable();
		return defRetValue;
	}

	/** Adds a pair to the map; if the key is already present, it is moved to the last position of the iteration order.
	 *
	 * @param k the key.
	 * @param v the value.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value was present for the given key. */
	public char putAndMoveToLast( final float k, final char v ) {
		int pos;
		if ( ( Float.floatToIntBits( k ) == 0 ) ) {
			if ( containsNullKey ) {
				moveIndexToLast( n );
				return setValue( n, v );
			}
			containsNullKey = true;
			pos = n;
		}
		else {
			float curr;
			final float[] key = this.key;
			// The starting point.
			if ( !( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) {
				if ( ( Float.floatToIntBits( curr ) == Float.floatToIntBits( k ) ) ) {
					moveIndexToLast( pos );
					return setValue( pos, v );
				}
				while ( !( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) )
					if ( ( Float.floatToIntBits( curr ) == Float.floatToIntBits( k ) ) ) {
						moveIndexToLast( pos );
						return setValue( pos, v );
					}
			}
			key[ pos ] = k;
		}
		value[ pos ] = v;
		if ( size == 0 ) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER( link[ pos ], -1, -1 );
			link[ pos ] = -1L;
		}
		else {
			link[ last ] ^= ( ( link[ last ] ^ ( pos & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
			link[ pos ] = ( ( last & 0xFFFFFFFFL ) << 32 ) | ( -1 & 0xFFFFFFFFL );
			last = pos;
		}
		if ( size++ >= maxFill ) rehash( arraySize( size, f ) );
		if ( ASSERTS ) checkTable();
		return defRetValue;
	}

	public Character get( final Float ok ) {
		final float k = ( ( ok ).floatValue() );
		if ( ( Float.floatToIntBits( k ) == 0 ) ) return containsNullKey ? ( Character.valueOf( value[ n ] ) ) : ( null );
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ( ( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) return ( null );
		if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) return ( Character.valueOf( value[ pos ] ) );
		// There's always an unused entry.
		while ( true ) {
			if ( ( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) ) return ( null );
			if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) return ( Character.valueOf( value[ pos ] ) );
		}
	}

	public char get( final float k ) {
		if ( ( Float.floatToIntBits( k ) == 0 ) ) return containsNullKey ? value[ n ] : defRetValue;
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ( ( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) return defRetValue;
		if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) return value[ pos ];
		// There's always an unused entry.
		while ( true ) {
			if ( ( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) ) return defRetValue;
			if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) return value[ pos ];
		}
	}

	public boolean containsKey( final float k ) {
		if ( ( Float.floatToIntBits( k ) == 0 ) ) return containsNullKey;
		float curr;
		final float[] key = this.key;
		int pos;
		// The starting point.
		if ( ( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) return false;
		if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) return true;
		// There's always an unused entry.
		while ( true ) {
			if ( ( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) ) return false;
			if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) return true;
		}
	}

	public boolean containsValue( final char v ) {
		final char value[] = this.value;
		final float key[] = this.key;
		if ( containsNullKey && ( ( value[ n ] ) == ( v ) ) ) return true;
		for ( int i = n; i-- != 0; )
			if ( !( Float.floatToIntBits( key[ i ] ) == 0 ) && ( ( value[ i ] ) == ( v ) ) ) return true;
		return false;
	}

	/* Removes all elements from this map.
	 * 
	 * <P>To increase object reuse, this method does not change the table size. If you want to reduce the table size, you must use {@link #trim()}. */
	public void clear() {
		if ( size == 0 ) return;
		size = 0;
		containsNullKey = false;
		Arrays.fill( key, ( 0 ) );
		first = last = -1;
	}

	public int size() {
		return size;
	}

	public boolean isEmpty() {
		return size == 0;
	}

	/** A no-op for backward compatibility.
	 * 
	 * @param growthFactor unused.
	 * @deprecated Since <code>fastutil</code> 6.1.0, hash tables are doubled when they are too full. */
	@Deprecated
	public void growthFactor( int growthFactor ) {}

	/** Gets the growth factor (2).
	 *
	 * @return the growth factor of this set, which is fixed (2).
	 * @see #growthFactor(int)
	 * @deprecated Since <code>fastutil</code> 6.1.0, hash tables are doubled when they are too full. */
	@Deprecated
	public int growthFactor() {
		return 16;
	}

	/** The entry class for a hash map does not record key and value, but rather the position in the hash table of the corresponding entry. This is necessary so that calls to
	 * {@link java.util.Map.Entry#setValue(Object)} are reflected in the map */
	final class MapEntry implements Float2CharMap.Entry, Map.Entry<Float, Character> {
		// The table index this entry refers to, or -1 if this entry has been deleted.
		int index;

		MapEntry( final int index ) {
			this.index = index;
		}

		MapEntry() {}

		public Float getKey() {
			return ( Float.valueOf( key[ index ] ) );
		}

		public float getFloatKey() {
			return key[ index ];
		}

		public Character getValue() {
			return ( Character.valueOf( value[ index ] ) );
		}

		public char getCharValue() {
			return value[ index ];
		}

		public char setValue( final char v ) {
			final char oldValue = value[ index ];
			value[ index ] = v;
			return oldValue;
		}

		public Character setValue( final Character v ) {
			return ( Character.valueOf( setValue( ( ( v ).charValue() ) ) ) );
		}

		@SuppressWarnings("unchecked")
		public boolean equals( final Object o ) {
			if ( !( o instanceof Map.Entry ) ) return false;
			Map.Entry<Float, Character> e = (Map.Entry<Float, Character>)o;
			return ( Float.floatToIntBits( key[ index ] ) == Float.floatToIntBits( ( ( e.getKey() ).floatValue() ) ) ) && ( ( value[ index ] ) == ( ( ( e.getValue() ).charValue() ) ) );
		}

		public int hashCode() {
			return it.unimi.dsi.fastutil.HashCommon.float2int( key[ index ] ) ^ ( value[ index ] );
		}

		public String toString() {
			return key[ index ] + "=>" + value[ index ];
		}
	}

	/** Modifies the {@link #link} vector so that the given entry is removed. This method will complete in constant time.
	 *
	 * @param i the index of an entry. */
	protected void fixPointers( final int i ) {
		if ( size == 0 ) {
			first = last = -1;
			return;
		}
		if ( first == i ) {
			first = (int)link[ i ];
			if ( 0 <= first ) {
				// Special case of SET_PREV( link[ first ], -1 )
				link[ first ] |= ( -1 & 0xFFFFFFFFL ) << 32;
			}
			return;
		}
		if ( last == i ) {
			last = (int)( link[ i ] >>> 32 );
			if ( 0 <= last ) {
				// Special case of SET_NEXT( link[ last ], -1 )
				link[ last ] |= -1 & 0xFFFFFFFFL;
			}
			return;
		}
		final long linki = link[ i ];
		final int prev = (int)( linki >>> 32 );
		final int next = (int)linki;
		link[ prev ] ^= ( ( link[ prev ] ^ ( linki & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
		link[ next ] ^= ( ( link[ next ] ^ ( linki & 0xFFFFFFFF00000000L ) ) & 0xFFFFFFFF00000000L );
	}

	/** Modifies the {@link #link} vector for a shift from s to d. <P>This method will complete in constant time.
	 *
	 * @param s the source position.
	 * @param d the destination position. */
	protected void fixPointers( int s, int d ) {
		if ( size == 1 ) {
			first = last = d;
			// Special case of SET_UPPER_LOWER( link[ d ], -1, -1 )
			link[ d ] = -1L;
			return;
		}
		if ( first == s ) {
			first = d;
			link[ (int)link[ s ] ] ^= ( ( link[ (int)link[ s ] ] ^ ( ( d & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
			link[ d ] = link[ s ];
			return;
		}
		if ( last == s ) {
			last = d;
			link[ (int)( link[ s ] >>> 32 ) ] ^= ( ( link[ (int)( link[ s ] >>> 32 ) ] ^ ( d & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
			link[ d ] = link[ s ];
			return;
		}
		final long links = link[ s ];
		final int prev = (int)( links >>> 32 );
		final int next = (int)links;
		link[ prev ] ^= ( ( link[ prev ] ^ ( d & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
		link[ next ] ^= ( ( link[ next ] ^ ( ( d & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
		link[ d ] = links;
	}

	/** Returns the first key of this map in iteration order.
	 *
	 * @return the first key in iteration order. */
	public float firstFloatKey() {
		if ( size == 0 ) throw new NoSuchElementException();
		return key[ first ];
	}

	/** Returns the last key of this map in iteration order.
	 *
	 * @return the last key in iteration order. */
	public float lastFloatKey() {
		if ( size == 0 ) throw new NoSuchElementException();
		return key[ last ];
	}

	public FloatComparator comparator() {
		return null;
	}

	public Float2CharSortedMap tailMap( float from ) {
		throw new UnsupportedOperationException();
	}

	public Float2CharSortedMap headMap( float to ) {
		throw new UnsupportedOperationException();
	}

	public Float2CharSortedMap subMap( float from, float to ) {
		throw new UnsupportedOperationException();
	}

	/** A list iterator over a linked map.
	 *
	 * <P>This class provides a list iterator over a linked hash map. The constructor runs in constant time. */
	private class MapIterator {
		/** The entry that will be returned by the next call to {@link java.util.ListIterator#previous()} (or <code>null</code> if no previous entry exists). */
		int prev = -1;
		/** The entry that will be returned by the next call to {@link java.util.ListIterator#next()} (or <code>null</code> if no next entry exists). */
		int next = -1;
		/** The last entry that was returned (or -1 if we did not iterate or used {@link java.util.Iterator#remove()}). */
		int curr = -1;
		/** The current index (in the sense of a {@link java.util.ListIterator}). Note that this value is not meaningful when this iterator has been created using the nonempty constructor. */
		int index = -1;

		private MapIterator() {
			next = first;
			index = 0;
		}

		private MapIterator( final float from ) {
			if ( ( Float.floatToIntBits( from ) == 0 ) ) {
				if ( Float2CharLinkedOpenHashMap.this.containsNullKey ) {
					next = (int)link[ n ];
					prev = n;
					return;
				}
				else throw new NoSuchElementException( "The key " + from + " does not belong to this map." );
			}
			if ( ( Float.floatToIntBits( key[ last ] ) == Float.floatToIntBits( from ) ) ) {
				prev = last;
				index = size;
				return;
			}
			// The starting point.
			int pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( from ) ) & mask;
			// There's always an unused entry.
			while ( !( Float.floatToIntBits( key[ pos ] ) == 0 ) ) {
				if ( ( Float.floatToIntBits( key[ pos ] ) == Float.floatToIntBits( from ) ) ) {
					// Note: no valid index known.
					next = (int)link[ pos ];
					prev = pos;
					return;
				}
				pos = ( pos + 1 ) & mask;
			}
			throw new NoSuchElementException( "The key " + from + " does not belong to this map." );
		}

		public boolean hasNext() {
			return next != -1;
		}

		public boolean hasPrevious() {
			return prev != -1;
		}

		private final void ensureIndexKnown() {
			if ( index >= 0 ) return;
			if ( prev == -1 ) {
				index = 0;
				return;
			}
			if ( next == -1 ) {
				index = size;
				return;
			}
			int pos = first;
			index = 1;
			while ( pos != prev ) {
				pos = (int)link[ pos ];
				index++;
			}
		}

		public int nextIndex() {
			ensureIndexKnown();
			return index;
		}

		public int previousIndex() {
			ensureIndexKnown();
			return index - 1;
		}

		public int nextEntry() {
			if ( !hasNext() ) throw new NoSuchElementException();
			curr = next;
			next = (int)link[ curr ];
			prev = curr;
			if ( index >= 0 ) index++;
			return curr;
		}

		public int previousEntry() {
			if ( !hasPrevious() ) throw new NoSuchElementException();
			curr = prev;
			prev = (int)( link[ curr ] >>> 32 );
			next = curr;
			if ( index >= 0 ) index--;
			return curr;
		}

		public void remove() {
			ensureIndexKnown();
			if ( curr == -1 ) throw new IllegalStateException();
			if ( curr == prev ) {
				/*
				 * If the last operation was a next(), we are removing an entry that preceeds the current index, and thus we must decrement it. */
				index--;
				prev = (int)( link[ curr ] >>> 32 );
			}
			else next = (int)link[ curr ];
			size--;
			/*
			 * Now we manually fix the pointers. Because of our knowledge of next and prev, this is going to be faster than calling fixPointers(). */
			if ( prev == -1 ) first = next;
			else link[ prev ] ^= ( ( link[ prev ] ^ ( next & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
			if ( next == -1 ) last = prev;
			else link[ next ] ^= ( ( link[ next ] ^ ( ( prev & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
			int last, slot, pos = curr;
			curr = -1;
			if ( pos == n ) {
				Float2CharLinkedOpenHashMap.this.containsNullKey = false;
			}
			else {
				float curr;
				final float[] key = Float2CharLinkedOpenHashMap.this.key;
				// We have to horribly duplicate the shiftKeys() code because we need to update next/prev.
				for ( ;; ) {
					pos = ( ( last = pos ) + 1 ) & mask;
					for ( ;; ) {
						if ( ( Float.floatToIntBits( curr = key[ pos ] ) == 0 ) ) {
							key[ last ] = ( 0 );
							return;
						}
						slot = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( curr ) ) & mask;
						if ( last <= pos ? last >= slot || slot > pos : last >= slot && slot > pos ) break;
						pos = ( pos + 1 ) & mask;
					}
					key[ last ] = curr;
					value[ last ] = value[ pos ];
					if ( next == pos ) next = last;
					if ( prev == pos ) prev = last;
					fixPointers( pos, last );
				}
			}
		}

		public int skip( final int n ) {
			int i = n;
			while ( i-- != 0 && hasNext() )
				nextEntry();
			return n - i - 1;
		}

		public int back( final int n ) {
			int i = n;
			while ( i-- != 0 && hasPrevious() )
				previousEntry();
			return n - i - 1;
		}
	}

	private class EntryIterator extends MapIterator implements ObjectListIterator<Float2CharMap.Entry> {
		private MapEntry entry;

		public EntryIterator() {}

		public EntryIterator( float from ) {
			super( from );
		}

		public MapEntry next() {
			return entry = new MapEntry( nextEntry() );
		}

		public MapEntry previous() {
			return entry = new MapEntry( previousEntry() );
		}

		@Override
		public void remove() {
			super.remove();
			entry.index = -1; // You cannot use a deleted entry.
		}

		public void set( Float2CharMap.Entry ok ) {
			throw new UnsupportedOperationException();
		}

		public void add( Float2CharMap.Entry ok ) {
			throw new UnsupportedOperationException();
		}
	}

	private class FastEntryIterator extends MapIterator implements ObjectListIterator<Float2CharMap.Entry> {
		final MapEntry entry = new MapEntry();

		public FastEntryIterator() {}

		public FastEntryIterator( float from ) {
			super( from );
		}

		public MapEntry next() {
			entry.index = nextEntry();
			return entry;
		}

		public MapEntry previous() {
			entry.index = previousEntry();
			return entry;
		}

		public void set( Float2CharMap.Entry ok ) {
			throw new UnsupportedOperationException();
		}

		public void add( Float2CharMap.Entry ok ) {
			throw new UnsupportedOperationException();
		}
	}

	private final class MapEntrySet extends AbstractObjectSortedSet<Float2CharMap.Entry> implements FastSortedEntrySet {
		public ObjectBidirectionalIterator<Float2CharMap.Entry> iterator() {
			return new EntryIterator();
		}

		public Comparator<? super Float2CharMap.Entry> comparator() {
			return null;
		}

		public ObjectSortedSet<Float2CharMap.Entry> subSet( Float2CharMap.Entry fromElement, Float2CharMap.Entry toElement ) {
			throw new UnsupportedOperationException();
		}

		public ObjectSortedSet<Float2CharMap.Entry> headSet( Float2CharMap.Entry toElement ) {
			throw new UnsupportedOperationException();
		}

		public ObjectSortedSet<Float2CharMap.Entry> tailSet( Float2CharMap.Entry fromElement ) {
			throw new UnsupportedOperationException();
		}

		public Float2CharMap.Entry first() {
			if ( size == 0 ) throw new NoSuchElementException();
			return new MapEntry( Float2CharLinkedOpenHashMap.this.first );
		}

		public Float2CharMap.Entry last() {
			if ( size == 0 ) throw new NoSuchElementException();
			return new MapEntry( Float2CharLinkedOpenHashMap.this.last );
		}

		@SuppressWarnings("unchecked")
		public boolean contains( final Object o ) {
			if ( !( o instanceof Map.Entry ) ) return false;
			final Map.Entry<Float, Character> e = (Map.Entry<Float, Character>)o;
			final float k = ( ( e.getKey() ).floatValue() );
			if ( ( Float.floatToIntBits( k ) == 0 ) ) return ( Float2CharLinkedOpenHashMap.this.containsNullKey && ( ( value[ n ] ) == ( ( ( e.getValue() ).charValue() ) ) ) );
			float curr;
			final float[] key = Float2CharLinkedOpenHashMap.this.key;
			int pos;
			// The starting point.
			if ( ( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) return false;
			if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) return ( ( value[ pos ] ) == ( ( ( e.getValue() ).charValue() ) ) );
			// There's always an unused entry.
			while ( true ) {
				if ( ( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) ) return false;
				if ( ( Float.floatToIntBits( k ) == Float.floatToIntBits( curr ) ) ) return ( ( value[ pos ] ) == ( ( ( e.getValue() ).charValue() ) ) );
			}
		}

		@SuppressWarnings("unchecked")
		public boolean remove( final Object o ) {
			if ( !( o instanceof Map.Entry ) ) return false;
			final Map.Entry<Float, Character> e = (Map.Entry<Float, Character>)o;
			final float k = ( ( e.getKey() ).floatValue() );
			final char v = ( ( e.getValue() ).charValue() );
			if ( ( Float.floatToIntBits( k ) == 0 ) ) {
				if ( containsNullKey && ( ( value[ n ] ) == ( v ) ) ) {
					removeNullEntry();
					return true;
				}
				return false;
			}
			float curr;
			final float[] key = Float2CharLinkedOpenHashMap.this.key;
			int pos;
			// The starting point.
			if ( ( Float.floatToIntBits( curr = key[ pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask ] ) == 0 ) ) return false;
			if ( ( Float.floatToIntBits( curr ) == Float.floatToIntBits( k ) ) ) {
				if ( ( ( value[ pos ] ) == ( v ) ) ) {
					removeEntry( pos );
					return true;
				}
				return false;
			}
			while ( true ) {
				if ( ( Float.floatToIntBits( curr = key[ pos = ( pos + 1 ) & mask ] ) == 0 ) ) return false;
				if ( ( Float.floatToIntBits( curr ) == Float.floatToIntBits( k ) ) ) {
					if ( ( ( value[ pos ] ) == ( v ) ) ) {
						removeEntry( pos );
						return true;
					}
				}
			}
		}

		public int size() {
			return size;
		}

		public void clear() {
			Float2CharLinkedOpenHashMap.this.clear();
		}

		public ObjectBidirectionalIterator<Float2CharMap.Entry> iterator( final Float2CharMap.Entry from ) {
			return new EntryIterator( ( ( from.getKey() ).floatValue() ) );
		}

		public ObjectBidirectionalIterator<Float2CharMap.Entry> fastIterator() {
			return new FastEntryIterator();
		}

		public ObjectBidirectionalIterator<Float2CharMap.Entry> fastIterator( final Float2CharMap.Entry from ) {
			return new FastEntryIterator( ( ( from.getKey() ).floatValue() ) );
		}
	}

	public FastSortedEntrySet float2CharEntrySet() {
		if ( entries == null ) entries = new MapEntrySet();
		return entries;
	}

	/** An iterator on keys.
	 *
	 * <P>We simply override the {@link java.util.ListIterator#next()}/{@link java.util.ListIterator#previous()} methods (and possibly their type-specific counterparts) so that they return keys
	 * instead of entries. */
	private final class KeyIterator extends MapIterator implements FloatListIterator {
		public KeyIterator( final float k ) {
			super( k );
		}

		public float previousFloat() {
			return key[ previousEntry() ];
		}

		public void set( float k ) {
			throw new UnsupportedOperationException();
		}

		public void add( float k ) {
			throw new UnsupportedOperationException();
		}

		public Float previous() {
			return ( Float.valueOf( key[ previousEntry() ] ) );
		}

		public void set( Float ok ) {
			throw new UnsupportedOperationException();
		}

		public void add( Float ok ) {
			throw new UnsupportedOperationException();
		}

		public KeyIterator() {
			super();
		}

		public float nextFloat() {
			return key[ nextEntry() ];
		}

		public Float next() {
			return ( Float.valueOf( key[ nextEntry() ] ) );
		}
	}

	private final class KeySet extends AbstractFloatSortedSet {
		public FloatListIterator iterator( final float from ) {
			return new KeyIterator( from );
		}

		public FloatListIterator iterator() {
			return new KeyIterator();
		}

		public int size() {
			return size;
		}

		public boolean contains( float k ) {
			return containsKey( k );
		}

		public boolean remove( float k ) {
			final int oldSize = size;
			Float2CharLinkedOpenHashMap.this.remove( k );
			return size != oldSize;
		}

		public void clear() {
			Float2CharLinkedOpenHashMap.this.clear();
		}

		public float firstFloat() {
			if ( size == 0 ) throw new NoSuchElementException();
			return key[ first ];
		}

		public float lastFloat() {
			if ( size == 0 ) throw new NoSuchElementException();
			return key[ last ];
		}

		public FloatComparator comparator() {
			return null;
		}

		final public FloatSortedSet tailSet( float from ) {
			throw new UnsupportedOperationException();
		}

		final public FloatSortedSet headSet( float to ) {
			throw new UnsupportedOperationException();
		}

		final public FloatSortedSet subSet( float from, float to ) {
			throw new UnsupportedOperationException();
		}
	}

	public FloatSortedSet keySet() {
		if ( keys == null ) keys = new KeySet();
		return keys;
	}

	/** An iterator on values.
	 *
	 * <P>We simply override the {@link java.util.ListIterator#next()}/{@link java.util.ListIterator#previous()} methods (and possibly their type-specific counterparts) so that they return values
	 * instead of entries. */
	private final class ValueIterator extends MapIterator implements CharListIterator {
		public char previousChar() {
			return value[ previousEntry() ];
		}

		public Character previous() {
			return ( Character.valueOf( value[ previousEntry() ] ) );
		}

		public void set( Character ok ) {
			throw new UnsupportedOperationException();
		}

		public void add( Character ok ) {
			throw new UnsupportedOperationException();
		}

		public void set( char v ) {
			throw new UnsupportedOperationException();
		}

		public void add( char v ) {
			throw new UnsupportedOperationException();
		}

		public ValueIterator() {
			super();
		}

		public char nextChar() {
			return value[ nextEntry() ];
		}

		public Character next() {
			return ( Character.valueOf( value[ nextEntry() ] ) );
		}
	}

	public CharCollection values() {
		if ( values == null ) values = new AbstractCharCollection() {
			public CharIterator iterator() {
				return new ValueIterator();
			}

			public int size() {
				return size;
			}

			public boolean contains( char v ) {
				return containsValue( v );
			}

			public void clear() {
				Float2CharLinkedOpenHashMap.this.clear();
			}
		};
		return values;
	}

	/** A no-op for backward compatibility. The kind of tables implemented by this class never need rehashing.
	 *
	 * <P>If you need to reduce the table size to fit exactly this set, use {@link #trim()}.
	 *
	 * @return true.
	 * @see #trim()
	 * @deprecated A no-op. */
	@Deprecated
	public boolean rehash() {
		return true;
	}

	/** Rehashes the map, making the table as small as possible.
	 * 
	 * <P>This method rehashes the table to the smallest size satisfying the load factor. It can be used when the set will not be changed anymore, so to optimize access speed and size.
	 *
	 * <P>If the table size is already the minimum possible, this method does nothing.
	 *
	 * @return true if there was enough memory to trim the map.
	 * @see #trim(int) */
	public boolean trim() {
		final int l = arraySize( size, f );
		if ( l >= n ) return true;
		try {
			rehash( l );
		}
		catch ( OutOfMemoryError cantDoIt ) {
			return false;
		}
		return true;
	}

	/** Rehashes this map if the table is too large.
	 * 
	 * <P>Let <var>N</var> be the smallest table size that can hold <code>max(n,{@link #size()})</code> entries, still satisfying the load factor. If the current table size is smaller than or equal to
	 * <var>N</var>, this method does nothing. Otherwise, it rehashes this map in a table of size <var>N</var>.
	 *
	 * <P>This method is useful when reusing maps. {@linkplain #clear() Clearing a map} leaves the table size untouched. If you are reusing a map many times, you can call this method with a typical
	 * size to avoid keeping around a very large table just because of a few large transient maps.
	 *
	 * @param n the threshold for the trimming.
	 * @return true if there was enough memory to trim the map.
	 * @see #trim() */
	public boolean trim( final int n ) {
		final int l = HashCommon.nextPowerOfTwo( (int)Math.ceil( n / f ) );
		if ( this.n <= l ) return true;
		try {
			rehash( l );
		}
		catch ( OutOfMemoryError cantDoIt ) {
			return false;
		}
		return true;
	}

	/** Rehashes the map.
	 *
	 * <P>This method implements the basic rehashing strategy, and may be overriden by subclasses implementing different rehashing strategies (e.g., disk-based rehashing). However, you should not
	 * override this method unless you understand the internal workings of this class.
	 *
	 * @param newN the new size */

	protected void rehash( final int newN ) {
		final float key[] = this.key;
		final char value[] = this.value;
		final int mask = newN - 1; // Note that this is used by the hashing macro
		final float newKey[] = new float[ newN + 1 ];
		final char newValue[] = new char[ newN + 1 ];
		int i = first, prev = -1, newPrev = -1, t, pos;
		final long link[] = this.link;
		final long newLink[] = new long[ newN + 1 ];
		first = -1;
		for ( int j = size; j-- != 0; ) {
			if ( ( Float.floatToIntBits( key[ i ] ) == 0 ) ) pos = newN;
			else {
				pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( key[ i ] ) ) & mask;
				while ( !( Float.floatToIntBits( newKey[ pos ] ) == 0 ) )
					pos = ( pos + 1 ) & mask;
				newKey[ pos ] = key[ i ];
			}
			newValue[ pos ] = value[ i ];
			if ( prev != -1 ) {
				newLink[ newPrev ] ^= ( ( newLink[ newPrev ] ^ ( pos & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
				newLink[ pos ] ^= ( ( newLink[ pos ] ^ ( ( newPrev & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
				newPrev = pos;
			}
			else {
				newPrev = first = pos;
				// Special case of SET(newLink[ pos ], -1, -1);
				newLink[ pos ] = -1L;
			}
			t = i;
			i = (int)link[ i ];
			prev = t;
		}
		this.link = newLink;
		this.last = newPrev;
		if ( newPrev != -1 )
		// Special case of SET_NEXT( newLink[ newPrev ], -1 );
		newLink[ newPrev ] |= -1 & 0xFFFFFFFFL;
		n = newN;
		this.mask = mask;
		maxFill = maxFill( n, f );
		this.key = newKey;
		this.value = newValue;
	}

	/** Returns a deep copy of this map.
	 *
	 * <P>This method performs a deep copy of this hash map; the data stored in the map, however, is not cloned. Note that this makes a difference only for object keys.
	 *
	 * @return a deep copy of this map. */

	public Float2CharLinkedOpenHashMap clone() {
		Float2CharLinkedOpenHashMap c;
		try {
			c = (Float2CharLinkedOpenHashMap)super.clone();
		}
		catch ( CloneNotSupportedException cantHappen ) {
			throw new InternalError();
		}
		c.keys = null;
		c.values = null;
		c.entries = null;
		c.containsNullKey = containsNullKey;
		c.key = key.clone();
		c.value = value.clone();
		c.link = link.clone();
		return c;
	}

	/** Returns a hash code for this map.
	 *
	 * This method overrides the generic method provided by the superclass. Since <code>equals()</code> is not overriden, it is important that the value returned by this method is the same value as
	 * the one returned by the overriden method.
	 *
	 * @return a hash code for this map. */
	public int hashCode() {
		int h = 0;
		for ( int j = realSize(), i = 0, t = 0; j-- != 0; ) {
			while ( ( Float.floatToIntBits( key[ i ] ) == 0 ) )
				i++;
			t = it.unimi.dsi.fastutil.HashCommon.float2int( key[ i ] );
			t ^= ( value[ i ] );
			h += t;
			i++;
		}
		// Zero / null keys have hash zero.
		if ( containsNullKey ) h += ( value[ n ] );
		return h;
	}

	private void writeObject( java.io.ObjectOutputStream s ) throws java.io.IOException {
		final float key[] = this.key;
		final char value[] = this.value;
		final MapIterator i = new MapIterator();
		s.defaultWriteObject();
		for ( int j = size, e; j-- != 0; ) {
			e = i.nextEntry();
			s.writeFloat( key[ e ] );
			s.writeChar( value[ e ] );
		}
	}

	private void readObject( java.io.ObjectInputStream s ) throws java.io.IOException, ClassNotFoundException {
		s.defaultReadObject();
		n = arraySize( size, f );
		maxFill = maxFill( n, f );
		mask = n - 1;
		final float key[] = this.key = new float[ n + 1 ];
		final char value[] = this.value = new char[ n + 1 ];
		final long link[] = this.link = new long[ n + 1 ];
		int prev = -1;
		first = last = -1;
		float k;
		char v;
		for ( int i = size, pos; i-- != 0; ) {
			k = s.readFloat();
			v = s.readChar();
			if ( ( Float.floatToIntBits( k ) == 0 ) ) {
				pos = n;
				containsNullKey = true;
			}
			else {
				pos = it.unimi.dsi.fastutil.HashCommon.mix( it.unimi.dsi.fastutil.HashCommon.float2int( k ) ) & mask;
				while ( !( Float.floatToIntBits( key[ pos ] ) == 0 ) )
					pos = ( pos + 1 ) & mask;
				key[ pos ] = k;
			}
			value[ pos ] = v;
			if ( first != -1 ) {
				link[ prev ] ^= ( ( link[ prev ] ^ ( pos & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
				link[ pos ] ^= ( ( link[ pos ] ^ ( ( prev & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
				prev = pos;
			}
			else {
				prev = first = pos;
				// Special case of SET_PREV( newLink[ pos ], -1 );
				link[ pos ] |= ( -1L & 0xFFFFFFFFL ) << 32;
			}
		}
		last = prev;
		if ( prev != -1 )
		// Special case of SET_NEXT( link[ prev ], -1 );
		link[ prev ] |= -1 & 0xFFFFFFFFL;
		if ( ASSERTS ) checkTable();
	}

	private void checkTable() {}
}
