/*
 * Copyright (C) 2012 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hbase.thirdparty.org.glassfish.org.apache.hbase.thirdparty.jersey.internal.guava;

import java.util.AbstractCollection;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import static org.apache.hbase.thirdparty.org.glassfish.org.apache.hbase.thirdparty.jersey.internal.guava.Preconditions.checkNotNull;

/**
 * A skeleton {@code Multimap} implementation, not necessarily in terms of a {@code Map}.
 *
 * @author Louis Wasserman
 */
abstract class AbstractMultimap<K, V> implements Multimap<K, V> {
    private transient Collection<Entry<K, V>> entries;
    private transient Set<K> keySet;
    private transient Collection<V> values;
    private transient Map<K, Collection<V>> asMap;

    @Override
    public boolean containsValue(Object value) {
        for (Collection<V> collection : asMap().values()) {
            if (collection.contains(value)) {
                return true;
            }
        }

        return false;
    }

    @Override
    public boolean containsEntry(Object key, Object value) {
        Collection<V> collection = asMap().get(key);
        return collection != null && collection.contains(value);
    }

    @Override
    public boolean remove(Object key, Object value) {
        Collection<V> collection = asMap().get(key);
        return collection != null && collection.remove(value);
    }

    @Override
    public boolean put(K key, V value) {
        return get(key).add(value);
    }

    @Override
    public boolean putAll(K key, Iterable<? extends V> values) {
        checkNotNull(values);
        // make sure we only call values.iterator() once
        // and we only call get(key) if values is nonempty
        if (values instanceof Collection) {
            Collection<? extends V> valueCollection = (Collection<? extends V>) values;
            return !valueCollection.isEmpty() && get(key).addAll(valueCollection);
        } else {
            Iterator<? extends V> valueItr = values.iterator();
            return valueItr.hasNext() && Iterators.addAll(get(key), valueItr);
        }
    }

    @Override
    public Collection<Entry<K, V>> entries() {
        Collection<Entry<K, V>> result = entries;
        return (result == null) ? entries = createEntries() : result;
    }

    private Collection<Entry<K, V>> createEntries() {
        if (this instanceof SetMultimap) {
            return new EntrySet();
        } else {
            return new Entries();
        }
    }

    abstract Iterator<Entry<K, V>> entryIterator();

    @Override
    public Set<K> keySet() {
        Set<K> result = keySet;
        return (result == null) ? keySet = createKeySet() : result;
    }

    Set<K> createKeySet() {
        return new Maps.KeySet<K, Collection<V>>(asMap());
    }

    @Override
    public Collection<V> values() {
        Collection<V> result = values;
        return (result == null) ? values = createValues() : result;
    }

    private Collection<V> createValues() {
        return new Values();
    }

    Iterator<V> valueIterator() {
        return Maps.valueIterator(entries().iterator());
    }

    @Override
    public Map<K, Collection<V>> asMap() {
        Map<K, Collection<V>> result = asMap;
        return (result == null) ? asMap = createAsMap() : result;
    }

    abstract Map<K, Collection<V>> createAsMap();

    @Override
    public boolean equals(Object object) {
        return Multimaps.equalsImpl(this, object);
    }

    /**
     * Returns the hash code for this multimap.
     * <p>
     * <p>The hash code of a multimap is defined as the hash code of the map view,
     * as returned by {@link Multimap#asMap}.
     *
     * @see Map#hashCode
     */
    @Override
    public int hashCode() {
        return asMap().hashCode();
    }

    /**
     * Returns a string representation of the multimap, generated by calling
     * {@code toString} on the map returned by {@link Multimap#asMap}.
     *
     * @return a string representation of the multimap
     */
    @Override
    public String toString() {
        return asMap().toString();
    }

    // Comparison and hashing

    private class Entries extends Multimaps.Entries<K, V> {
        @Override
        Multimap<K, V> multimap() {
            return AbstractMultimap.this;
        }

        @Override
        public Iterator<Entry<K, V>> iterator() {
            return entryIterator();
        }
    }

    private class EntrySet extends Entries implements Set<Entry<K, V>> {
        @Override
        public int hashCode() {
            return Sets.hashCodeImpl(this);
        }

        @Override
        public boolean equals(Object obj) {
            return Sets.equalsImpl(this, obj);
        }
    }

    private class Values extends AbstractCollection<V> {
        @Override
        public Iterator<V> iterator() {
            return valueIterator();
        }

        @Override
        public int size() {
            return AbstractMultimap.this.size();
        }

        @Override
        public boolean contains(Object o) {
            return AbstractMultimap.this.containsValue(o);
        }

        @Override
        public void clear() {
            AbstractMultimap.this.clear();
        }
    }
}
