/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.segment.local.customobject;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Ticker;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Iterators;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Ordering;
import com.google.common.collect.PeekingIterator;
import com.google.common.util.concurrent.AtomicDouble;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

public class QuantileDigest {
    private static final int MAX_BITS = 64;
    private static final double MAX_SIZE_FACTOR = 1.5;
    private static final int HEADER_BYTE_SIZE = 44;
    private static final int NODE_BYTE_SIZE = 18;
    static final long RESCALE_THRESHOLD_SECONDS = 50L;
    static final double ZERO_WEIGHT_THRESHOLD = 1.0E-5;
    private final double _maxError;
    private final Ticker _ticker;
    private final double _alpha;
    private final boolean _compressAutomatically;
    private Node _root;
    private double _weightedCount;
    private long _max = Long.MIN_VALUE;
    private long _min = Long.MAX_VALUE;
    private long _landmarkInSeconds;
    private int _totalNodeCount = 0;
    private int _nonZeroNodeCount = 0;
    private int _compressions = 0;

    public QuantileDigest(double maxError) {
        this(maxError, 0.0);
    }

    public QuantileDigest(double maxError, double alpha) {
        this(maxError, alpha, Ticker.systemTicker(), true);
    }

    @VisibleForTesting
    QuantileDigest(double maxError, double alpha, Ticker ticker, boolean compressAutomatically) {
        Preconditions.checkArgument((maxError >= 0.0 && maxError <= 1.0 ? 1 : 0) != 0, (Object)"maxError must be in range [0, 1]");
        Preconditions.checkArgument((alpha >= 0.0 && alpha < 1.0 ? 1 : 0) != 0, (Object)"alpha must be in range [0, 1)");
        this._maxError = maxError;
        this._alpha = alpha;
        this._ticker = ticker;
        this._compressAutomatically = compressAutomatically;
        this._landmarkInSeconds = TimeUnit.NANOSECONDS.toSeconds(ticker.read());
    }

    public QuantileDigest(QuantileDigest quantileDigest) {
        this(quantileDigest.getMaxError(), quantileDigest.getAlpha());
        this.merge(quantileDigest);
    }

    public double getMaxError() {
        return this._maxError;
    }

    public double getAlpha() {
        return this._alpha;
    }

    public void add(long value) {
        this.add(value, 1L);
    }

    public void add(long value, long count) {
        Preconditions.checkArgument((count > 0L ? 1 : 0) != 0, (Object)"count must be > 0");
        long nowInSeconds = TimeUnit.NANOSECONDS.toSeconds(this._ticker.read());
        int maxExpectedNodeCount = 3 * this.calculateCompressionFactor();
        if (nowInSeconds - this._landmarkInSeconds >= 50L) {
            this.rescale(nowInSeconds);
            this.compress();
        } else if ((double)this._nonZeroNodeCount > 1.5 * (double)maxExpectedNodeCount && this._compressAutomatically) {
            this.compress();
        }
        double weight = this.weight(TimeUnit.NANOSECONDS.toSeconds(this._ticker.read())) * (double)count;
        this._max = Math.max(this._max, value);
        this._min = Math.min(this._min, value);
        this.insert(QuantileDigest.longToBits(value), weight);
    }

    public void merge(QuantileDigest other) {
        this.rescaleToCommonLandmark(this, other);
        this._root = this.merge(this._root, other._root);
        this._max = Math.max(this._max, other._max);
        this._min = Math.min(this._min, other._min);
        this.compress();
    }

    public List<Long> getQuantiles(List<Double> quantiles) {
        Preconditions.checkArgument((boolean)Ordering.natural().isOrdered(quantiles), (Object)"quantiles must be sorted in increasing order");
        for (double quantile : quantiles) {
            Preconditions.checkArgument((quantile >= 0.0 && quantile <= 1.0 ? 1 : 0) != 0, (Object)"quantile must be between [0,1]");
        }
        final ImmutableList.Builder builder = ImmutableList.builder();
        final PeekingIterator iterator = Iterators.peekingIterator(quantiles.iterator());
        this.postOrderTraversal(this._root, new Callback(){
            private double _sum = 0.0;

            @Override
            public boolean process(Node node) {
                this._sum += node._weightedCount;
                while (iterator.hasNext() && this._sum > (Double)iterator.peek() * QuantileDigest.this._weightedCount) {
                    iterator.next();
                    long value = Math.min(node.getUpperBound(), QuantileDigest.this._max);
                    builder.add((Object)value);
                }
                return iterator.hasNext();
            }
        });
        while (iterator.hasNext()) {
            builder.add((Object)this._max);
            iterator.next();
        }
        return builder.build();
    }

    public long getQuantile(double quantile) {
        return this.getQuantiles((List<Double>)ImmutableList.of((Object)quantile)).get(0);
    }

    public double getCount() {
        return this._weightedCount / this.weight(TimeUnit.NANOSECONDS.toSeconds(this._ticker.read()));
    }

    public List<Bucket> getHistogram(List<Long> bucketUpperBounds) {
        Preconditions.checkArgument((boolean)Ordering.natural().isOrdered(bucketUpperBounds), (Object)"buckets must be sorted in increasing order");
        final ImmutableList.Builder builder = ImmutableList.builder();
        final PeekingIterator iterator = Iterators.peekingIterator(bucketUpperBounds.iterator());
        final AtomicDouble sum = new AtomicDouble();
        final AtomicDouble lastSum = new AtomicDouble();
        final AtomicDouble bucketWeightedSum = new AtomicDouble();
        final double normalizationFactor = this.weight(TimeUnit.NANOSECONDS.toSeconds(this._ticker.read()));
        this.postOrderTraversal(this._root, new Callback(){

            @Override
            public boolean process(Node node) {
                while (iterator.hasNext() && (Long)iterator.peek() <= node.getUpperBound()) {
                    double bucketCount = sum.get() - lastSum.get();
                    Bucket bucket = new Bucket(bucketCount / normalizationFactor, bucketWeightedSum.get() / bucketCount);
                    builder.add((Object)bucket);
                    lastSum.set(sum.get());
                    bucketWeightedSum.set(0.0);
                    iterator.next();
                }
                bucketWeightedSum.addAndGet((double)node.getMiddle() * node._weightedCount);
                sum.addAndGet(node._weightedCount);
                return iterator.hasNext();
            }
        });
        while (iterator.hasNext()) {
            double bucketCount = sum.get() - lastSum.get();
            Bucket bucket = new Bucket(bucketCount / normalizationFactor, bucketWeightedSum.get() / bucketCount);
            builder.add((Object)bucket);
            iterator.next();
        }
        return builder.build();
    }

    public long getMin() {
        final AtomicLong chosen = new AtomicLong(this._min);
        this.postOrderTraversal(this._root, new Callback(){

            @Override
            public boolean process(Node node) {
                if (node._weightedCount >= 1.0E-5) {
                    chosen.set(node.getLowerBound());
                    return false;
                }
                return true;
            }
        }, TraversalOrder.FORWARD);
        return Math.max(this._min, chosen.get());
    }

    public long getMax() {
        final AtomicLong chosen = new AtomicLong(this._max);
        this.postOrderTraversal(this._root, new Callback(){

            @Override
            public boolean process(Node node) {
                if (node._weightedCount >= 1.0E-5) {
                    chosen.set(node.getUpperBound());
                    return false;
                }
                return true;
            }
        }, TraversalOrder.REVERSE);
        return Math.min(this._max, chosen.get());
    }

    public int getByteSize() {
        return 44 + this._totalNodeCount * 18;
    }

    public byte[] toBytes() {
        byte[] bytes = new byte[this.getByteSize()];
        ByteBuffer byteBuffer = ByteBuffer.wrap(bytes);
        byteBuffer.putDouble(this._maxError);
        byteBuffer.putDouble(this._alpha);
        byteBuffer.putLong(this._landmarkInSeconds);
        byteBuffer.putLong(this._min);
        byteBuffer.putLong(this._max);
        byteBuffer.putInt(this._totalNodeCount);
        this.postOrderTraversal(this._root, node -> {
            this.serializeNode(byteBuffer, node);
            return true;
        });
        return bytes;
    }

    private void serializeNode(ByteBuffer byteBuffer, Node node) {
        byte flags = 0;
        if (node._left != null) {
            flags = (byte)(flags | 1);
        }
        if (node._right != null) {
            flags = (byte)(flags | 2);
        }
        byteBuffer.put(flags);
        byteBuffer.put((byte)node._level);
        byteBuffer.putLong(node._bits);
        byteBuffer.putDouble(node._weightedCount);
    }

    public static QuantileDigest fromBytes(byte[] bytes) {
        return QuantileDigest.fromByteBuffer(ByteBuffer.wrap(bytes));
    }

    public static QuantileDigest fromByteBuffer(ByteBuffer byteBuffer) {
        int numNodes;
        double maxError = byteBuffer.getDouble();
        double alpha = byteBuffer.getDouble();
        QuantileDigest quantileDigest = new QuantileDigest(maxError, alpha);
        quantileDigest._landmarkInSeconds = byteBuffer.getLong();
        quantileDigest._min = byteBuffer.getLong();
        quantileDigest._max = byteBuffer.getLong();
        quantileDigest._totalNodeCount = numNodes = byteBuffer.getInt();
        if (numNodes == 0) {
            return quantileDigest;
        }
        Stack<Node> stack = new Stack<Node>();
        for (int i = 0; i < numNodes; ++i) {
            byte flags = byteBuffer.get();
            int level = byteBuffer.get() & 0xFF;
            long bits = byteBuffer.getLong();
            double weightedCount = byteBuffer.getDouble();
            Node node = new Node(bits, level, weightedCount);
            if ((flags & 2) != 0) {
                node._right = (Node)stack.pop();
            }
            if ((flags & 1) != 0) {
                node._left = (Node)stack.pop();
            }
            stack.push(node);
            quantileDigest._weightedCount += weightedCount;
            if (!(node._weightedCount >= 1.0E-5)) continue;
            ++quantileDigest._nonZeroNodeCount;
        }
        Preconditions.checkState((stack.size() == 1 ? 1 : 0) != 0, (Object)"Tree is corrupted, expecting a single root node");
        quantileDigest._root = (Node)stack.pop();
        return quantileDigest;
    }

    @VisibleForTesting
    int getTotalNodeCount() {
        return this._totalNodeCount;
    }

    @VisibleForTesting
    int getNonZeroNodeCount() {
        return this._nonZeroNodeCount;
    }

    @VisibleForTesting
    int getCompressions() {
        return this._compressions;
    }

    @VisibleForTesting
    void compress() {
        ++this._compressions;
        final int compressionFactor = this.calculateCompressionFactor();
        this.postOrderTraversal(this._root, new Callback(){

            @Override
            public boolean process(Node node) {
                if (node.isLeaf()) {
                    return true;
                }
                double leftWeight = 0.0;
                if (node._left != null) {
                    leftWeight = node._left._weightedCount;
                }
                double rightWeight = 0.0;
                if (node._right != null) {
                    rightWeight = node._right._weightedCount;
                }
                boolean shouldCompress = node._weightedCount + leftWeight + rightWeight < (double)((int)(QuantileDigest.this._weightedCount / (double)compressionFactor));
                double oldNodeWeight = node._weightedCount;
                if (shouldCompress || leftWeight < 1.0E-5) {
                    node._left = QuantileDigest.this.tryRemove(node._left);
                    QuantileDigest.this._weightedCount += leftWeight;
                    node._weightedCount += leftWeight;
                }
                if (shouldCompress || rightWeight < 1.0E-5) {
                    node._right = QuantileDigest.this.tryRemove(node._right);
                    QuantileDigest.this._weightedCount += rightWeight;
                    node._weightedCount += rightWeight;
                }
                if (oldNodeWeight < 1.0E-5 && node._weightedCount >= 1.0E-5) {
                    ++QuantileDigest.this._nonZeroNodeCount;
                }
                return true;
            }
        });
        if (this._root != null && this._root._weightedCount < 1.0E-5) {
            this._root = this.tryRemove(this._root);
        }
    }

    private double weight(long timestamp) {
        return Math.exp(this._alpha * (double)(timestamp - this._landmarkInSeconds));
    }

    private void rescale(long newLandmarkInSeconds) {
        final double factor = Math.exp(-this._alpha * (double)(newLandmarkInSeconds - this._landmarkInSeconds));
        this._weightedCount *= factor;
        this.postOrderTraversal(this._root, new Callback(){

            @Override
            public boolean process(Node node) {
                double oldWeight = node._weightedCount;
                node._weightedCount *= factor;
                if (oldWeight >= 1.0E-5 && node._weightedCount < 1.0E-5) {
                    --QuantileDigest.this._nonZeroNodeCount;
                }
                return true;
            }
        });
        this._landmarkInSeconds = newLandmarkInSeconds;
    }

    private int calculateCompressionFactor() {
        if (this._root == null) {
            return 1;
        }
        return Math.max((int)((double)(this._root._level + 1) / this._maxError), 1);
    }

    private void insert(long bits, double weight) {
        long lastBranch = 0L;
        Node parent = null;
        Node current = this._root;
        while (true) {
            if (current == null) {
                this.setChild(parent, lastBranch, this.createLeaf(bits, weight));
                return;
            }
            if (!QuantileDigest.inSameSubtree(bits, current._bits, current._level)) {
                this.setChild(parent, lastBranch, this.makeSiblings(current, this.createLeaf(bits, weight)));
                return;
            }
            if (current._level == 0 && current._bits == bits) {
                double oldWeight = current._weightedCount;
                current._weightedCount += weight;
                if (current._weightedCount >= 1.0E-5 && oldWeight < 1.0E-5) {
                    ++this._nonZeroNodeCount;
                }
                this._weightedCount += weight;
                return;
            }
            long branch = bits & current.getBranchMask();
            parent = current;
            lastBranch = branch;
            if (branch == 0L) {
                current = current._left;
                continue;
            }
            current = current._right;
        }
    }

    private void setChild(Node parent, long branch, Node child) {
        if (parent == null) {
            this._root = child;
        } else if (branch == 0L) {
            parent._left = child;
        } else {
            parent._right = child;
        }
    }

    private Node makeSiblings(Node node, Node sibling) {
        int parentLevel = 64 - Long.numberOfLeadingZeros(node._bits ^ sibling._bits);
        Node parent = this.createNode(node._bits, parentLevel, 0.0);
        long branch = sibling._bits & parent.getBranchMask();
        if (branch == 0L) {
            parent._left = sibling;
            parent._right = node;
        } else {
            parent._left = node;
            parent._right = sibling;
        }
        return parent;
    }

    private Node createLeaf(long bits, double weight) {
        return this.createNode(bits, 0, weight);
    }

    private Node createNode(long bits, int level, double weight) {
        this._weightedCount += weight;
        ++this._totalNodeCount;
        if (weight >= 1.0E-5) {
            ++this._nonZeroNodeCount;
        }
        return new Node(bits, level, weight);
    }

    private Node merge(Node node, Node other) {
        if (node == null) {
            return this.copyRecursive(other);
        }
        if (other == null) {
            return node;
        }
        if (!QuantileDigest.inSameSubtree(node._bits, other._bits, Math.max(node._level, other._level))) {
            return this.makeSiblings(node, this.copyRecursive(other));
        }
        if (node._level > other._level) {
            long branch = other._bits & node.getBranchMask();
            if (branch == 0L) {
                node._left = this.merge(node._left, other);
            } else {
                node._right = this.merge(node._right, other);
            }
            return node;
        }
        if (node._level < other._level) {
            Node result = this.createNode(other._bits, other._level, other._weightedCount);
            long branch = node._bits & other.getBranchMask();
            if (branch == 0L) {
                result._left = this.merge(node, other._left);
                result._right = this.copyRecursive(other._right);
            } else {
                result._left = this.copyRecursive(other._left);
                result._right = this.merge(node, other._right);
            }
            return result;
        }
        double oldWeight = node._weightedCount;
        this._weightedCount += other._weightedCount;
        node._weightedCount += other._weightedCount;
        node._left = this.merge(node._left, other._left);
        node._right = this.merge(node._right, other._right);
        if (oldWeight < 1.0E-5 && node._weightedCount >= 1.0E-5) {
            ++this._nonZeroNodeCount;
        }
        return node;
    }

    private static boolean inSameSubtree(long bitsA, long bitsB, int level) {
        return level == 64 || bitsA >>> level == bitsB >>> level;
    }

    private Node copyRecursive(Node node) {
        Node result = null;
        if (node != null) {
            result = this.createNode(node._bits, node._level, node._weightedCount);
            result._left = this.copyRecursive(node._left);
            result._right = this.copyRecursive(node._right);
        }
        return result;
    }

    private Node tryRemove(Node node) {
        if (node == null) {
            return null;
        }
        if (node._weightedCount >= 1.0E-5) {
            --this._nonZeroNodeCount;
        }
        this._weightedCount -= node._weightedCount;
        Node result = null;
        if (node.isLeaf()) {
            --this._totalNodeCount;
        } else if (node.hasSingleChild()) {
            result = node.getSingleChild();
            --this._totalNodeCount;
        } else {
            node._weightedCount = 0.0;
            result = node;
        }
        return result;
    }

    private boolean postOrderTraversal(Node node, Callback callback) {
        return this.postOrderTraversal(node, callback, TraversalOrder.FORWARD);
    }

    private boolean postOrderTraversal(Node node, Callback callback, TraversalOrder order) {
        Node second;
        Node first;
        if (node == null) {
            return false;
        }
        if (order == TraversalOrder.FORWARD) {
            first = node._left;
            second = node._right;
        } else {
            first = node._right;
            second = node._left;
        }
        if (first != null && !this.postOrderTraversal(first, callback, order)) {
            return false;
        }
        if (second != null && !this.postOrderTraversal(second, callback, order)) {
            return false;
        }
        return callback.process(node);
    }

    public double getConfidenceFactor() {
        return this.computeMaxPathWeight(this._root) * 1.0 / this._weightedCount;
    }

    public boolean equivalent(QuantileDigest other) {
        this.rescaleToCommonLandmark(this, other);
        return this._totalNodeCount == other._totalNodeCount && this._nonZeroNodeCount == other._nonZeroNodeCount && this._min == other._min && this._max == other._max && this._weightedCount == other._weightedCount && Objects.equal((Object)this._root, (Object)other._root);
    }

    private void rescaleToCommonLandmark(QuantileDigest one, QuantileDigest two) {
        long targetLandmark;
        long nowInSeconds = TimeUnit.NANOSECONDS.toSeconds(this._ticker.read());
        if (nowInSeconds - (targetLandmark = Math.max(one._landmarkInSeconds, two._landmarkInSeconds)) >= 50L) {
            targetLandmark = nowInSeconds;
        }
        if (targetLandmark != one._landmarkInSeconds) {
            one.rescale(targetLandmark);
        }
        if (targetLandmark != two._landmarkInSeconds) {
            two.rescale(targetLandmark);
        }
    }

    private double computeMaxPathWeight(Node node) {
        if (node == null || node._level == 0) {
            return 0.0;
        }
        double leftMaxWeight = this.computeMaxPathWeight(node._left);
        double rightMaxWeight = this.computeMaxPathWeight(node._right);
        return Math.max(leftMaxWeight, rightMaxWeight) + node._weightedCount;
    }

    @VisibleForTesting
    void validate() {
        final AtomicDouble sumOfWeights = new AtomicDouble();
        final AtomicInteger actualNodeCount = new AtomicInteger();
        final AtomicInteger actualNonZeroNodeCount = new AtomicInteger();
        if (this._root != null) {
            this.validateStructure(this._root);
            this.postOrderTraversal(this._root, new Callback(){

                @Override
                public boolean process(Node node) {
                    sumOfWeights.addAndGet(node._weightedCount);
                    actualNodeCount.incrementAndGet();
                    if (node._weightedCount >= 1.0E-5) {
                        actualNonZeroNodeCount.incrementAndGet();
                    }
                    return true;
                }
            });
        }
        Preconditions.checkState((Math.abs(sumOfWeights.get() - this._weightedCount) < 1.0E-5 ? 1 : 0) != 0, (String)"Computed weight (%s) doesn't match summary (%s)", (Object)sumOfWeights.get(), (Object)this._weightedCount);
        Preconditions.checkState((actualNodeCount.get() == this._totalNodeCount ? 1 : 0) != 0, (String)"Actual node count (%s) doesn't match summary (%s)", (int)actualNodeCount.get(), (int)this._totalNodeCount);
        Preconditions.checkState((actualNonZeroNodeCount.get() == this._nonZeroNodeCount ? 1 : 0) != 0, (String)"Actual non-zero node count (%s) doesn't match summary (%s)", (int)actualNonZeroNodeCount.get(), (int)this._nonZeroNodeCount);
    }

    private void validateStructure(Node node) {
        Preconditions.checkState((node._level >= 0 ? 1 : 0) != 0);
        if (node._left != null) {
            this.validateBranchStructure(node, node._left, node._right, true);
            this.validateStructure(node._left);
        }
        if (node._right != null) {
            this.validateBranchStructure(node, node._right, node._left, false);
            this.validateStructure(node._right);
        }
    }

    private void validateBranchStructure(Node parent, Node child, Node otherChild, boolean isLeft) {
        Preconditions.checkState((child._level < parent._level ? 1 : 0) != 0, (String)"Child level (%s) should be smaller than parent level (%s)", (int)child._level, (int)parent._level);
        long branch = child._bits & 1L << parent._level - 1;
        Preconditions.checkState((branch == 0L && isLeft || branch != 0L && !isLeft ? 1 : 0) != 0, (Object)"Value of child node is inconsistent with its branch");
        Preconditions.checkState((parent._weightedCount >= 1.0E-5 || child._weightedCount >= 1.0E-5 || otherChild != null ? 1 : 0) != 0, (Object)"Found a linear chain of zero-weight nodes");
    }

    public String toGraphviz() {
        StringBuilder builder = new StringBuilder();
        builder.append("digraph QuantileDigest {\n").append("\tgraph [ordering=\"out\"];");
        final ArrayList nodes = new ArrayList();
        this.postOrderTraversal(this._root, new Callback(){

            @Override
            public boolean process(Node node) {
                nodes.add(node);
                return true;
            }
        });
        ImmutableListMultimap nodesByLevel = Multimaps.index(nodes, (Function)new Function<Node, Integer>(){

            public Integer apply(Node input) {
                return input._level;
            }
        });
        for (Map.Entry entry : nodesByLevel.asMap().entrySet()) {
            builder.append("\tsubgraph level_" + entry.getKey() + " {\n").append("\t\trank = same;\n");
            for (Node node : (Collection)entry.getValue()) {
                builder.append(String.format("\t\t%s [label=\"[%s..%s]@%s\\n%s\", shape=rect, style=filled,color=%s];\n", QuantileDigest.idFor(node), node.getLowerBound(), node.getUpperBound(), node._level, node._weightedCount, node._weightedCount > 0.0 ? "salmon2" : "white"));
            }
            builder.append("\t}\n");
        }
        for (Node node : nodes) {
            if (node._left != null) {
                builder.append(String.format("\t%s -> %s;\n", QuantileDigest.idFor(node), QuantileDigest.idFor(node._left)));
            }
            if (node._right == null) continue;
            builder.append(String.format("\t%s -> %s;\n", QuantileDigest.idFor(node), QuantileDigest.idFor(node._right)));
        }
        builder.append("}\n");
        return builder.toString();
    }

    private static String idFor(Node node) {
        return String.format("node_%x_%x", node._bits, node._level);
    }

    private static long longToBits(long value) {
        return value ^ Long.MIN_VALUE;
    }

    private static long bitsToLong(long bits) {
        return bits ^ Long.MIN_VALUE;
    }

    public static <T> T firstNonNull(T first, T second) {
        return (T)(first != null ? first : Preconditions.checkNotNull(second));
    }

    public void offer(long value) {
        this.add(value);
    }

    public static QuantileDigest merge(List<QuantileDigest> digests) {
        if (digests.isEmpty()) {
            throw new RuntimeException("Digests to be unioned should not be empty!");
        }
        QuantileDigest ret = digests.get(0);
        for (int i = 1; i < digests.size(); ++i) {
            ret.merge(digests.get(i));
        }
        return ret;
    }

    private static class Flags {
        public static final int HAS_LEFT = 1;
        public static final int HAS_RIGHT = 2;

        private Flags() {
        }
    }

    private static interface Callback {
        public boolean process(Node var1);
    }

    private static class Node {
        private double _weightedCount;
        private int _level;
        private long _bits;
        private Node _left;
        private Node _right;

        private Node(long bits, int level, double weightedCount) {
            this._bits = bits;
            this._level = level;
            this._weightedCount = weightedCount;
        }

        public boolean isLeaf() {
            return this._left == null && this._right == null;
        }

        public boolean hasSingleChild() {
            return this._left == null && this._right != null || this._left != null && this._right == null;
        }

        public Node getSingleChild() {
            Preconditions.checkState((boolean)this.hasSingleChild(), (Object)"Node does not have a single child");
            return QuantileDigest.firstNonNull(this._left, this._right);
        }

        public long getUpperBound() {
            long mask = 0L;
            if (this._level > 0) {
                mask = -1L >>> 64 - this._level;
            }
            return QuantileDigest.bitsToLong(this._bits | mask);
        }

        public long getBranchMask() {
            return 1L << this._level - 1;
        }

        public long getLowerBound() {
            long mask = 0L;
            if (this._level > 0) {
                mask = -1L >>> 64 - this._level;
            }
            return QuantileDigest.bitsToLong(this._bits & (mask ^ 0xFFFFFFFFFFFFFFFFL));
        }

        public long getMiddle() {
            return this.getLowerBound() + (this.getUpperBound() - this.getLowerBound()) / 2L;
        }

        public String toString() {
            return String.format("%s (level = %d, count = %s, left = %s, right = %s)", this._bits, this._level, this._weightedCount, this._left != null, this._right != null);
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this._weightedCount, this._level, this._bits, this._left, this._right});
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            Node other = (Node)obj;
            return Objects.equal((Object)this._weightedCount, (Object)other._weightedCount) && Objects.equal((Object)this._level, (Object)other._level) && Objects.equal((Object)this._bits, (Object)other._bits) && Objects.equal((Object)this._left, (Object)other._left) && Objects.equal((Object)this._right, (Object)other._right);
        }
    }

    public static class Bucket {
        private double _count;
        private double _mean;

        public Bucket(double count, double mean) {
            this._count = count;
            this._mean = mean;
        }

        public double getCount() {
            return this._count;
        }

        public double getMean() {
            return this._mean;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Bucket bucket = (Bucket)o;
            if (Double.compare(bucket._count, this._count) != 0) {
                return false;
            }
            return Double.compare(bucket._mean, this._mean) == 0;
        }

        public int hashCode() {
            long temp = this._count != 0.0 ? Double.doubleToLongBits(this._count) : 0L;
            int result = (int)(temp ^ temp >>> 32);
            temp = this._mean != 0.0 ? Double.doubleToLongBits(this._mean) : 0L;
            result = 31 * result + (int)(temp ^ temp >>> 32);
            return result;
        }

        public String toString() {
            return String.format("[count: %f, mean: %f]", this._count, this._mean);
        }
    }

    private static enum TraversalOrder {
        FORWARD,
        REVERSE;

    }
}

