/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.segment.spi.memory;

import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.pinot.segment.spi.memory.ByteBufferPinotBufferFactory;
import org.apache.pinot.segment.spi.memory.LArrayPinotBufferFactory;
import org.apache.pinot.segment.spi.memory.PinotBufferFactory;
import org.apache.pinot.segment.spi.memory.SmallWithFallbackPinotBufferFactory;
import org.apache.pinot.segment.spi.memory.unsafe.UnsafePinotBufferFactory;
import org.apache.pinot.segment.spi.utils.JavaVersion;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.apache.pinot.spi.plugin.PluginManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public abstract class PinotDataBuffer
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(PinotDataBuffer.class);
    public static final ByteOrder NATIVE_ORDER = ByteOrder.nativeOrder();
    public static final ByteOrder NON_NATIVE_ORDER = NATIVE_ORDER == ByteOrder.BIG_ENDIAN ? ByteOrder.LITTLE_ENDIAN : ByteOrder.BIG_ENDIAN;
    public static final int BULK_BYTES_PROCESSING_THRESHOLD = 10;
    private static final AtomicLong DIRECT_BUFFER_COUNT = new AtomicLong();
    private static final AtomicLong DIRECT_BUFFER_USAGE = new AtomicLong();
    private static final AtomicLong MMAP_BUFFER_COUNT = new AtomicLong();
    private static final AtomicLong MMAP_BUFFER_USAGE = new AtomicLong();
    private static final AtomicLong ALLOCATION_FAILURE_COUNT = new AtomicLong();
    private static final Map<PinotDataBuffer, BufferContext> BUFFER_CONTEXT_MAP = new WeakHashMap<PinotDataBuffer, BufferContext>();
    private static final String OFFHEAP_BUFFER_FACTORY_CONFIG = "pinot.offheap.buffer.factory";
    private static final String OFFHEAP_BUFFER_PRIORITIZE_BYTE_BUFFER_CONFIG = "pinot.offheap.prioritize.bytebuffer";
    private static PinotBufferFactory _defaultFactory = PinotDataBuffer.createDefaultFactory();
    private static final ThreadLocal<PinotBufferFactory> _FACTORY = new ThreadLocal();
    private volatile boolean _closeable;

    public static void useFactory(PinotBufferFactory factory) {
        _FACTORY.set(factory);
    }

    public static PinotBufferFactory getFactory() {
        PinotBufferFactory pinotBufferFactory = _FACTORY.get();
        if (pinotBufferFactory == null) {
            pinotBufferFactory = _defaultFactory;
        }
        return pinotBufferFactory;
    }

    public static PinotBufferFactory createDefaultFactory() {
        return PinotDataBuffer.createDefaultFactory(true);
    }

    public static PinotBufferFactory createDefaultFactory(boolean prioritizeByteBuffer) {
        String factoryClassName;
        if (JavaVersion.VERSION < 16) {
            LOGGER.info("Using LArray as buffer on JVM version {}", (Object)JavaVersion.VERSION);
            factoryClassName = LArrayPinotBufferFactory.class.getCanonicalName();
        } else {
            LOGGER.info("Using Unsafe as buffer on JVM version {}", (Object)JavaVersion.VERSION);
            factoryClassName = UnsafePinotBufferFactory.class.getCanonicalName();
        }
        return PinotDataBuffer.createFactory(factoryClassName, prioritizeByteBuffer);
    }

    private static PinotBufferFactory createFactory(String factoryClassName, boolean prioritizeByteBuffer) {
        try {
            LOGGER.info("Instantiating Pinot buffer factory class {}", (Object)factoryClassName);
            PinotBufferFactory factory = (PinotBufferFactory)PluginManager.get().createInstance(factoryClassName);
            if (prioritizeByteBuffer) {
                factory = new SmallWithFallbackPinotBufferFactory(new ByteBufferPinotBufferFactory(), factory);
            }
            return factory;
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static void loadDefaultFactory(PinotConfiguration configuration) {
        boolean prioritizeByteBuffer = configuration.getProperty(OFFHEAP_BUFFER_PRIORITIZE_BYTE_BUFFER_CONFIG, true);
        String factoryClassName = configuration.getProperty(OFFHEAP_BUFFER_FACTORY_CONFIG);
        if (factoryClassName != null) {
            _defaultFactory = PinotDataBuffer.createFactory(factoryClassName, prioritizeByteBuffer);
        } else {
            LOGGER.info("No custom Pinot buffer factory class found in configuration. Using default factory with prioritize bytebuffer = {}", (Object)prioritizeByteBuffer);
            _defaultFactory = PinotDataBuffer.createDefaultFactory(prioritizeByteBuffer);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static PinotDataBuffer allocateDirect(long size, ByteOrder byteOrder, @Nullable String description) {
        PinotDataBuffer buffer;
        try {
            buffer = PinotDataBuffer.getFactory().allocateDirect(size, byteOrder);
        }
        catch (Exception e) {
            LOGGER.error("Caught exception while allocating direct buffer of size: {} with description: {}", new Object[]{size, description, e});
            LOGGER.error("Buffer stats: {}", (Object)PinotDataBuffer.getBufferStats());
            ALLOCATION_FAILURE_COUNT.getAndIncrement();
            throw e;
        }
        DIRECT_BUFFER_COUNT.getAndIncrement();
        DIRECT_BUFFER_USAGE.getAndAdd(size);
        Map<PinotDataBuffer, BufferContext> map = BUFFER_CONTEXT_MAP;
        synchronized (map) {
            BUFFER_CONTEXT_MAP.put(buffer, new BufferContext(BufferContext.Type.DIRECT, size, null, description));
        }
        return buffer;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static PinotDataBuffer loadFile(File file, long offset, long size, ByteOrder byteOrder, @Nullable String description) throws IOException {
        PinotDataBuffer buffer;
        try {
            buffer = PinotDataBuffer.getFactory().readFile(file, offset, size, byteOrder);
        }
        catch (Exception e) {
            LOGGER.error("Caught exception while loading file: {} from offset: {} of size: {} with description: {}", new Object[]{file.getAbsolutePath(), offset, size, description, e});
            LOGGER.error("Buffer stats: {}", (Object)PinotDataBuffer.getBufferStats());
            ALLOCATION_FAILURE_COUNT.getAndIncrement();
            throw e;
        }
        DIRECT_BUFFER_COUNT.getAndIncrement();
        DIRECT_BUFFER_USAGE.getAndAdd(size);
        Map<PinotDataBuffer, BufferContext> map = BUFFER_CONTEXT_MAP;
        synchronized (map) {
            BUFFER_CONTEXT_MAP.put(buffer, new BufferContext(BufferContext.Type.DIRECT, size, file.getAbsolutePath().intern(), description));
        }
        return buffer;
    }

    @VisibleForTesting
    public static PinotDataBuffer loadBigEndianFile(File file) throws IOException {
        return PinotDataBuffer.loadFile(file, 0L, file.length(), ByteOrder.BIG_ENDIAN, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static PinotDataBuffer mapFile(File file, boolean readOnly, long offset, long size, ByteOrder byteOrder, @Nullable String description) throws IOException {
        PinotDataBuffer buffer;
        try {
            buffer = PinotDataBuffer.getFactory().mapFile(file, readOnly, offset, size, byteOrder);
        }
        catch (Exception e) {
            LOGGER.error("Caught exception while mapping file: {} from offset: {} of size: {} with description: {}", new Object[]{file.getAbsolutePath(), offset, size, description, e});
            LOGGER.error("Buffer stats: {}", (Object)PinotDataBuffer.getBufferStats());
            ALLOCATION_FAILURE_COUNT.getAndIncrement();
            throw e;
        }
        MMAP_BUFFER_COUNT.getAndIncrement();
        MMAP_BUFFER_USAGE.getAndAdd(size);
        Map<PinotDataBuffer, BufferContext> map = BUFFER_CONTEXT_MAP;
        synchronized (map) {
            BUFFER_CONTEXT_MAP.put(buffer, new BufferContext(BufferContext.Type.MMAP, size, file.getAbsolutePath().intern(), description));
        }
        return buffer;
    }

    @VisibleForTesting
    public static PinotDataBuffer mapReadOnlyBigEndianFile(File file) throws IOException {
        return PinotDataBuffer.mapFile(file, true, 0L, file.length(), ByteOrder.BIG_ENDIAN, null);
    }

    public static long getDirectBufferCount() {
        return DIRECT_BUFFER_COUNT.get();
    }

    public static long getDirectBufferUsage() {
        return DIRECT_BUFFER_USAGE.get();
    }

    public static long getMmapBufferCount() {
        return MMAP_BUFFER_COUNT.get();
    }

    public static long getMmapBufferUsage() {
        return MMAP_BUFFER_USAGE.get();
    }

    public static long getAllocationFailureCount() {
        return ALLOCATION_FAILURE_COUNT.get();
    }

    @VisibleForTesting
    protected static void cleanStats() {
        DIRECT_BUFFER_COUNT.set(0L);
        DIRECT_BUFFER_USAGE.set(0L);
        MMAP_BUFFER_COUNT.set(0L);
        MMAP_BUFFER_USAGE.set(0L);
        ALLOCATION_FAILURE_COUNT.set(0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<String> getBufferInfo() {
        Map<PinotDataBuffer, BufferContext> map = BUFFER_CONTEXT_MAP;
        synchronized (map) {
            ArrayList<String> bufferInfo = new ArrayList<String>(BUFFER_CONTEXT_MAP.size());
            for (BufferContext bufferContext : BUFFER_CONTEXT_MAP.values()) {
                bufferInfo.add(bufferContext.toString());
            }
            return bufferInfo;
        }
    }

    private static String getBufferStats() {
        return String.format("Direct buffer count: %s, size: %s; Mmap buffer count: %s, size: %s", DIRECT_BUFFER_COUNT.get(), DIRECT_BUFFER_USAGE.get(), MMAP_BUFFER_COUNT.get(), MMAP_BUFFER_USAGE.get());
    }

    protected PinotDataBuffer(boolean closeable) {
        this._closeable = closeable;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized void close() throws IOException {
        if (this._closeable) {
            BufferContext bufferContext;
            this.flush();
            this.release();
            Map<PinotDataBuffer, BufferContext> map = BUFFER_CONTEXT_MAP;
            synchronized (map) {
                bufferContext = BUFFER_CONTEXT_MAP.remove(this);
            }
            if (bufferContext != null) {
                if (bufferContext._type == BufferContext.Type.DIRECT) {
                    DIRECT_BUFFER_COUNT.getAndDecrement();
                    DIRECT_BUFFER_USAGE.getAndAdd(-bufferContext._size);
                } else {
                    MMAP_BUFFER_COUNT.getAndDecrement();
                    MMAP_BUFFER_USAGE.getAndAdd(-bufferContext._size);
                }
            }
            this._closeable = false;
        }
    }

    public byte getByte(int offset) {
        return this.getByte((long)offset);
    }

    public abstract byte getByte(long var1);

    public void putByte(int offset, byte value) {
        this.putByte((long)offset, value);
    }

    public abstract void putByte(long var1, byte var3);

    public char getChar(int offset) {
        return this.getChar((long)offset);
    }

    public abstract char getChar(long var1);

    public void putChar(int offset, char value) {
        this.putChar((long)offset, value);
    }

    public abstract void putChar(long var1, char var3);

    public short getShort(int offset) {
        return this.getShort((long)offset);
    }

    public abstract short getShort(long var1);

    public void putShort(int offset, short value) {
        this.putShort((long)offset, value);
    }

    public abstract void putShort(long var1, short var3);

    public int getInt(int offset) {
        return this.getInt((long)offset);
    }

    public abstract int getInt(long var1);

    public void putInt(int offset, int value) {
        this.putInt((long)offset, value);
    }

    public abstract void putInt(long var1, int var3);

    public long getLong(int offset) {
        return this.getLong((long)offset);
    }

    public abstract long getLong(long var1);

    public void putLong(int offset, long value) {
        this.putLong((long)offset, value);
    }

    public abstract void putLong(long var1, long var3);

    public float getFloat(int offset) {
        return this.getFloat((long)offset);
    }

    public abstract float getFloat(long var1);

    public void putFloat(int offset, float value) {
        this.putFloat((long)offset, value);
    }

    public abstract void putFloat(long var1, float var3);

    public double getDouble(int offset) {
        return this.getDouble((long)offset);
    }

    public abstract double getDouble(long var1);

    public void putDouble(int offset, double value) {
        this.putDouble((long)offset, value);
    }

    public abstract void putDouble(long var1, double var3);

    public void copyTo(long offset, byte[] buffer, int destOffset, int size) {
        if (size <= 10) {
            int end = destOffset + size;
            for (int i = destOffset; i < end; ++i) {
                buffer[i] = this.getByte(offset++);
            }
        } else {
            this.toDirectByteBuffer(offset, size).get(buffer, destOffset, size);
        }
    }

    public void copyTo(long offset, byte[] buffer) {
        this.copyTo(offset, buffer, 0, buffer.length);
    }

    public void copyTo(long offset, PinotDataBuffer buffer, long destOffset, long size) {
        int pageSize = Integer.MAX_VALUE;
        long alreadyCopied = 0L;
        while (size - alreadyCopied > 0L) {
            long remaining = size - alreadyCopied;
            int step = remaining > (long)pageSize ? pageSize : (int)remaining;
            ByteBuffer destBb = buffer.toDirectByteBuffer(destOffset + alreadyCopied, step);
            ByteBuffer myView = this.toDirectByteBuffer(offset + alreadyCopied, step);
            destBb.put(myView);
            alreadyCopied += (long)step;
        }
    }

    public void readFrom(long offset, byte[] buffer, int srcOffset, int size) {
        assert (offset <= Integer.MAX_VALUE);
        int intOffset = (int)offset;
        if (size <= 10) {
            int end = srcOffset + size;
            for (int i = srcOffset; i < end; ++i) {
                this.putByte(intOffset++, buffer[i]);
            }
        } else {
            this.toDirectByteBuffer(offset, size).put(buffer, srcOffset, size);
        }
    }

    public void readFrom(long offset, byte[] buffer) {
        this.readFrom(offset, buffer, 0, buffer.length);
    }

    public void readFrom(long offset, ByteBuffer buffer) {
        this.toDirectByteBuffer(offset, buffer.remaining()).put(buffer);
    }

    public void readFrom(long offset, File file, long srcOffset, long size) throws IOException {
        try (RandomAccessFile raf = new RandomAccessFile(file, "r");
             FileChannel fileChannel = raf.getChannel();){
            ByteBuffer bb;
            int step = 0x3FFFFFFF;
            while (size > Integer.MAX_VALUE) {
                bb = this.toDirectByteBuffer(offset, step);
                fileChannel.read(bb, srcOffset);
                offset += (long)step;
                srcOffset += (long)step;
                size -= (long)step;
            }
            bb = this.toDirectByteBuffer(offset, (int)size);
            fileChannel.read(bb, srcOffset);
        }
    }

    public abstract long size();

    public abstract ByteOrder order();

    public abstract PinotDataBuffer view(long var1, long var3, ByteOrder var5);

    public PinotDataBuffer view(long start, long end) {
        return this.view(start, end, this.order());
    }

    public abstract ByteBuffer toDirectByteBuffer(long var1, int var3, ByteOrder var4);

    public ByteBuffer toDirectByteBuffer(long offset, int size) {
        return this.toDirectByteBuffer(offset, size, this.order());
    }

    public abstract void flush();

    public abstract void release() throws IOException;

    public boolean isCloseable() {
        return this._closeable;
    }

    protected static void checkLimits(long capacity, long offset, long size) {
        if (offset < 0L) {
            throw new IllegalArgumentException("Offset " + offset + " cannot be negative");
        }
        if (size < 0L) {
            throw new IllegalArgumentException("Size " + size + " cannot be negative");
        }
        if (offset + size > capacity) {
            throw new IllegalArgumentException("Size (" + size + ") + offset (" + offset + ") exceeds the capacity of " + capacity);
        }
    }

    private static class BufferContext {
        final Type _type;
        final long _size;
        final String _filePath;
        final String _description;

        BufferContext(Type type, long size, @Nullable String filePath, @Nullable String description) {
            this._type = type;
            this._size = size;
            this._filePath = filePath;
            this._description = description;
        }

        public String toString() {
            String context = "Type: " + this._type + ", Size: " + this._size;
            if (this._filePath != null) {
                context = context + ", File Path: " + this._filePath;
            }
            if (this._description != null) {
                context = context + ", Description: " + this._description;
            }
            return context;
        }

        static enum Type {
            DIRECT,
            MMAP;

        }
    }
}

