/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.mledger.util;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import io.netty.util.ReferenceCounted;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.apache.commons.lang3.tuple.Pair;

public class RangeCache<Key extends Comparable<Key>, Value extends ReferenceCounted> {
    private final ConcurrentNavigableMap<Key, Value> entries;
    private AtomicLong size = new AtomicLong(0L);
    private final Weighter<Value> weighter;
    private final TimestampExtractor<Value> timestampExtractor;

    public RangeCache() {
        this(new DefaultWeighter(), x -> System.nanoTime());
    }

    public RangeCache(Weighter<Value> weighter, TimestampExtractor<Value> timestampExtractor) {
        this.entries = new ConcurrentSkipListMap<Key, Value>();
        this.weighter = weighter;
        this.timestampExtractor = timestampExtractor;
    }

    public boolean put(Key key, Value value) {
        MutableBoolean flag = new MutableBoolean();
        this.entries.computeIfAbsent(key, k -> {
            this.size.addAndGet(this.weighter.getSize(value));
            flag.setValue(true);
            return value;
        });
        return flag.booleanValue();
    }

    public Value get(Key key) {
        ReferenceCounted value = (ReferenceCounted)this.entries.get(key);
        if (value == null) {
            return null;
        }
        try {
            value.retain();
            return (Value)value;
        }
        catch (Throwable t) {
            return null;
        }
    }

    public Collection<Value> getRange(Key first, Key last) {
        ArrayList values = Lists.newArrayList();
        for (ReferenceCounted value : this.entries.subMap((Object)first, true, (Object)last, true).values()) {
            try {
                value.retain();
                values.add(value);
            }
            catch (Throwable throwable) {}
        }
        return values;
    }

    public Pair<Integer, Long> removeRange(Key first, Key last, boolean lastInclusive) {
        NavigableMap subMap = this.entries.subMap((Object)first, true, (Object)last, lastInclusive);
        int removedEntries = 0;
        long removedSize = 0L;
        for (Comparable key : subMap.keySet()) {
            ReferenceCounted value = (ReferenceCounted)this.entries.remove(key);
            if (value == null) continue;
            removedSize += this.weighter.getSize(value);
            value.release();
            ++removedEntries;
        }
        this.size.addAndGet(-removedSize);
        return Pair.of((Object)removedEntries, (Object)removedSize);
    }

    public Pair<Integer, Long> evictLeastAccessedEntries(long minSize) {
        Map.Entry entry;
        long removedSize;
        ReferenceCounted value;
        Preconditions.checkArgument((minSize > 0L ? 1 : 0) != 0);
        int removedEntries = 0;
        for (removedSize = 0L; removedSize < minSize && (entry = this.entries.pollFirstEntry()) != null; removedSize += this.weighter.getSize(value)) {
            value = (ReferenceCounted)entry.getValue();
            ++removedEntries;
            value.release();
        }
        this.size.addAndGet(-removedSize);
        return Pair.of((Object)removedEntries, (Object)removedSize);
    }

    public Pair<Integer, Long> evictLEntriesBeforeTimestamp(long maxTimestamp) {
        Map.Entry entry;
        long removedSize = 0L;
        int removedCount = 0;
        while ((entry = this.entries.firstEntry()) != null && this.timestampExtractor.getTimestamp((ReferenceCounted)entry.getValue()) <= maxTimestamp) {
            ReferenceCounted value = (ReferenceCounted)entry.getValue();
            boolean removeHits = this.entries.remove(entry.getKey(), value);
            if (!removeHits) break;
            removedSize += this.weighter.getSize(value);
            ++removedCount;
            value.release();
        }
        this.size.addAndGet(-removedSize);
        return Pair.of((Object)removedCount, (Object)removedSize);
    }

    protected long getNumberOfEntries() {
        return this.entries.size();
    }

    public long getSize() {
        return this.size.get();
    }

    public synchronized Pair<Integer, Long> clear() {
        Map.Entry entry;
        long removedSize = 0L;
        int removedCount = 0;
        while ((entry = this.entries.pollFirstEntry()) != null) {
            ReferenceCounted value = (ReferenceCounted)entry.getValue();
            removedSize += this.weighter.getSize(value);
            ++removedCount;
            value.release();
        }
        this.entries.clear();
        this.size.getAndAdd(-removedSize);
        return Pair.of((Object)removedCount, (Object)removedSize);
    }

    private static class DefaultWeighter<Value>
    implements Weighter<Value> {
        private DefaultWeighter() {
        }

        @Override
        public long getSize(Value value) {
            return 1L;
        }
    }

    public static interface TimestampExtractor<ValueT> {
        public long getTimestamp(ValueT var1);
    }

    public static interface Weighter<ValueT> {
        public long getSize(ValueT var1);
    }
}

