/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.operator.aggregation;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Doubles;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import it.unimi.dsi.fastutil.Arrays;
import it.unimi.dsi.fastutil.Swapper;
import it.unimi.dsi.fastutil.ints.AbstractIntComparator;
import it.unimi.dsi.fastutil.ints.IntComparator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.PriorityQueue;
import org.openjdk.jol.info.ClassLayout;

public class NumericHistogram {
    private static final byte FORMAT_TAG = 0;
    private static final int INSTANCE_SIZE = ClassLayout.parseClass(NumericHistogram.class).instanceSize();
    private final int maxBuckets;
    private final double[] values;
    private final double[] weights;
    private int nextIndex;

    public NumericHistogram(int maxBuckets) {
        this(maxBuckets, Math.max((int)((double)maxBuckets * 0.2), 1));
    }

    public NumericHistogram(int maxBuckets, int buffer) {
        Preconditions.checkArgument((maxBuckets >= 2 ? 1 : 0) != 0, (Object)"maxBuckets must be >= 2");
        Preconditions.checkArgument((buffer >= 1 ? 1 : 0) != 0, (Object)"buffer must be >= 1");
        this.maxBuckets = maxBuckets;
        this.values = new double[maxBuckets + buffer];
        this.weights = new double[maxBuckets + buffer];
    }

    public NumericHistogram(Slice serialized, int buffer) {
        Objects.requireNonNull(serialized, "serialized is null");
        Preconditions.checkArgument((buffer >= 1 ? 1 : 0) != 0, (Object)"buffer must be >= 1");
        BasicSliceInput input = serialized.getInput();
        Preconditions.checkArgument((input.readByte() == 0 ? 1 : 0) != 0, (Object)"Unsupported format tag");
        this.maxBuckets = input.readInt();
        this.nextIndex = input.readInt();
        this.values = new double[this.maxBuckets + buffer];
        this.weights = new double[this.maxBuckets + buffer];
        input.readBytes(Slices.wrappedDoubleArray((double[])this.values), this.nextIndex * 8);
        input.readBytes(Slices.wrappedDoubleArray((double[])this.weights), this.nextIndex * 8);
    }

    public Slice serialize() {
        this.compact();
        int requiredBytes = 9 + 8 * this.nextIndex + 8 * this.nextIndex;
        return Slices.allocate((int)requiredBytes).getOutput().appendByte(0).appendInt(this.maxBuckets).appendInt(this.nextIndex).appendBytes(Slices.wrappedDoubleArray((double[])this.values, (int)0, (int)this.nextIndex)).appendBytes(Slices.wrappedDoubleArray((double[])this.weights, (int)0, (int)this.nextIndex)).getUnderlyingSlice();
    }

    public long estimatedInMemorySize() {
        return (long)INSTANCE_SIZE + SizeOf.sizeOf((double[])this.values) + SizeOf.sizeOf((double[])this.weights);
    }

    public void add(double value) {
        this.add(value, 1.0);
    }

    public void add(double value, double weight) {
        if (this.nextIndex == this.values.length) {
            this.compact();
        }
        this.values[this.nextIndex] = value;
        this.weights[this.nextIndex] = weight;
        ++this.nextIndex;
    }

    public void mergeWith(NumericHistogram other) {
        int count = this.nextIndex + other.nextIndex;
        double[] newValues = new double[count];
        double[] newWeights = new double[count];
        NumericHistogram.concat(newValues, this.values, this.nextIndex, other.values, other.nextIndex);
        NumericHistogram.concat(newWeights, this.weights, this.nextIndex, other.weights, other.nextIndex);
        count = NumericHistogram.mergeSameBuckets(newValues, newWeights, count);
        if (count <= this.maxBuckets) {
            System.arraycopy(newValues, 0, this.values, 0, count);
            System.arraycopy(newWeights, 0, this.weights, 0, count);
            this.nextIndex = count;
            return;
        }
        NumericHistogram.sort(newValues, newWeights, count);
        this.store(NumericHistogram.mergeBuckets(newValues, newWeights, count, this.maxBuckets));
    }

    public Map<Double, Double> getBuckets() {
        this.compact();
        LinkedHashMap<Double, Double> result = new LinkedHashMap<Double, Double>();
        for (int i = 0; i < this.nextIndex; ++i) {
            result.put(this.values[i], this.weights[i]);
        }
        return result;
    }

    @VisibleForTesting
    void compact() {
        this.nextIndex = NumericHistogram.mergeSameBuckets(this.values, this.weights, this.nextIndex);
        if (this.nextIndex <= this.maxBuckets) {
            return;
        }
        this.store(NumericHistogram.mergeBuckets(this.values, this.weights, this.nextIndex, this.maxBuckets));
    }

    private static PriorityQueue<Entry> mergeBuckets(double[] values, double[] weights, int count, int targetCount) {
        Preconditions.checkArgument((targetCount > 0 ? 1 : 0) != 0, (Object)"targetCount must be > 0");
        PriorityQueue<Entry> queue = NumericHistogram.initializeQueue(values, weights, count);
        while (count > targetCount) {
            Entry current = queue.poll();
            if (!current.isValid()) continue;
            --count;
            Entry right = current.getRight();
            Preconditions.checkState((right != null ? 1 : 0) != 0, (Object)"Expected right to be != null");
            Preconditions.checkState((boolean)right.isValid(), (Object)"Expected right to be valid");
            double newWeight = current.getWeight() + right.getWeight();
            double newValue = (current.getValue() * current.getWeight() + right.getValue() * right.getWeight()) / newWeight;
            right.invalidate();
            Entry merged = new Entry(current.getId(), newValue, newWeight, right.getRight());
            queue.add(merged);
            Entry left = current.getLeft();
            if (left == null) continue;
            Preconditions.checkState((boolean)left.isValid(), (Object)"Expected left to be valid");
            left.invalidate();
            queue.add(new Entry(left.getId(), left.getValue(), left.getWeight(), left.getLeft(), merged));
        }
        return queue;
    }

    private void store(PriorityQueue<Entry> queue) {
        this.nextIndex = 0;
        for (Entry entry : queue) {
            if (!entry.isValid()) continue;
            this.values[this.nextIndex] = entry.getValue();
            this.weights[this.nextIndex] = entry.getWeight();
            ++this.nextIndex;
        }
        NumericHistogram.sort(this.values, this.weights, this.nextIndex);
    }

    private static void concat(double[] target, double[] first, int firstLength, double[] second, int secondLength) {
        System.arraycopy(first, 0, target, 0, firstLength);
        System.arraycopy(second, 0, target, firstLength, secondLength);
    }

    private static int mergeSameBuckets(double[] values, double[] weights, int nextIndex) {
        NumericHistogram.sort(values, weights, nextIndex);
        int current = 0;
        for (int i = 1; i < nextIndex; ++i) {
            if (values[current] == values[i]) {
                int n = current;
                weights[n] = weights[n] + weights[i];
                continue;
            }
            values[++current] = values[i];
            weights[current] = weights[i];
        }
        return current + 1;
    }

    private static PriorityQueue<Entry> initializeQueue(double[] values, double[] weights, int nextIndex) {
        Preconditions.checkArgument((nextIndex > 0 ? 1 : 0) != 0, (Object)"nextIndex must be > 0");
        PriorityQueue<Entry> queue = new PriorityQueue<Entry>(nextIndex);
        Entry right = new Entry(nextIndex - 1, values[nextIndex - 1], weights[nextIndex - 1], null);
        queue.add(right);
        for (int i = nextIndex - 2; i >= 0; --i) {
            Entry current = new Entry(i, values[i], weights[i], right);
            queue.add(current);
            right = current;
        }
        return queue;
    }

    private static void sort(final double[] values, final double[] weights, int nextIndex) {
        Arrays.quickSort((int)0, (int)nextIndex, (IntComparator)new AbstractIntComparator(){

            public int compare(int a, int b) {
                return Doubles.compare((double)values[a], (double)values[b]);
            }
        }, (Swapper)new Swapper(){

            public void swap(int a, int b) {
                double temp = values[a];
                values[a] = values[b];
                values[b] = temp;
                temp = weights[a];
                weights[a] = weights[b];
                weights[b] = temp;
            }
        });
    }

    private static double computePenalty(double value1, double value2, double weight1, double weight2) {
        double weight = value2 + weight2;
        double squaredDifference = (value1 - weight1) * (value1 - weight1);
        double proportionsProduct = value2 * weight2 / ((value2 + weight2) * (value2 + weight2));
        return weight * squaredDifference * proportionsProduct;
    }

    private static class Entry
    implements Comparable<Entry> {
        private final double penalty;
        private final int id;
        private final double value;
        private final double weight;
        private boolean valid = true;
        private Entry left;
        private Entry right;

        private Entry(int id, double value, double weight, Entry right) {
            this(id, value, weight, null, right);
        }

        private Entry(int id, double value, double weight, Entry left, Entry right) {
            this.id = id;
            this.value = value;
            this.weight = weight;
            this.right = right;
            this.left = left;
            if (right != null) {
                right.left = this;
                this.penalty = NumericHistogram.computePenalty(value, weight, right.value, right.weight);
            } else {
                this.penalty = Double.POSITIVE_INFINITY;
            }
            if (left != null) {
                left.right = this;
            }
        }

        public int getId() {
            return this.id;
        }

        public Entry getLeft() {
            return this.left;
        }

        public Entry getRight() {
            return this.right;
        }

        public double getValue() {
            return this.value;
        }

        public double getWeight() {
            return this.weight;
        }

        public boolean isValid() {
            return this.valid;
        }

        public void invalidate() {
            this.valid = false;
        }

        @Override
        public int compareTo(Entry other) {
            int result = Double.compare(this.penalty, other.penalty);
            if (result == 0) {
                result = Integer.compare(this.id, other.id);
            }
            return result;
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("id", this.id).add("value", this.value).add("weight", this.weight).add("penalty", this.penalty).add("valid", this.valid).toString();
        }
    }
}

