/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.examples.filestore;

import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Objects;
import java.util.Random;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.ratis.client.api.DataStreamOutput;
import org.apache.ratis.datastream.DataStreamTestUtils;
import org.apache.ratis.examples.filestore.FileStoreClient;
import org.apache.ratis.io.StandardWriteOption;
import org.apache.ratis.io.WriteOption;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.DataStreamReply;
import org.apache.ratis.protocol.RoutingTable;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.thirdparty.io.netty.util.internal.ThreadLocalRandom;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.SizeInBytes;
import org.apache.ratis.util.StringUtils;
import org.apache.ratis.util.function.CheckedSupplier;
import org.junit.Assert;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class FileStoreWriter
implements Closeable {
    public static final Logger LOG = LoggerFactory.getLogger(FileStoreWriter.class);
    final long seed = ThreadLocalRandom.current().nextLong();
    final byte[] buffer = new byte[4096];
    final String fileName;
    final SizeInBytes fileSize;
    final FileStoreClient client;
    final Executor asyncExecutor;
    final int bufferSize;

    static Builder newBuilder() {
        return new Builder();
    }

    private FileStoreWriter(String fileName, SizeInBytes fileSize, Executor asyncExecutor, CheckedSupplier<FileStoreClient, IOException> clientSupplier, int bufferSize) throws IOException {
        this.fileName = fileName;
        this.fileSize = fileSize;
        this.client = (FileStoreClient)clientSupplier.get();
        this.asyncExecutor = asyncExecutor;
        this.bufferSize = bufferSize;
    }

    ByteBuffer randomBytes(int length, Random random) {
        Preconditions.assertTrue((length <= this.buffer.length ? 1 : 0) != 0);
        random.nextBytes(this.buffer);
        ByteBuffer b = ByteBuffer.wrap(this.buffer);
        b.limit(length);
        return b;
    }

    FileStoreWriter write(boolean sync) throws IOException {
        int length;
        Random r = new Random(this.seed);
        int size = this.fileSize.getSizeInt();
        for (int offset = 0; offset < size; offset += length) {
            int remaining = size - offset;
            length = Math.min(remaining, this.buffer.length);
            boolean close = length == remaining;
            ByteBuffer b = this.randomBytes(length, r);
            LOG.trace("write {}, offset={}, length={}, close? {}", new Object[]{this.fileName, offset, length, close});
            long written = this.client.write(this.fileName, (long)offset, close, b, sync);
            Assert.assertEquals((long)length, (long)written);
        }
        return this;
    }

    public FileStoreWriter streamWriteAndVerify(RoutingTable routingTable) {
        int length;
        int size = this.fileSize.getSizeInt();
        DataStreamOutput dataStreamOutput = this.client.getStreamOutput(this.fileName, (long)size, routingTable);
        ArrayList<CompletableFuture> futures = new ArrayList<CompletableFuture>();
        ArrayList<Integer> sizes = new ArrayList<Integer>();
        for (int offset = 0; offset < size; offset += length) {
            int remaining = size - offset;
            length = Math.min(remaining, this.bufferSize);
            boolean close = length == remaining;
            LOG.trace("write {}, offset={}, length={}, close? {}", new Object[]{this.fileName, offset, length, close});
            ByteBuffer bf = DataStreamTestUtils.initBuffer((int)0, (int)length);
            futures.add(close ? dataStreamOutput.writeAsync(bf, new WriteOption[]{StandardWriteOption.CLOSE}) : dataStreamOutput.writeAsync(bf, new WriteOption[0]));
            sizes.add(length);
        }
        DataStreamReply reply = (DataStreamReply)dataStreamOutput.closeAsync().join();
        Assert.assertTrue((boolean)reply.isSuccess());
        for (int i = 0; i < futures.size(); ++i) {
            reply = (DataStreamReply)((CompletableFuture)futures.get(i)).join();
            Assert.assertTrue((boolean)reply.isSuccess());
            Assert.assertEquals((long)((Integer)sizes.get(i)).longValue(), (long)reply.getBytesWritten());
            Assert.assertEquals((Object)reply.getType(), (Object)RaftProtos.DataStreamPacketHeaderProto.Type.STREAM_DATA);
        }
        return this;
    }

    CompletableFuture<FileStoreWriter> writeAsync(boolean sync) {
        Objects.requireNonNull(this.asyncExecutor, "asyncExecutor == null");
        Random r = new Random(this.seed);
        int size = this.fileSize.getSizeInt();
        CompletableFuture<FileStoreWriter> returnFuture = new CompletableFuture<FileStoreWriter>();
        AtomicInteger callCount = new AtomicInteger();
        AtomicInteger n = new AtomicInteger();
        while (n.get() < size) {
            int offset = n.get();
            int remaining = size - offset;
            int length = Math.min(remaining, this.buffer.length);
            boolean close = length == remaining;
            ByteBuffer b = this.randomBytes(length, r);
            callCount.incrementAndGet();
            n.addAndGet(length);
            LOG.trace("writeAsync {}, offset={}, length={}, close? {}", new Object[]{this.fileName, offset, length, close});
            ((CompletableFuture)((CompletableFuture)this.client.writeAsync(this.fileName, (long)offset, close, b, sync).thenAcceptAsync(written -> Assert.assertEquals((long)length, (long)written), this.asyncExecutor)).thenRun(() -> {
                int count = callCount.decrementAndGet();
                LOG.trace("writeAsync {}, offset={}, length={}, close? {}: n={}, callCount={}", new Object[]{this.fileName, offset, length, close, n.get(), count});
                if (n.get() == size && count == 0) {
                    returnFuture.complete(this);
                }
            })).exceptionally(e -> {
                returnFuture.completeExceptionally((Throwable)e);
                return null;
            });
        }
        return returnFuture;
    }

    FileStoreWriter verify() throws IOException {
        int n;
        Random r = new Random(this.seed);
        int size = this.fileSize.getSizeInt();
        for (int offset = 0; offset < size; offset += n) {
            int remaining = size - offset;
            n = Math.min(remaining, this.buffer.length);
            ByteString read = this.client.read(this.fileName, (long)offset, (long)n);
            ByteBuffer expected = this.randomBytes(n, r);
            this.verify(read, offset, n, expected);
        }
        return this;
    }

    CompletableFuture<FileStoreWriter> verifyAsync() {
        Objects.requireNonNull(this.asyncExecutor, "asyncExecutor == null");
        Random r = new Random(this.seed);
        int size = this.fileSize.getSizeInt();
        CompletableFuture<FileStoreWriter> returnFuture = new CompletableFuture<FileStoreWriter>();
        AtomicInteger callCount = new AtomicInteger();
        AtomicInteger n = new AtomicInteger();
        while (n.get() < size) {
            int offset = n.get();
            int remaining = size - offset;
            int length = Math.min(remaining, this.buffer.length);
            callCount.incrementAndGet();
            n.addAndGet(length);
            ByteBuffer expected = ByteString.copyFrom((ByteBuffer)this.randomBytes(length, r)).asReadOnlyByteBuffer();
            ((CompletableFuture)((CompletableFuture)this.client.readAsync(this.fileName, (long)offset, (long)length).thenAcceptAsync(read -> this.verify((ByteString)read, offset, length, expected), this.asyncExecutor)).thenRun(() -> {
                int count = callCount.decrementAndGet();
                LOG.trace("verifyAsync {}, offset={}, length={}: n={}, callCount={}", new Object[]{this.fileName, offset, length, n.get(), count});
                if (n.get() == size && count == 0) {
                    returnFuture.complete(this);
                }
            })).exceptionally(e -> {
                returnFuture.completeExceptionally((Throwable)e);
                return null;
            });
        }
        Assert.assertEquals((long)size, (long)n.get());
        return returnFuture;
    }

    void verify(ByteString read, int offset, int length, ByteBuffer expected) {
        Assert.assertEquals((long)length, (long)read.size());
        FileStoreWriter.assertBuffers(offset, length, expected, read.asReadOnlyByteBuffer());
    }

    CompletableFuture<FileStoreWriter> deleteAsync() {
        Objects.requireNonNull(this.asyncExecutor, "asyncExecutor == null");
        return this.client.deleteAsync(this.fileName).thenApplyAsync(reply -> this, this.asyncExecutor);
    }

    FileStoreWriter delete() throws IOException {
        this.client.delete(this.fileName);
        return this;
    }

    @Override
    public void close() throws IOException {
        this.client.close();
    }

    static void assertBuffers(int offset, int length, ByteBuffer expected, ByteBuffer computed) {
        try {
            Assert.assertEquals((Object)expected, (Object)computed);
        }
        catch (AssertionError e) {
            LOG.error("Buffer mismatched at offset=" + offset + ", length=" + length + "\n  expected = " + StringUtils.bytes2HexString((ByteBuffer)expected) + "\n  computed = " + StringUtils.bytes2HexString((ByteBuffer)computed), (Throwable)((Object)e));
            throw e;
        }
    }

    static class Builder {
        private String fileName;
        private SizeInBytes fileSize;
        private CheckedSupplier<FileStoreClient, IOException> clientSupplier;
        private Executor asyncExecutor;
        private int bufferSize;

        Builder() {
        }

        public Builder setFileName(String fileName) {
            this.fileName = fileName;
            return this;
        }

        public Builder setFileSize(SizeInBytes size) {
            this.fileSize = size;
            return this;
        }

        public Builder setFileStoreClientSupplier(CheckedSupplier<FileStoreClient, IOException> supplier) {
            this.clientSupplier = supplier;
            return this;
        }

        public Builder setAsyncExecutor(Executor asyncExecutor) {
            this.asyncExecutor = asyncExecutor;
            return this;
        }

        public Builder setBufferSize(int bufferSize) {
            this.bufferSize = bufferSize;
            return this;
        }

        public FileStoreWriter build() throws IOException {
            return new FileStoreWriter(this.fileName, this.fileSize, this.asyncExecutor, this.clientSupplier, this.bufferSize);
        }
    }
}

