/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.impl;

import java.io.IOException;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.ClientId;
import org.apache.ratis.protocol.RaftClientReply;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.protocol.TransferLeadershipRequest;
import org.apache.ratis.protocol.exceptions.TransferLeadershipException;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.LeaderStateImpl;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.leader.FollowerInfo;
import org.apache.ratis.server.leader.LogAppender;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.MemoizedSupplier;
import org.apache.ratis.util.StringUtils;
import org.apache.ratis.util.TimeDuration;
import org.apache.ratis.util.TimeoutExecutor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransferLeadership {
    public static final Logger LOG = LoggerFactory.getLogger(TransferLeadership.class);
    private final RaftServerImpl server;
    private final TimeDuration requestTimeout;
    private final TimeoutExecutor scheduler = TimeoutExecutor.getInstance();
    private final AtomicReference<PendingRequest> pending = new AtomicReference();

    TransferLeadership(RaftServerImpl server, RaftProperties properties) {
        this.server = server;
        this.requestTimeout = RaftServerConfigKeys.Rpc.requestTimeout(properties);
    }

    private Optional<RaftPeerId> getTransferee() {
        return Optional.ofNullable(this.pending.get()).map(r -> r.getRequest().getNewLeader());
    }

    boolean isSteppingDown() {
        return this.pending.get() != null;
    }

    static Result isFollowerUpToDate(FollowerInfo follower, TermIndex leaderLastEntry) {
        if (follower == null) {
            return Result.NULL_FOLLOWER;
        }
        if (leaderLastEntry == null) {
            return new Result(Result.Type.NOT_UP_TO_DATE, "leaderLastEntry is null");
        }
        long followerMatchIndex = follower.getMatchIndex();
        if (followerMatchIndex < leaderLastEntry.getIndex()) {
            return new Result(Result.Type.NOT_UP_TO_DATE, "followerMatchIndex = " + followerMatchIndex + " < leaderLastEntry.getIndex() = " + leaderLastEntry.getIndex());
        }
        return Result.SUCCESS;
    }

    private Result sendStartLeaderElection(FollowerInfo follower) {
        TermIndex lastEntry = this.server.getState().getLastEntry();
        Result result = TransferLeadership.isFollowerUpToDate(follower, lastEntry);
        if (result != Result.SUCCESS) {
            return result;
        }
        RaftPeerId transferee = follower.getId();
        LOG.info("{}: sendStartLeaderElection to follower {}, lastEntry={}", this.server.getMemberId(), transferee, lastEntry);
        RaftProtos.StartLeaderElectionRequestProto r = ServerProtoUtils.toStartLeaderElectionRequestProto(this.server.getMemberId(), transferee, lastEntry);
        CompletionStage f = CompletableFuture.supplyAsync(() -> {
            this.server.getLeaderElectionMetrics().onTransferLeadership();
            try {
                return this.server.getServerRpc().startLeaderElection(r);
            }
            catch (IOException e) {
                throw new CompletionException("Failed to sendStartLeaderElection to follower " + transferee, e);
            }
        }, this.server.getServerExecutor()).whenComplete((reply, exception) -> {
            if (reply != null) {
                LOG.info("{}: Received startLeaderElection reply from {}: success? {}", this.server.getMemberId(), transferee, reply.getServerReply().getSuccess());
            } else if (exception != null) {
                LOG.warn(this.server.getMemberId() + ": Failed to startLeaderElection for " + transferee, (Throwable)exception);
            }
        });
        if (((CompletableFuture)f).isCompletedExceptionally()) {
            try {
                ((CompletableFuture)f).join();
            }
            catch (Throwable t2) {
                return new Result(t2);
            }
        }
        return Result.SUCCESS;
    }

    void onFollowerAppendEntriesReply(FollowerInfo follower) {
        if (!this.getTransferee().filter(t2 -> t2.equals(follower.getId())).isPresent()) {
            return;
        }
        Result result = this.sendStartLeaderElection(follower);
        if (result == Result.SUCCESS) {
            LOG.info("{}: sent StartLeaderElection to transferee {} after received AppendEntriesResponse", (Object)this.server.getMemberId(), (Object)follower.getId());
        }
    }

    private Result tryTransferLeadership(Context context) {
        RaftPeerId transferee = context.getTransfereeId();
        LOG.info("{}: start transferring leadership to {}", (Object)this.server.getMemberId(), (Object)transferee);
        LogAppender appender = context.getTransfereeLogAppender();
        if (appender == null) {
            return Result.NULL_LOG_APPENDER;
        }
        FollowerInfo follower = appender.getFollower();
        Result result = this.sendStartLeaderElection(follower);
        if (result.getType() == Result.Type.SUCCESS) {
            LOG.info("{}: {} sent StartLeaderElection to transferee {} immediately as it already has up-to-date log", this.server.getMemberId(), result, transferee);
        } else if (result.getType() == Result.Type.NOT_UP_TO_DATE) {
            LOG.info("{}: {} notifying LogAppender to send AppendEntries to transferee {}", this.server.getMemberId(), result, transferee);
            appender.notifyLogAppender();
        }
        return result;
    }

    void start(LogAppender transferee) {
        long timeout = this.server.properties().minRpcTimeoutMs();
        TransferLeadershipRequest request = new TransferLeadershipRequest(ClientId.emptyClientId(), this.server.getId(), this.server.getMemberId().getGroupId(), 0L, transferee.getFollowerId(), timeout);
        this.start(new Context(request, () -> transferee));
    }

    CompletableFuture<RaftClientReply> start(LeaderStateImpl leaderState, TransferLeadershipRequest request) {
        Context context = new Context(request, JavaUtils.memoize(() -> leaderState.getLogAppender(request.getNewLeader()).orElse(null)));
        return this.start(context);
    }

    private CompletableFuture<RaftClientReply> start(Context context) {
        TransferLeadershipRequest request = context.getRequest();
        MemoizedSupplier<PendingRequest> supplier = JavaUtils.memoize(() -> new PendingRequest(request));
        PendingRequest previous = this.pending.getAndUpdate(f -> f != null ? f : (PendingRequest)supplier.get());
        if (previous != null) {
            return this.createReplyFutureFromPreviousRequest(request, previous);
        }
        boolean previousLeaseEnabled = this.server.getRole().getLeaderState().map(l -> l.getAndSetLeaseEnabled(false)).orElse(false);
        PendingRequest pendingRequest = supplier.get();
        Result result = this.tryTransferLeadership(context);
        Result.Type type = result.getType();
        if (type != Result.Type.SUCCESS && type != Result.Type.NOT_UP_TO_DATE) {
            pendingRequest.complete(result);
        } else {
            TimeDuration timeout = request.getTimeoutMs() == 0L ? this.requestTimeout : TimeDuration.valueOf(request.getTimeoutMs(), TimeUnit.MILLISECONDS);
            this.scheduler.onTimeout(timeout, () -> this.complete(new Result(Result.Type.TIMED_OUT, timeout.toString(TimeUnit.SECONDS, 3))), LOG, () -> "Failed to handle timeout");
        }
        pendingRequest.getReplyFuture().whenCompleteAsync((reply, ex) -> {
            if (ex != null || !reply.isSuccess()) {
                this.server.getRole().getLeaderState().ifPresent(l -> l.getAndSetLeaseEnabled(previousLeaseEnabled));
            }
        });
        return pendingRequest.getReplyFuture();
    }

    private CompletableFuture<RaftClientReply> createReplyFutureFromPreviousRequest(TransferLeadershipRequest request, PendingRequest previous) {
        if (request.getNewLeader().equals(previous.getRequest().getNewLeader())) {
            CompletableFuture<RaftClientReply> replyFuture = new CompletableFuture<RaftClientReply>();
            previous.getReplyFuture().whenComplete((r, e) -> {
                if (e != null) {
                    replyFuture.completeExceptionally((Throwable)e);
                } else {
                    replyFuture.complete(r.isSuccess() ? this.server.newSuccessReply(request) : this.server.newExceptionReply(request, r.getException()));
                }
            });
            return replyFuture;
        }
        TransferLeadershipException tle = new TransferLeadershipException(this.server.getMemberId() + "Failed to transfer leadership to " + request.getNewLeader() + ": a previous " + previous + " exists");
        return CompletableFuture.completedFuture(this.server.newExceptionReply(request, tle));
    }

    void complete(Result result) {
        Optional.ofNullable(this.pending.getAndSet(null)).ifPresent(r -> r.complete(result));
    }

    class PendingRequest {
        private final TransferLeadershipRequest request;
        private final CompletableFuture<RaftClientReply> replyFuture = new CompletableFuture();

        PendingRequest(TransferLeadershipRequest request) {
            this.request = request;
        }

        TransferLeadershipRequest getRequest() {
            return this.request;
        }

        CompletableFuture<RaftClientReply> getReplyFuture() {
            return this.replyFuture;
        }

        void complete(Result result) {
            if (this.replyFuture.isDone()) {
                return;
            }
            RaftPeerId currentLeader = TransferLeadership.this.server.getState().getLeaderId();
            if (currentLeader != null && currentLeader.equals(this.request.getNewLeader())) {
                this.replyFuture.complete(TransferLeadership.this.server.newSuccessReply(this.request));
            } else {
                if (result.getType() == Result.Type.SUCCESS) {
                    result = Result.DIFFERENT_LEADER;
                }
                TransferLeadershipException tle = new TransferLeadershipException(TransferLeadership.this.server.getMemberId() + ": Failed to transfer leadership to " + this.request.getNewLeader() + " (the current leader is " + currentLeader + "): " + result);
                this.replyFuture.complete(TransferLeadership.this.server.newExceptionReply(this.request, tle));
            }
        }

        public String toString() {
            return this.request.toString();
        }
    }

    static class Result {
        static final Result SUCCESS = new Result(Type.SUCCESS);
        static final Result DIFFERENT_LEADER = new Result(Type.DIFFERENT_LEADER);
        static final Result NULL_FOLLOWER = new Result(Type.NULL_FOLLOWER);
        static final Result NULL_LOG_APPENDER = new Result(Type.NULL_LOG_APPENDER);
        private final Type type;
        private final String errorMessage;
        private final Throwable exception;

        private Result(Type type) {
            this(type, null);
        }

        private Result(Type type, String errorMessage, Throwable exception) {
            this.type = type;
            this.errorMessage = errorMessage;
            this.exception = exception;
        }

        Result(Type type, String errorMessage) {
            this(type, errorMessage, null);
        }

        Result(Throwable t2) {
            this(Type.COMPLETED_EXCEPTIONALLY, null, t2);
        }

        Type getType() {
            return this.type;
        }

        public String toString() {
            if (this.exception == null) {
                return (Object)((Object)this.type) + (this.errorMessage == null ? "" : "(" + this.errorMessage + ")");
            }
            return (Object)((Object)this.type) + ": " + StringUtils.stringifyException(this.exception);
        }

        static enum Type {
            SUCCESS,
            DIFFERENT_LEADER,
            NULL_FOLLOWER,
            NULL_LOG_APPENDER,
            NOT_UP_TO_DATE,
            TIMED_OUT,
            FAILED_TO_START,
            COMPLETED_EXCEPTIONALLY;

        }
    }

    private static class Context {
        private final TransferLeadershipRequest request;
        private final Supplier<LogAppender> transferee;

        Context(TransferLeadershipRequest request, Supplier<LogAppender> transferee) {
            this.request = request;
            this.transferee = transferee;
        }

        TransferLeadershipRequest getRequest() {
            return this.request;
        }

        RaftPeerId getTransfereeId() {
            return this.request.getNewLeader();
        }

        LogAppender getTransfereeLogAppender() {
            return this.transferee.get();
        }
    }
}

