/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.storage;

import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Matcher;
import java.util.stream.Collectors;
import org.apache.ratis.BaseTest;
import org.apache.ratis.RaftTestUtil;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.server.storage.RaftStorageDirectoryImpl;
import org.apache.ratis.server.storage.RaftStorageImpl;
import org.apache.ratis.server.storage.RaftStorageMetadata;
import org.apache.ratis.server.storage.RaftStorageMetadataFileImpl;
import org.apache.ratis.statemachine.SnapshotRetentionPolicy;
import org.apache.ratis.statemachine.impl.SimpleStateMachineStorage;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.SizeInBytes;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

public class TestRaftStorage
extends BaseTest {
    private File storageDir;

    static RaftStorageImpl newRaftStorage(File dir) throws IOException {
        RaftStorage impl = RaftStorage.newBuilder().setDirectory(dir).setOption(RaftStorage.StartupOption.RECOVER).setStorageFreeSpaceMin(RaftServerConfigKeys.STORAGE_FREE_SPACE_MIN_DEFAULT).build();
        impl.initialize();
        return (RaftStorageImpl)Preconditions.assertInstanceOf((Object)impl, RaftStorageImpl.class);
    }

    @BeforeEach
    public void setup() {
        this.storageDir = this.getTestDir();
    }

    @AfterEach
    public void tearDown() throws Exception {
        if (this.storageDir != null) {
            FileUtils.deleteFully((File)this.storageDir.getParentFile());
        }
    }

    static RaftStorageImpl formatRaftStorage(File dir) throws IOException {
        RaftStorageImpl impl = (RaftStorageImpl)RaftStorage.newBuilder().setDirectory(dir).setOption(RaftStorage.StartupOption.FORMAT).setStorageFreeSpaceMin(SizeInBytes.valueOf((long)0L)).build();
        impl.initialize();
        return impl;
    }

    @Test
    public void testNotExistent() throws IOException {
        FileUtils.deleteFully((File)this.storageDir);
        RaftStorageImpl storage = TestRaftStorage.newRaftStorage(this.storageDir);
        Assertions.assertEquals((Object)RaftStorageDirectoryImpl.StorageState.NORMAL, (Object)storage.getState());
        try {
            TestRaftStorage.formatRaftStorage(this.storageDir).close();
            Assertions.fail((String)"the format should fail since the storage is still locked");
        }
        catch (IOException e) {
            Assertions.assertTrue((boolean)e.getMessage().contains("directory is already locked"));
        }
        storage.close();
        FileUtils.deleteFully((File)this.storageDir);
        Assertions.assertTrue((boolean)this.storageDir.createNewFile());
        try (RaftStorageImpl ignored = TestRaftStorage.newRaftStorage(this.storageDir);){
            Assertions.fail();
        }
        catch (IOException e) {
            Assertions.assertTrue((boolean)e.getMessage().contains(RaftStorageDirectoryImpl.StorageState.NON_EXISTENT.name()));
        }
    }

    @Test
    public void testStorage() throws Exception {
        RaftStorageDirectoryImpl sd = new RaftStorageDirectoryImpl(this.storageDir, SizeInBytes.ZERO);
        try {
            RaftStorageDirectoryImpl.StorageState state = sd.analyzeStorage(true);
            Assertions.assertEquals((Object)RaftStorageDirectoryImpl.StorageState.NOT_FORMATTED, (Object)state);
            Assertions.assertTrue((boolean)sd.isCurrentEmpty());
        }
        finally {
            sd.unlock();
        }
        RaftStorageImpl storage = TestRaftStorage.newRaftStorage(this.storageDir);
        Assertions.assertEquals((Object)RaftStorageDirectoryImpl.StorageState.NORMAL, (Object)storage.getState());
        storage.close();
        Assertions.assertEquals((Object)RaftStorageDirectoryImpl.StorageState.NORMAL, (Object)sd.analyzeStorage(false));
        TestRaftStorage.assertMetadataFile(sd.getMetaFile());
        storage = TestRaftStorage.formatRaftStorage(this.storageDir);
        Assertions.assertEquals((Object)RaftStorageDirectoryImpl.StorageState.NORMAL, (Object)storage.getState());
        RaftStorageMetadataFileImpl metaFile = new RaftStorageMetadataFileImpl(sd.getMetaFile());
        Assertions.assertEquals((Object)RaftStorageMetadata.getDefault(), (Object)metaFile.getMetadata());
        storage.close();
    }

    static void assertMetadataFile(File m) throws Exception {
        Assertions.assertTrue((boolean)m.exists());
        RaftStorageMetadataFileImpl metaFile = new RaftStorageMetadataFileImpl(m);
        Assertions.assertEquals((Object)RaftStorageMetadata.getDefault(), (Object)metaFile.getMetadata());
        RaftPeerId peer1 = RaftPeerId.valueOf((String)"peer1");
        RaftStorageMetadata metadata = RaftStorageMetadata.valueOf((long)123L, (RaftPeerId)peer1);
        metaFile.persist(metadata);
        Assertions.assertEquals((long)metadata.getTerm(), (long)123L);
        Assertions.assertEquals((Object)metadata.getVotedFor(), (Object)peer1);
        Assertions.assertEquals((Object)metadata, (Object)metaFile.getMetadata());
        RaftStorageMetadataFileImpl metaFile2 = new RaftStorageMetadataFileImpl(m);
        Assertions.assertNull(((AtomicReference)RaftTestUtil.getDeclaredField((Object)metaFile2, (String)"metadata")).get());
        Assertions.assertEquals((Object)metadata, (Object)metaFile2.getMetadata());
    }

    @Test
    public void testMetaFile() throws Exception {
        RaftStorageImpl storage = TestRaftStorage.formatRaftStorage(this.storageDir);
        TestRaftStorage.assertMetadataFile(storage.getStorageDir().getMetaFile());
        storage.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testCleanMetaTmpFile() throws Exception {
        RaftStorageImpl storage = TestRaftStorage.newRaftStorage(this.storageDir);
        Assertions.assertEquals((Object)RaftStorageDirectoryImpl.StorageState.NORMAL, (Object)storage.getState());
        storage.close();
        RaftStorageDirectoryImpl sd = new RaftStorageDirectoryImpl(this.storageDir, SizeInBytes.ZERO);
        File metaFile = sd.getMetaFile();
        FileUtils.move((File)metaFile, (File)sd.getMetaTmpFile(), (CopyOption[])new CopyOption[0]);
        Assertions.assertEquals((Object)RaftStorageDirectoryImpl.StorageState.NOT_FORMATTED, (Object)sd.analyzeStorage(false));
        TestRaftStorage.newRaftStorage(this.storageDir).close();
        Assertions.assertTrue((boolean)sd.getMetaFile().exists());
        Assertions.assertTrue((boolean)sd.getMetaTmpFile().createNewFile());
        Assertions.assertTrue((boolean)sd.getMetaTmpFile().exists());
        try {
            storage = TestRaftStorage.newRaftStorage(this.storageDir);
            Assertions.assertEquals((Object)RaftStorageDirectoryImpl.StorageState.NORMAL, (Object)storage.getState());
            Assertions.assertFalse((boolean)sd.getMetaTmpFile().exists());
            Assertions.assertTrue((boolean)sd.getMetaFile().exists());
        }
        finally {
            storage.close();
        }
    }

    @Test
    public void testSnapshotFileName() {
        long term = ThreadLocalRandom.current().nextLong(Long.MAX_VALUE);
        long index = ThreadLocalRandom.current().nextLong(Long.MAX_VALUE);
        String name = SimpleStateMachineStorage.getSnapshotFileName((long)term, (long)index);
        System.out.println("name = " + name);
        File file = new File(this.storageDir, name);
        TermIndex ti = SimpleStateMachineStorage.getTermIndexFromSnapshotFile((File)file);
        System.out.println("file = " + file);
        Assertions.assertEquals((long)term, (long)ti.getTerm());
        Assertions.assertEquals((long)index, (long)ti.getIndex());
        System.out.println("ti = " + ti);
        File foo = new File(this.storageDir, "foo");
        try {
            SimpleStateMachineStorage.getTermIndexFromSnapshotFile((File)foo);
            Assertions.fail();
        }
        catch (IllegalArgumentException iae) {
            System.out.println("Good " + iae);
        }
    }

    @Test
    public void testSnapshotCleanup() throws IOException {
        File file;
        long index;
        long term;
        SnapshotRetentionPolicy snapshotRetentionPolicy = new SnapshotRetentionPolicy(){

            public int getNumSnapshotsRetained() {
                return 3;
            }
        };
        SimpleStateMachineStorage simpleStateMachineStorage = new SimpleStateMachineStorage();
        RaftStorageImpl storage = TestRaftStorage.newRaftStorage(this.storageDir);
        simpleStateMachineStorage.init((RaftStorage)storage);
        HashSet<TermIndex> termIndexSet = new HashSet<TermIndex>();
        while (termIndexSet.size() < 5) {
            term = ThreadLocalRandom.current().nextLong(1L, 10L);
            if (!termIndexSet.add(TermIndex.valueOf((long)term, (long)(index = ThreadLocalRandom.current().nextLong(100L, 1000L))))) continue;
            file = simpleStateMachineStorage.getSnapshotFile(term, index);
            Assertions.assertTrue((boolean)file.createNewFile());
        }
        while (termIndexSet.size() < 7) {
            term = 1L;
            index = ThreadLocalRandom.current().nextLong(0L, 100L);
            if (!termIndexSet.add(TermIndex.valueOf((long)1L, (long)index))) continue;
            file = simpleStateMachineStorage.getSnapshotFile(1L, index);
            File snapshotFile = new File(file.getParent(), file.getName() + ".md5");
            Assertions.assertTrue((boolean)snapshotFile.createNewFile());
        }
        File stateMachineDir = storage.getStorageDir().getStateMachineDir();
        TestRaftStorage.assertFileCount(stateMachineDir, 7);
        simpleStateMachineStorage.cleanupOldSnapshots(snapshotRetentionPolicy);
        File[] remainingFiles = TestRaftStorage.assertFileCount(stateMachineDir, 3);
        List remainingIndices = termIndexSet.stream().map(TermIndex::getIndex).sorted(Collections.reverseOrder()).limit(3L).collect(Collectors.toList());
        for (File file2 : remainingFiles) {
            System.out.println(file2.getName());
            Matcher matcher = SimpleStateMachineStorage.SNAPSHOT_REGEX.matcher(file2.getName());
            if (!matcher.matches()) continue;
            Assertions.assertTrue((boolean)remainingIndices.contains(Long.parseLong(matcher.group(2))));
        }
        simpleStateMachineStorage.cleanupOldSnapshots(snapshotRetentionPolicy);
        TestRaftStorage.assertFileCount(stateMachineDir, 3);
        for (int i = 0; i < 2; ++i) {
            long term2 = ThreadLocalRandom.current().nextLong(1L, 10L);
            long index2 = ThreadLocalRandom.current().nextLong(1000L);
            File file3 = simpleStateMachineStorage.getSnapshotFile(term2, index2);
            Assertions.assertTrue((boolean)file3.createNewFile());
        }
        simpleStateMachineStorage.cleanupOldSnapshots(new SnapshotRetentionPolicy(){});
        TestRaftStorage.assertFileCount(stateMachineDir, 5);
    }

    private static File[] assertFileCount(File dir, int expected) {
        Object[] files = dir.listFiles();
        Assertions.assertNotNull((Object)files);
        Assertions.assertEquals((int)expected, (int)files.length, (String)Arrays.toString(files));
        return files;
    }

    @Test
    public void testNotEnoughSpace() throws IOException {
        File mockStorageDir = (File)Mockito.spy((Object)this.storageDir);
        Mockito.when((Object)mockStorageDir.getFreeSpace()).thenReturn((Object)100L);
        RaftStorageDirectoryImpl sd = new RaftStorageDirectoryImpl(mockStorageDir, SizeInBytes.valueOf((String)"100M"));
        RaftStorageDirectoryImpl.StorageState state = sd.analyzeStorage(false);
        Assertions.assertEquals((Object)RaftStorageDirectoryImpl.StorageState.NO_SPACE, (Object)state);
    }
}

