package org.apache.xml.security.keys.keyresolver.implementations;

import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;

import javax.crypto.SecretKey;

import org.apache.xml.security.exceptions.XMLSecurityException;
import org.apache.xml.security.keys.content.DEREncodedKeyValue;
import org.apache.xml.security.keys.keyresolver.KeyResolverException;
import org.apache.xml.security.keys.keyresolver.KeyResolverSpi;
import org.apache.xml.security.keys.storage.StorageResolver;
import org.apache.xml.security.utils.Constants;
import org.apache.xml.security.utils.XMLUtils;
import org.w3c.dom.Element;

/**
 * KeyResolverSpi implementation which resolves public keys from a 
 * <code>dsig11:DEREncodedKeyValue</code> element.
 * 
 * @author Brent Putman (putmanb@georgetown.edu)
 */
public class DEREncodedKeyValueResolver extends KeyResolverSpi {
	
    /** {@link org.apache.commons.logging} logging facility */
    private static org.apache.commons.logging.Log log = 
        org.apache.commons.logging.LogFactory.getLog(DEREncodedKeyValueResolver.class);

	/** {@inheritDoc}. */
	public boolean engineCanResolve(Element element, String baseURI, StorageResolver storage) {
		return XMLUtils.elementIsInSignature11Space(element, Constants._TAG_DERENCODEDKEYVALUE);
	}

	/** {@inheritDoc}. */
	public PublicKey engineLookupAndResolvePublicKey(Element element, String baseURI, StorageResolver storage)
			throws KeyResolverException {
		
        if (log.isDebugEnabled()) {
            log.debug("Can I resolve " + element.getTagName());
        }
        
        if (!engineCanResolve(element, baseURI, storage)) {
        	return null;
        }

        try {
            DEREncodedKeyValue derKeyValue = new DEREncodedKeyValue(element, baseURI);
            return derKeyValue.getPublicKey();
        } catch (XMLSecurityException e) {
            if (log.isDebugEnabled()) {
                log.debug("XMLSecurityException", e);
            }
        }

        return null;
	}

	/** {@inheritDoc}. */
	public X509Certificate engineLookupResolveX509Certificate(Element element, String baseURI, StorageResolver storage)
			throws KeyResolverException {
		return null;
	}

	/** {@inheritDoc}. */
	public SecretKey engineLookupAndResolveSecretKey(Element element, String baseURI, StorageResolver storage)
			throws KeyResolverException {
		return null;
	}

	/** {@inheritDoc}. */
	public PrivateKey engineLookupAndResolvePrivateKey(Element element, String baseURI, StorageResolver storage)
			throws KeyResolverException {
		return null;
	}
	
	

}
