/*
 * Decompiled with CFR 0.152.
 */
package org.apache.seatunnel.connectors.seatunnel.jdbc.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.Serializable;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import org.apache.seatunnel.common.exception.CommonErrorCode;
import org.apache.seatunnel.common.exception.SeaTunnelErrorCode;
import org.apache.seatunnel.connectors.seatunnel.jdbc.exception.JdbcConnectorErrorCode;
import org.apache.seatunnel.connectors.seatunnel.jdbc.exception.JdbcConnectorException;
import org.apache.seatunnel.connectors.seatunnel.jdbc.internal.connection.JdbcConnectionProvider;
import org.apache.seatunnel.connectors.seatunnel.jdbc.internal.executor.JdbcBatchStatementExecutor;
import org.apache.seatunnel.connectors.seatunnel.jdbc.internal.options.JdbcConnectionOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcOutputFormat<I, E extends JdbcBatchStatementExecutor<I>>
implements Serializable {
    protected final JdbcConnectionProvider connectionProvider;
    private static final long serialVersionUID = 1L;
    private static final Logger LOG = LoggerFactory.getLogger(JdbcOutputFormat.class);
    private final JdbcConnectionOptions jdbcConnectionOptions;
    private final StatementExecutorFactory<E> statementExecutorFactory;
    private transient E jdbcStatementExecutor;
    private transient int batchCount = 0;
    private volatile transient boolean closed = false;
    private transient ScheduledExecutorService scheduler;
    private transient ScheduledFuture<?> scheduledFuture;
    private volatile transient Exception flushException;

    public JdbcOutputFormat(JdbcConnectionProvider connectionProvider, JdbcConnectionOptions jdbcConnectionOptions, StatementExecutorFactory<E> statementExecutorFactory) {
        this.connectionProvider = Preconditions.checkNotNull(connectionProvider);
        this.jdbcConnectionOptions = Preconditions.checkNotNull(jdbcConnectionOptions);
        this.statementExecutorFactory = Preconditions.checkNotNull(statementExecutorFactory);
    }

    public void open() throws IOException {
        try {
            this.connectionProvider.getOrEstablishConnection();
        }
        catch (Exception e) {
            throw new JdbcConnectorException(JdbcConnectorErrorCode.CONNECT_DATABASE_FAILED, "unable to open JDBC writer", e);
        }
        this.jdbcStatementExecutor = this.createAndOpenStatementExecutor(this.statementExecutorFactory);
        if (this.jdbcConnectionOptions.getBatchIntervalMs() != 0 && this.jdbcConnectionOptions.getBatchSize() != 1) {
            this.scheduler = Executors.newScheduledThreadPool(1, runnable -> {
                AtomicInteger cnt = new AtomicInteger(0);
                Thread thread = new Thread(runnable);
                thread.setDaemon(true);
                thread.setName("jdbc-upsert-output-format-" + cnt.incrementAndGet());
                return thread;
            });
            this.scheduledFuture = this.scheduler.scheduleWithFixedDelay(() -> {
                JdbcOutputFormat jdbcOutputFormat = this;
                synchronized (jdbcOutputFormat) {
                    if (!this.closed) {
                        try {
                            this.flush();
                        }
                        catch (Exception e) {
                            this.flushException = e;
                        }
                    }
                }
            }, this.jdbcConnectionOptions.getBatchIntervalMs(), this.jdbcConnectionOptions.getBatchIntervalMs(), TimeUnit.MILLISECONDS);
        }
    }

    private E createAndOpenStatementExecutor(StatementExecutorFactory<E> statementExecutorFactory) {
        JdbcBatchStatementExecutor exec = (JdbcBatchStatementExecutor)statementExecutorFactory.get();
        try {
            exec.prepareStatements(this.connectionProvider.getConnection());
        }
        catch (SQLException e) {
            throw new JdbcConnectorException((SeaTunnelErrorCode)CommonErrorCode.SQL_OPERATION_FAILED, "unable to open JDBC writer", e);
        }
        return (E)exec;
    }

    private void checkFlushException() {
        if (this.flushException != null) {
            throw new JdbcConnectorException((SeaTunnelErrorCode)CommonErrorCode.FLUSH_DATA_FAILED, "Writing records to JDBC failed.", this.flushException);
        }
    }

    public final synchronized void writeRecord(I record) {
        this.checkFlushException();
        try {
            this.addToBatch(record);
            ++this.batchCount;
            if (this.jdbcConnectionOptions.getBatchSize() > 0 && this.batchCount >= this.jdbcConnectionOptions.getBatchSize()) {
                this.flush();
            }
        }
        catch (Exception e) {
            throw new JdbcConnectorException((SeaTunnelErrorCode)CommonErrorCode.SQL_OPERATION_FAILED, "Writing records to JDBC failed.", e);
        }
    }

    protected void addToBatch(I record) throws SQLException {
        this.jdbcStatementExecutor.addToBatch(record);
    }

    public synchronized void flush() throws IOException {
        this.checkFlushException();
        int sleepMs = 1000;
        for (int i = 0; i <= this.jdbcConnectionOptions.getMaxRetries(); ++i) {
            try {
                this.attemptFlush();
                this.batchCount = 0;
                break;
            }
            catch (SQLException e) {
                LOG.error("JDBC executeBatch error, retry times = {}", (Object)i, (Object)e);
                if (i >= this.jdbcConnectionOptions.getMaxRetries()) {
                    throw new JdbcConnectorException((SeaTunnelErrorCode)CommonErrorCode.FLUSH_DATA_FAILED, e);
                }
                try {
                    if (!this.connectionProvider.isConnectionValid()) {
                        this.updateExecutor(true);
                    }
                }
                catch (Exception exception) {
                    LOG.error("JDBC connection is not valid, and reestablish connection failed.", (Throwable)exception);
                    throw new JdbcConnectorException(JdbcConnectorErrorCode.CONNECT_DATABASE_FAILED, "Reestablish JDBC connection failed", exception);
                }
                try {
                    Thread.sleep(1000 * i);
                    continue;
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                    throw new JdbcConnectorException((SeaTunnelErrorCode)CommonErrorCode.FLUSH_DATA_FAILED, "unable to flush; interrupted while doing another attempt", e);
                }
            }
        }
    }

    protected void attemptFlush() throws SQLException {
        this.jdbcStatementExecutor.executeBatch();
    }

    public synchronized void close() {
        if (!this.closed) {
            this.closed = true;
            if (this.scheduledFuture != null) {
                this.scheduledFuture.cancel(false);
                this.scheduler.shutdown();
            }
            if (this.batchCount > 0) {
                try {
                    this.flush();
                }
                catch (Exception e) {
                    LOG.warn("Writing records to JDBC failed.", (Throwable)e);
                    throw new JdbcConnectorException((SeaTunnelErrorCode)CommonErrorCode.FLUSH_DATA_FAILED, "Writing records to JDBC failed.", e);
                }
            }
            try {
                if (this.jdbcStatementExecutor != null) {
                    this.jdbcStatementExecutor.closeStatements();
                }
            }
            catch (SQLException e) {
                LOG.warn("Close JDBC writer failed.", (Throwable)e);
            }
        }
        this.connectionProvider.closeConnection();
        this.checkFlushException();
    }

    public void updateExecutor(boolean reconnect) throws SQLException, ClassNotFoundException {
        this.jdbcStatementExecutor.closeStatements();
        this.jdbcStatementExecutor.prepareStatements(reconnect ? this.connectionProvider.reestablishConnection() : this.connectionProvider.getConnection());
    }

    @VisibleForTesting
    public Connection getConnection() {
        return this.connectionProvider.getConnection();
    }

    public static interface StatementExecutorFactory<T extends JdbcBatchStatementExecutor<?>>
    extends Supplier<T>,
    Serializable {
    }
}

