/*
 * Decompiled with CFR 0.152.
 */
package org.tikv.shade.com.google.common.collect;

import java.util.Arrays;
import org.checkerframework.checker.nullness.compatqual.NullableDecl;
import org.tikv.shade.com.google.common.annotations.GwtCompatible;
import org.tikv.shade.com.google.common.annotations.VisibleForTesting;
import org.tikv.shade.com.google.common.base.Objects;
import org.tikv.shade.com.google.common.base.Preconditions;
import org.tikv.shade.com.google.common.collect.CollectPreconditions;
import org.tikv.shade.com.google.common.collect.Hashing;
import org.tikv.shade.com.google.common.collect.Multiset;
import org.tikv.shade.com.google.common.collect.Multisets;
import org.tikv.shade.com.google.errorprone.annotations.CanIgnoreReturnValue;

@GwtCompatible(serializable=true, emulated=true)
class ObjectCountHashMap<K> {
    private static final int MAXIMUM_CAPACITY = 0x40000000;
    static final float DEFAULT_LOAD_FACTOR = 1.0f;
    private static final long NEXT_MASK = 0xFFFFFFFFL;
    private static final long HASH_MASK = -4294967296L;
    static final int DEFAULT_SIZE = 3;
    static final int UNSET = -1;
    transient Object[] keys;
    transient int[] values;
    transient int size;
    transient int modCount;
    private transient int[] table;
    @VisibleForTesting
    transient long[] entries;
    private transient float loadFactor;
    private transient int threshold;

    public static <K> ObjectCountHashMap<K> create() {
        return new ObjectCountHashMap<K>();
    }

    public static <K> ObjectCountHashMap<K> createWithExpectedSize(int expectedSize) {
        return new ObjectCountHashMap<K>(expectedSize);
    }

    ObjectCountHashMap() {
        this.init(3, 1.0f);
    }

    ObjectCountHashMap(ObjectCountHashMap<? extends K> map) {
        this.init(map.size(), 1.0f);
        int i = map.firstIndex();
        while (i != -1) {
            this.put(map.getKey(i), map.getValue(i));
            i = map.nextIndex(i);
        }
    }

    ObjectCountHashMap(int capacity) {
        this(capacity, 1.0f);
    }

    ObjectCountHashMap(int expectedSize, float loadFactor) {
        this.init(expectedSize, loadFactor);
    }

    void init(int expectedSize, float loadFactor) {
        Preconditions.checkArgument(expectedSize >= 0, "Initial capacity must be non-negative");
        Preconditions.checkArgument(loadFactor > 0.0f, "Illegal load factor");
        int buckets = Hashing.closedTableSize(expectedSize, loadFactor);
        this.table = ObjectCountHashMap.newTable(buckets);
        this.loadFactor = loadFactor;
        this.keys = new Object[expectedSize];
        this.values = new int[expectedSize];
        this.entries = ObjectCountHashMap.newEntries(expectedSize);
        this.threshold = Math.max(1, (int)((float)buckets * loadFactor));
    }

    private static int[] newTable(int size) {
        int[] array = new int[size];
        Arrays.fill(array, -1);
        return array;
    }

    private static long[] newEntries(int size) {
        long[] array = new long[size];
        Arrays.fill(array, -1L);
        return array;
    }

    private int hashTableMask() {
        return this.table.length - 1;
    }

    int firstIndex() {
        return this.size == 0 ? -1 : 0;
    }

    int nextIndex(int index) {
        return index + 1 < this.size ? index + 1 : -1;
    }

    int nextIndexAfterRemove(int oldNextIndex, int removedIndex) {
        return oldNextIndex - 1;
    }

    int size() {
        return this.size;
    }

    K getKey(int index) {
        Preconditions.checkElementIndex(index, this.size);
        return (K)this.keys[index];
    }

    int getValue(int index) {
        Preconditions.checkElementIndex(index, this.size);
        return this.values[index];
    }

    void setValue(int index, int newValue) {
        Preconditions.checkElementIndex(index, this.size);
        this.values[index] = newValue;
    }

    Multiset.Entry<K> getEntry(int index) {
        Preconditions.checkElementIndex(index, this.size);
        return new MapEntry(index);
    }

    private static int getHash(long entry) {
        return (int)(entry >>> 32);
    }

    private static int getNext(long entry) {
        return (int)entry;
    }

    private static long swapNext(long entry, int newNext) {
        return 0xFFFFFFFF00000000L & entry | 0xFFFFFFFFL & (long)newNext;
    }

    void ensureCapacity(int minCapacity) {
        if (minCapacity > this.entries.length) {
            this.resizeEntries(minCapacity);
        }
        if (minCapacity >= this.threshold) {
            int newTableSize = Math.max(2, Integer.highestOneBit(minCapacity - 1) << 1);
            this.resizeTable(newTableSize);
        }
    }

    @CanIgnoreReturnValue
    public int put(@NullableDecl K key, int value) {
        CollectPreconditions.checkPositive(value, "count");
        long[] entries = this.entries;
        Object[] keys = this.keys;
        int[] values = this.values;
        int hash = Hashing.smearedHash(key);
        int tableIndex = hash & this.hashTableMask();
        int newEntryIndex = this.size;
        int next = this.table[tableIndex];
        if (next == -1) {
            this.table[tableIndex] = newEntryIndex;
        } else {
            long entry;
            do {
                int last = next;
                entry = entries[next];
                if (ObjectCountHashMap.getHash(entry) != hash || !Objects.equal(key, keys[next])) continue;
                int oldValue = values[next];
                values[next] = value;
                return oldValue;
            } while ((next = ObjectCountHashMap.getNext(entry)) != -1);
            entries[last] = ObjectCountHashMap.swapNext(entry, newEntryIndex);
        }
        if (newEntryIndex == Integer.MAX_VALUE) {
            throw new IllegalStateException("Cannot contain more than Integer.MAX_VALUE elements!");
        }
        int newSize = newEntryIndex + 1;
        this.resizeMeMaybe(newSize);
        this.insertEntry(newEntryIndex, key, value, hash);
        this.size = newSize;
        if (newEntryIndex >= this.threshold) {
            this.resizeTable(2 * this.table.length);
        }
        ++this.modCount;
        return 0;
    }

    void insertEntry(int entryIndex, @NullableDecl K key, int value, int hash) {
        this.entries[entryIndex] = (long)hash << 32 | 0xFFFFFFFFL;
        this.keys[entryIndex] = key;
        this.values[entryIndex] = value;
    }

    private void resizeMeMaybe(int newSize) {
        int entriesSize = this.entries.length;
        if (newSize > entriesSize) {
            int newCapacity = entriesSize + Math.max(1, entriesSize >>> 1);
            if (newCapacity < 0) {
                newCapacity = Integer.MAX_VALUE;
            }
            if (newCapacity != entriesSize) {
                this.resizeEntries(newCapacity);
            }
        }
    }

    void resizeEntries(int newCapacity) {
        this.keys = Arrays.copyOf(this.keys, newCapacity);
        this.values = Arrays.copyOf(this.values, newCapacity);
        long[] entries = this.entries;
        int oldCapacity = entries.length;
        entries = Arrays.copyOf(entries, newCapacity);
        if (newCapacity > oldCapacity) {
            Arrays.fill(entries, oldCapacity, newCapacity, -1L);
        }
        this.entries = entries;
    }

    private void resizeTable(int newCapacity) {
        int[] oldTable = this.table;
        int oldCapacity = oldTable.length;
        if (oldCapacity >= 0x40000000) {
            this.threshold = Integer.MAX_VALUE;
            return;
        }
        int newThreshold = 1 + (int)((float)newCapacity * this.loadFactor);
        int[] newTable = ObjectCountHashMap.newTable(newCapacity);
        long[] entries = this.entries;
        int mask = newTable.length - 1;
        for (int i = 0; i < this.size; ++i) {
            long oldEntry = entries[i];
            int hash = ObjectCountHashMap.getHash(oldEntry);
            int tableIndex = hash & mask;
            int next = newTable[tableIndex];
            newTable[tableIndex] = i;
            entries[i] = (long)hash << 32 | 0xFFFFFFFFL & (long)next;
        }
        this.threshold = newThreshold;
        this.table = newTable;
    }

    int indexOf(@NullableDecl Object key) {
        int hash = Hashing.smearedHash(key);
        int next = this.table[hash & this.hashTableMask()];
        while (next != -1) {
            long entry = this.entries[next];
            if (ObjectCountHashMap.getHash(entry) == hash && Objects.equal(key, this.keys[next])) {
                return next;
            }
            next = ObjectCountHashMap.getNext(entry);
        }
        return -1;
    }

    public boolean containsKey(@NullableDecl Object key) {
        return this.indexOf(key) != -1;
    }

    public int get(@NullableDecl Object key) {
        int index = this.indexOf(key);
        return index == -1 ? 0 : this.values[index];
    }

    @CanIgnoreReturnValue
    public int remove(@NullableDecl Object key) {
        return this.remove(key, Hashing.smearedHash(key));
    }

    private int remove(@NullableDecl Object key, int hash) {
        int tableIndex = hash & this.hashTableMask();
        int next = this.table[tableIndex];
        if (next == -1) {
            return 0;
        }
        int last = -1;
        do {
            if (ObjectCountHashMap.getHash(this.entries[next]) == hash && Objects.equal(key, this.keys[next])) {
                int oldValue = this.values[next];
                if (last == -1) {
                    this.table[tableIndex] = ObjectCountHashMap.getNext(this.entries[next]);
                } else {
                    this.entries[last] = ObjectCountHashMap.swapNext(this.entries[last], ObjectCountHashMap.getNext(this.entries[next]));
                }
                this.moveLastEntry(next);
                --this.size;
                ++this.modCount;
                return oldValue;
            }
            last = next;
        } while ((next = ObjectCountHashMap.getNext(this.entries[next])) != -1);
        return 0;
    }

    @CanIgnoreReturnValue
    int removeEntry(int entryIndex) {
        return this.remove(this.keys[entryIndex], ObjectCountHashMap.getHash(this.entries[entryIndex]));
    }

    void moveLastEntry(int dstIndex) {
        int srcIndex = this.size() - 1;
        if (dstIndex < srcIndex) {
            long lastEntry;
            this.keys[dstIndex] = this.keys[srcIndex];
            this.values[dstIndex] = this.values[srcIndex];
            this.keys[srcIndex] = null;
            this.values[srcIndex] = 0;
            this.entries[dstIndex] = lastEntry = this.entries[srcIndex];
            this.entries[srcIndex] = -1L;
            int tableIndex = ObjectCountHashMap.getHash(lastEntry) & this.hashTableMask();
            int lastNext = this.table[tableIndex];
            if (lastNext == srcIndex) {
                this.table[tableIndex] = dstIndex;
            } else {
                long entry;
                do {
                    int previous = lastNext;
                } while ((lastNext = ObjectCountHashMap.getNext(entry = this.entries[lastNext])) != srcIndex);
                this.entries[previous] = ObjectCountHashMap.swapNext(entry, dstIndex);
            }
        } else {
            this.keys[dstIndex] = null;
            this.values[dstIndex] = 0;
            this.entries[dstIndex] = -1L;
        }
    }

    public void clear() {
        ++this.modCount;
        Arrays.fill(this.keys, 0, this.size, null);
        Arrays.fill(this.values, 0, this.size, 0);
        Arrays.fill(this.table, -1);
        Arrays.fill(this.entries, -1L);
        this.size = 0;
    }

    class MapEntry
    extends Multisets.AbstractEntry<K> {
        @NullableDecl
        final K key;
        int lastKnownIndex;

        MapEntry(int index) {
            this.key = ObjectCountHashMap.this.keys[index];
            this.lastKnownIndex = index;
        }

        @Override
        public K getElement() {
            return this.key;
        }

        void updateLastKnownIndex() {
            if (this.lastKnownIndex == -1 || this.lastKnownIndex >= ObjectCountHashMap.this.size() || !Objects.equal(this.key, ObjectCountHashMap.this.keys[this.lastKnownIndex])) {
                this.lastKnownIndex = ObjectCountHashMap.this.indexOf(this.key);
            }
        }

        @Override
        public int getCount() {
            this.updateLastKnownIndex();
            return this.lastKnownIndex == -1 ? 0 : ObjectCountHashMap.this.values[this.lastKnownIndex];
        }

        @CanIgnoreReturnValue
        public int setCount(int count) {
            this.updateLastKnownIndex();
            if (this.lastKnownIndex == -1) {
                ObjectCountHashMap.this.put(this.key, count);
                return 0;
            }
            int old = ObjectCountHashMap.this.values[this.lastKnownIndex];
            ObjectCountHashMap.this.values[this.lastKnownIndex] = count;
            return old;
        }
    }
}

